// Reporte: Capacidades de Bodega
(function() {
  "use strict";

  const $ = window.jQuery;
  if (!$) return;

  const meta = document.querySelector('meta[name="base-url"]');
  const BASE = (meta && meta.content ? meta.content : '/').replace(/\/+$/, '');
  const joinUrl = (p) => (BASE.endsWith('/') ? BASE : BASE + '/') + p.replace(/^\/+/, '');
  
  const nfmt = (v, d = 0) => new Intl.NumberFormat('es-PY', { 
    minimumFractionDigits: d, 
    maximumFractionDigits: d 
  }).format(v ?? 0);

  // Google Charts no requiere mantener referencias a los objetos chart
  let currentData = null;

  // ============================================================================
  // CARGAR DATOS
  // ============================================================================
  function cargarDatos() {
    const depositoId = $('#filtroDeposito').val();
    const url = joinUrl('/api/reportes/capacidades_bodega.php') + (depositoId ? `?deposito_id=${depositoId}` : '');
    
    showLoading();
    
    fetch(url, { credentials: 'same-origin' })
      .then(r => r.json())
      .then(data => {
        if (window.Swal) Swal.close();
        
        if (!data || !data.ok) {
          notify('error', 'Error', data?.error || 'Error al cargar datos');
          return;
        }
        
        currentData = data;
        renderKPIs(data.totales);
        renderM2Utiles(data.m2_utiles, data.posiciones_utiles);
        renderTableDepositos(data.depositos);
        renderTableAmbientes(data.ambientes);
        renderMatriz(data.matriz, data.depositos, data.ambientes);
        renderCharts(data.depositos, data.ambientes);
        
        $('#timestamp').text(data.timestamp || new Date().toLocaleString('es-PY'));
      })
      .catch(err => {
        if (window.Swal) Swal.close();
        console.error('[capacidades] Error:', err);
        notify('error', 'Error', 'Error al cargar el reporte');
      });
  }

  // ============================================================================
  // RENDER KPIs
  // ============================================================================
  function renderKPIs(totales) {
    const total = parseInt(totales.total_posiciones) || 0;
    const ocupadas = parseInt(totales.posiciones_ocupadas) || 0;
    const libres = parseInt(totales.posiciones_libres) || 0;
    const capacidad = parseInt(totales.capacidad_total_pallets) || 0;
    const palletsAct = parseInt(totales.pallets_actuales) || 0;
    const palletsDisp = parseInt(totales.pallets_disponibles) || 0;
    
    $('#kpiTotalPosiciones').text(nfmt(total));
    $('#kpiPosLibres').text(nfmt(libres));
    $('#kpiPosOcupadas').text(nfmt(ocupadas));
    $('#kpiCapacidadPallets').text(nfmt(capacidad));
    $('#kpiPalletsActuales').text(nfmt(palletsAct));
    $('#kpiPalletsDisponibles').text(nfmt(palletsDisp));
    
    // Progress bars
    const pctLibres = total > 0 ? (libres / total) * 100 : 0;
    const pctOcupadas = total > 0 ? (ocupadas / total) * 100 : 0;
    
    $('#progressLibres').css('width', pctLibres + '%');
    $('#progressOcupadas').css('width', pctOcupadas + '%');
  }

  function renderM2Utiles(m2, posiciones) {
    $('#kpiM2Utiles').text(nfmt(m2, 2));
    $('#kpiPosicionesUtiles').text(nfmt(posiciones));
  }

  // ============================================================================
  // RENDER TABLAS
  // ============================================================================
  function renderTableDepositos(depositos) {
    const $tbody = $('#tableDepositos tbody');
    $tbody.empty();
    
    $('#badgeTotalDepositos').text(depositos.length);
    
    if (!depositos || depositos.length === 0) {
      $tbody.append('<tr><td colspan="8" class="text-center text-muted">No hay datos</td></tr>');
      return;
    }
    
    depositos.forEach(dep => {
      const pct = parseFloat(dep.porcentaje_ocupacion) || 0;
      const badgeClass = pct >= 90 ? 'danger' : pct >= 70 ? 'warning' : pct >= 50 ? 'info' : 'success';
      
      const row = `
        <tr>
          <td><strong>${escapeHtml(dep.nombre)}</strong></td>
          <td><code>${escapeHtml(dep.code)}</code></td>
          <td class="text-end">${nfmt(dep.total_posiciones)}</td>
          <td class="text-end text-warning">${nfmt(dep.posiciones_ocupadas)}</td>
          <td class="text-end text-success">${nfmt(dep.posiciones_libres)}</td>
          <td class="text-end">${nfmt(dep.capacidad_total_pallets)}</td>
          <td class="text-end">${nfmt(dep.pallets_actuales)}</td>
          <td class="text-center">
            <span class="badge badge-ocupacion bg-${badgeClass}">${nfmt(pct, 1)}%</span>
          </td>
        </tr>
      `;
      $tbody.append(row);
    });
  }

  function renderTableAmbientes(ambientes) {
    const $tbody = $('#tableAmbientes tbody');
    $tbody.empty();
    
    $('#badgeTotalAmbientes').text(ambientes.length);
    
    if (!ambientes || ambientes.length === 0) {
      $tbody.append('<tr><td colspan="8" class="text-center text-muted">No hay datos</td></tr>');
      return;
    }
    
    ambientes.forEach(amb => {
      const pct = parseFloat(amb.porcentaje_ocupacion) || 0;
      const badgeClass = pct >= 90 ? 'danger' : pct >= 70 ? 'warning' : pct >= 50 ? 'info' : 'success';
      
      const row = `
        <tr>
          <td><strong>${escapeHtml(amb.nombre)}</strong></td>
          <td><code>${escapeHtml(amb.code)}</code></td>
          <td class="text-end">${nfmt(amb.total_posiciones)}</td>
          <td class="text-end text-warning">${nfmt(amb.posiciones_ocupadas)}</td>
          <td class="text-end text-success">${nfmt(amb.posiciones_libres)}</td>
          <td class="text-end">${nfmt(amb.capacidad_total_pallets)}</td>
          <td class="text-end">${nfmt(amb.pallets_actuales)}</td>
          <td class="text-center">
            <span class="badge badge-ocupacion bg-${badgeClass}">${nfmt(pct, 1)}%</span>
          </td>
        </tr>
      `;
      $tbody.append(row);
    });
  }

  function renderMatriz(matriz, depositos, ambientes) {
    const $table = $('#tableMatriz');
    const $thead = $table.find('thead tr');
    const $tbody = $table.find('tbody');
    
    $thead.empty();
    $tbody.empty();
    
    // Construir headers (ambientes)
    $thead.append('<th>Depósito</th>');
    ambientes.forEach(amb => {
      $thead.append(`<th class="text-center">${escapeHtml(amb.nombre)}<br><small class="text-muted">${escapeHtml(amb.code)}</small></th>`);
    });
    
    // Construir filas (por depósito)
    depositos.forEach(dep => {
      const $row = $('<tr></tr>');
      $row.append(`<td><strong>${escapeHtml(dep.nombre)}</strong><br><small class="text-muted">${escapeHtml(dep.code)}</small></td>`);
      
      ambientes.forEach(amb => {
        // Buscar celda en matriz
        const celda = matriz.find(m => 
          parseInt(m.deposito_id) === parseInt(dep.id) && 
          parseInt(m.ambiente_id) === parseInt(amb.id)
        );
        
        if (celda && parseInt(celda.total_posiciones) > 0) {
          const ocupadas = parseInt(celda.posiciones_ocupadas) || 0;
          const total = parseInt(celda.total_posiciones) || 0;
          const pct = total > 0 ? (ocupadas / total) * 100 : 0;
          const bgClass = pct >= 90 ? 'table-danger' : pct >= 70 ? 'table-warning' : pct >= 50 ? 'table-info' : 'table-success';
          
          $row.append(`
            <td class="text-center ${bgClass}">
              <strong>${ocupadas}</strong> / ${total}
              <br><small class="text-muted">${nfmt(pct, 0)}%</small>
            </td>
          `);
        } else {
          $row.append('<td class="text-center text-muted">-</td>');
        }
      });
      
      $tbody.append($row);
    });
  }

  // ============================================================================
  // RENDER CHARTS
  // ============================================================================
  function renderCharts(depositos, ambientes, retries = 0) {
    const maxRetries = 20; // Máximo 2 segundos de espera
    
    console.log('[capacidades] renderCharts llamado con:', {
      depositos: depositos?.length || 0,
      ambientes: ambientes?.length || 0,
      retries: retries
    });
    
    // Verificar si Google Charts está disponible
    if (typeof google === 'undefined' || !google.charts) {
      if (retries >= maxRetries) {
        // console.error('[capacidades] Google Charts no se pudo cargar después de ' + maxRetries + ' intentos');
        notify('warning', 'Advertencia', 'Los gráficos no están disponibles. Intente recargar la página.');
        return;
      }
      
      setTimeout(function() {
        renderCharts(depositos, ambientes, retries + 1);
      }, 100);
      return;
    }
    
    // console.log('[capacidades] Google Charts disponible, cargando paquetes...');
    
    // Cargar Google Charts y renderizar cuando esté listo
    google.charts.load('current', {'packages':['corechart', 'bar']});
    google.charts.setOnLoadCallback(function() {
      console.log('[capacidades] Paquetes cargados, dibujando gráficos...');
      try {
        drawDepositosChart(depositos);
        drawAmbientesChart(ambientes);
        // console.log('[capacidades] Gráficos dibujados exitosamente');
      } catch(err) {
        console.error('[capacidades] Error al dibujar gráficos:', err);
      }
    });
  }

  // ============================================================================
  // DIBUJAR GRÁFICO DE DEPÓSITOS CON GOOGLE CHARTS (Material Design)
  // ============================================================================
  function drawDepositosChart(depositos) {
    console.log('[capacidades] drawDepositosChart:', depositos);
    
    const chartContainer = document.getElementById('chartDepositos');
    if (!chartContainer) {
      console.error('[capacidades] No se encontró el elemento chartDepositos');
      return;
    }
    
    // Preparar datos para Google Charts [Label, Ocupadas, Libres]
    var dataArray = [['Depósito', 'Ocupadas', 'Libres']];
    
    depositos.forEach(function(d) {
      dataArray.push([
        d.code,
        parseInt(d.posiciones_ocupadas) || 0,
        parseInt(d.posiciones_libres) || 0
      ]);
    });
    
    // console.log('[capacidades] Datos depósitos:', dataArray);
    
    var dataTable = google.visualization.arrayToDataTable(dataArray);
    
    var options = {
      title: 'Ocupación por Depósito',
      vAxis: {
        title: 'Cantidad de Posiciones',
        format: 'decimal',
        minValue: 0
      },
      hAxis: {
        title: 'Depósito'
      },
      height: 400,
      width: '100%',
      backgroundColor: 'transparent',  // Fondo transparente
      colors: ['#168eea', '#e2c636'],  // Azul para Ocupadas, Amarillo para Libres
      legend: { 
        position: 'top',
        textStyle: { color: '#999999' }
      },
      chartArea: { width: '80%', height: '70%' }
    };
    
    var chart = new google.visualization.ColumnChart(chartContainer);
    chart.draw(dataTable, options);
    console.log('[capacidades] Gráfico de depósitos dibujado');
  }

  // ============================================================================
  // DIBUJAR GRÁFICO DE AMBIENTES CON GOOGLE CHARTS
  // ============================================================================
  function drawAmbientesChart(ambientes) {
    console.log('[capacidades] drawAmbientesChart:', ambientes);
    
    const chartContainer = document.getElementById('chartAmbientes');
    if (!chartContainer) {
      console.error('[capacidades] No se encontró el elemento chartAmbientes');
      return;
    }
    
    // Preparar datos para Google Charts [Label, Value]
    const dataArray = [['Tipo Ambiente', 'Total Posiciones']];
    
    ambientes.forEach(function(a) {
      dataArray.push([
        a.nombre,
        parseInt(a.total_posiciones) || 0
      ]);
    });
    
    console.log('[capacidades] Datos ambientes:', dataArray);
    
    const dataTable = google.visualization.arrayToDataTable(dataArray);
    
    const options = {
      is3D: true,  // Gráfico 3D (Pie Chart 2 style)
      height: 400,
      width: '100%',
      backgroundColor: 'transparent',  // Fondo transparente
      colors: [
        '#762d91',  // Azul primario (vihoAdminConfig.primary)
        '#2d9141',  // Azul secundario (vihoAdminConfig.secondary)
        '#e2c636',  // Amarillo
        '#222222',  // Negro
        '#762d91'   // Gris
      ],
      legend: { 
        position: 'right',
        textStyle: { color: '#999999' }
      },
      chartArea: { width: '90%', height: '90%' }
    };
    
    const chart = new google.visualization.PieChart(chartContainer);
    chart.draw(dataTable, options);
    console.log('[capacidades] Gráfico de ambientes dibujado');
  }

  // ============================================================================
  // CARGAR DEPÓSITOS PARA FILTRO
  // ============================================================================
  function cargarDepositos() {
    fetch(joinUrl('/api/deposito/list.php'), { credentials: 'same-origin' })
      .then(r => r.json())
      .then(data => {
        const depositos = Array.isArray(data) ? data : (data.data || []);
        const $select = $('#filtroDeposito');
        
        depositos.forEach(dep => {
          $select.append(`<option value="${dep.id}">${dep.nombre} (${dep.code})</option>`);
        });
      })
      .catch(err => {
        console.warn('[capacidades] No se pudieron cargar depósitos:', err);
      });
  }

  // ============================================================================
  // EXPORTAR A EXCEL (simulado como CSV)
  // ============================================================================
  function exportarCSV() {
    if (!currentData) {
      notify('warning', 'Sin datos', 'No hay datos para exportar');
      return;
    }
    
    let csv = 'REPORTE DE CAPACIDADES DE BODEGA\n\n';
    csv += 'TOTALES GENERALES\n';
    csv += 'Total Posiciones,' + (currentData.totales.total_posiciones || 0) + '\n';
    csv += 'Posiciones Ocupadas,' + (currentData.totales.posiciones_ocupadas || 0) + '\n';
    csv += 'Posiciones Libres,' + (currentData.totales.posiciones_libres || 0) + '\n';
    csv += 'Capacidad Total (Pallets),' + (currentData.totales.capacidad_total_pallets || 0) + '\n';
    csv += 'M2 Útiles,' + (currentData.m2_utiles || 0) + '\n\n';
    
    csv += 'POR DEPÓSITO\n';
    csv += 'Depósito,Código,Total,Ocupadas,Libres,Capacidad,Pallets Actuales,% Ocupación\n';
    currentData.depositos.forEach(d => {
      csv += `${d.nombre},${d.code},${d.total_posiciones},${d.posiciones_ocupadas},${d.posiciones_libres},${d.capacidad_total_pallets},${d.pallets_actuales},${d.porcentaje_ocupacion}%\n`;
    });
    
    csv += '\nPOR AMBIENTE\n';
    csv += 'Ambiente,Código,Total,Ocupadas,Libres,Capacidad,Pallets Actuales,% Ocupación\n';
    currentData.ambientes.forEach(a => {
      csv += `${a.nombre},${a.code},${a.total_posiciones},${a.posiciones_ocupadas},${a.posiciones_libres},${a.capacidad_total_pallets},${a.pallets_actuales},${a.porcentaje_ocupacion}%\n`;
    });
    
    // Descargar
    const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    link.href = URL.createObjectURL(blob);
    link.download = 'capacidades_bodega_' + new Date().toISOString().slice(0,10) + '.csv';
    link.click();
    
    notify('success', 'Exportado', 'Archivo CSV descargado');
  }

  // ============================================================================
  // HELPERS
  // ============================================================================
  function escapeHtml(str) {
    const div = document.createElement('div');
    div.textContent = str || '';
    return div.innerHTML;
  }

  function showLoading() {
    if (window.Swal) {
      Swal.fire({
        title: 'Cargando reporte...',
        allowOutsideClick: false,
        didOpen: () => Swal.showLoading()
      });
    }
  }

  function notify(type, title, text) {
    if (window.Swal) {
      Swal.fire({ icon: type, title, text, timer: 3000, toast: true, position: 'top-end', showConfirmButton: false });
    } else {
      alert(title + ': ' + text);
    }
  }

  // ============================================================================
  // EVENT HANDLERS
  // ============================================================================
  $('#btnRefrescar').on('click', cargarDatos);
  $('#btnExportar').on('click', exportarCSV);
  $('#filtroDeposito').on('change', cargarDatos);

  // ============================================================================
  // INIT
  // ============================================================================
  $(document).ready(function() {
    cargarDepositos();
    cargarDatos();
  });

})();
