/**
 * SOL · JavaScript · Reporte Clientes OTIF
 */

(function(window, $) {
  'use strict';

  if (!$ || typeof $.ajax !== 'function') {
    console.error('[Reporte Clientes OTIF] jQuery es requerido.');
    return;
  }

  const baseUrl = document.querySelector('meta[name="base-url"]')?.getAttribute('content') ||
                  (typeof window.BASE_URL !== 'undefined' ? window.BASE_URL : '/');

  function joinUrl(path) {
    return baseUrl.replace(/\/$/, '') + '/' + String(path || '').replace(/^\//, '');
  }

  const numberFormatter = new Intl.NumberFormat('es-AR');
  const percentFormatter = new Intl.NumberFormat('es-AR', { minimumFractionDigits: 1, maximumFractionDigits: 1 });

  function formatNumber(value) {
    const numeric = Number(value);
    return numberFormatter.format(Number.isFinite(numeric) ? numeric : 0);
  }

  function formatPercent(value) {
    if (value === null || value === undefined) {
      return '0%';
    }
    const numeric = Number(value);
    if (!Number.isFinite(numeric)) {
      return '0%';
    }
    return percentFormatter.format(numeric) + '%';
  }

  function formatHours(value) {
    if (value === null || value === undefined) {
      return '—';
    }
    const numeric = Number(value);
    if (!Number.isFinite(numeric)) {
      return '—';
    }
    const rounded = Math.round(numeric * 100) / 100;
    const sign = rounded > 0 ? '+' : '';
    return sign + rounded.toFixed(2) + ' h';
  }

  function escapeHtml(value) {
    return String(value)
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;')
      .replace(/'/g, '&#039;');
  }

  function buildStatusBadge(isOk, label) {
    if (isOk) {
      return '<span class="badge bg-success me-1">' + escapeHtml(label) + '</span>';
    }
    return '<span class="badge bg-light text-muted border me-1">' + escapeHtml(label) + '</span>';
  }

  function buildDeltaBadge(sign, label) {
    if (!label) {
      return '<span class="badge bg-light text-muted border">—</span>';
    }
    let cls = 'bg-secondary';
    if (sign === 'early') {
      cls = 'bg-info';
    } else if (sign === 'ontime') {
      cls = 'bg-success';
    } else if (sign === 'late') {
      cls = 'bg-danger';
    }
    return '<span class="badge ' + cls + '">' + escapeHtml(label) + '</span>';
  }

  let dataTable = null;
  let combosLoaded = false;
  let currentData = [];
  let currentSummary = null;
  let currentAggregates = {};
  let currentLimit = 0;
  let currentTruncated = false;
  let defaultTolerance = 0;

  $(document).ready(function() {
    setDefaultDateRange();
    initDataTable();
    bindEvents();
    loadCombos();
  });

  function setDefaultDateRange() {
    const today = new Date();
    const start = new Date(today.getTime() - (29 * 24 * 60 * 60 * 1000));
    document.getElementById('fecha_hasta').value = today.toISOString().slice(0, 10);
    document.getElementById('fecha_desde').value = start.toISOString().slice(0, 10);
  }

  function applyDefaultTolerance(force) {
    const select = document.getElementById('tolerancia_horas');
    if (!select) {
      return;
    }
    if (force || !select.value) {
      select.value = String(defaultTolerance);
    }
  }

  function resetFormToDefaults() {
    const form = document.getElementById('frmClientesOtif');
    if (form) {
      form.reset();
    }
    setDefaultDateRange();
    applyDefaultTolerance(true);
  }

  function bindEvents() {
    $('#frmClientesOtif').on('submit', function(evt) {
      evt.preventDefault();
      loadData();
    });

    $('#btnLimpiar').on('click', function() {
      resetFormToDefaults();
      loadData();
    });
  }

  function loadCombos() {
    $.ajax({
      url: joinUrl('api/reportes/clientes_otif.php'),
      method: 'GET',
      data: { meta: 'filters' },
      dataType: 'json'
    })
      .done(function(response) {
        if (!response || !response.ok) {
          notify('error', 'Combos', 'No fue posible cargar los filtros.');
          return;
        }
        applyCombos(response);
        defaultTolerance = Number(response.default_tolerancia || 0);
        applyDefaultTolerance(true);
        combosLoaded = true;
        loadData();
      })
      .fail(function(xhr) {
        console.error('[Reporte Clientes OTIF] Error al cargar combos', xhr);
        notify('error', 'Combos', 'No se pudieron cargar los filtros.');
      });
  }

  function applyCombos(data) {
    const clientes = Array.isArray(data.clientes) ? data.clientes : [];
    const estados = Array.isArray(data.estados) ? data.estados : [];
    const tolerancias = Array.isArray(data.tolerancias) ? data.tolerancias : [];

    const $cliente = $('#cliente_id');
    $cliente.find('option:not(:first)').remove();
    clientes.forEach(function(cli) {
      $('<option></option>').val(cli.id).text(cli.nombre).appendTo($cliente);
    });

    const $estado = $('#estado_id');
    $estado.find('option:not(:first)').remove();
    estados.forEach(function(est) {
      const label = est.code ? est.nombre + ' [' + est.code + ']' : est.nombre;
      $('<option></option>').val(est.id).text(label).appendTo($estado);
    });

    const $tol = $('#tolerancia_horas');
    $tol.empty();
    tolerancias.forEach(function(item) {
      $('<option></option>').val(item.value).text(item.label).appendTo($tol);
    });
  }

  function initDataTable() {
    if (!$('#tblClientesOtif').length) {
      console.error('[Reporte Clientes OTIF] Tabla no encontrada');
      return;
    }

    dataTable = $('#tblClientesOtif').DataTable({
      processing: true,
      serverSide: false,
      responsive: true,
      language: {
        url: 'https://cdn.datatables.net/plug-ins/1.13.7/i18n/es-ES.json'
      },
      order: [[0, 'desc']],
      pageLength: 25,
      columns: [
        {
          data: 'fecha_pedido',
          title: 'Fecha pedido',
          render: function(data) {
            return data ? escapeHtml(data) : '—';
          }
        },
        {
          data: 'pedido_codigo',
          title: 'Pedido',
          render: function(data, type, row) {
            const codigo = data ? escapeHtml(data) : 'Sin código';
            const id = row && row.pedido_id ? ' #' + row.pedido_id : '';
            return '<span class="fw-semibold">' + codigo + '</span><div class="text-muted small">' + escapeHtml(id) + '</div>';
          }
        },
        {
          data: 'cliente_nombre',
          title: 'Cliente',
          render: function(data) {
            return data ? escapeHtml(data) : 'Sin cliente';
          }
        },
        {
          data: 'estado_nombre',
          title: 'Estado',
          render: function(data, type, row) {
            if (!row) {
              return data ? escapeHtml(data) : 'Sin estado';
            }
            const code = row.estado_code ? '<div class="text-muted small">' + escapeHtml(row.estado_code) + '</div>' : '';
            return '<span>' + escapeHtml(data || 'Sin estado') + '</span>' + code;
          }
        },
        {
          data: null,
          title: 'Destinos',
          className: 'text-end',
          render: function(_data, _type, row) {
            const total = Number(row.destinos_total || 0);
            const entregados = Number(row.destinos_entregados || 0);
            const pct = row.destinos_pct !== null && row.destinos_pct !== undefined
              ? formatPercent(row.destinos_pct)
              : '—';
            return '<div>' + formatNumber(entregados) + ' / ' + formatNumber(total) + '</div>' +
                   '<div class="text-muted small">Cumplimiento: ' + pct + '</div>';
          }
        },
        {
          data: null,
          title: 'Unidades',
          render: function(_data, _type, row) {
            const uc = Number(row.expected_uc || 0) > 0
              ? formatNumber(row.shipped_uc || 0) + ' / ' + formatNumber(row.expected_uc || 0)
              : '—';
            const uv = Number(row.expected_uv || 0) > 0
              ? formatNumber(row.shipped_uv || 0) + ' / ' + formatNumber(row.expected_uv || 0)
              : '—';
            return '<div><span class="text-muted">UC:</span> ' + uc + '</div>' +
                   '<div><span class="text-muted">UV:</span> ' + uv + '</div>';
          }
        },
        {
          data: null,
          title: 'Entrega',
          render: function(_data, _type, row) {
            if (!row || !row.primer_arribo_fmt) {
              return '<span class="text-muted">Sin registro</span>';
            }
            const llegada = '<div><span class="text-muted">Llegada:</span> ' + escapeHtml(row.primer_arribo_fmt) + '</div>';
            const objetivo = row.deadline_fmt
              ? '<div><span class="text-muted">Objetivo:</span> ' + escapeHtml(row.deadline_fmt) + '</div>'
              : '';
            return llegada + objetivo;
          }
        },
        {
          data: 'delta_label',
          title: 'Delta (h)',
          className: 'text-center',
          render: function(data, _type, row) {
            return buildDeltaBadge(row.delta_sign, data);
          }
        },
        {
          data: null,
          title: 'Métricas',
          render: function(_data, _type, row) {
            const onTime = buildStatusBadge(row.on_time, 'On Time');
            const inFull = buildStatusBadge(row.in_full, 'In Full');
            const otif = buildStatusBadge(row.otif, 'OTIF');
            return onTime + inFull + otif;
          }
        },
        {
          data: 'clasificacion_label',
          title: 'Clasificación',
          render: function(data) {
            return data ? escapeHtml(data) : '—';
          }
        }
      ]
    });
  }

  function loadData() {
    if (!combosLoaded) {
      return;
    }

    const formData = $('#frmClientesOtif').serialize();

    $.ajax({
      url: joinUrl('api/reportes/clientes_otif.php'),
      method: 'GET',
      data: formData,
      dataType: 'json'
    })
      .done(function(response) {
        if (!response || !response.ok) {
          const message = response && response.error ? response.error : 'No fue posible obtener los datos.';
          notify('error', 'Reporte OTIF', message);
          clearData();
          return;
        }

        currentData = Array.isArray(response.rows) ? response.rows : [];
        currentSummary = response.summary || null;
        currentAggregates = response.aggregates || {};
        currentLimit = response.limit || currentData.length || 0;
        currentTruncated = Boolean(response.truncated);

        refreshTable();
        updateSummaryCard();
        renderClientes(currentAggregates.clientes || []);
        renderClasificaciones(currentAggregates.clasificaciones || []);
        renderEstados(currentAggregates.estados || []);
        updateLimitNotice();
      })
      .fail(function(xhr) {
        console.error('[Reporte Clientes OTIF] Error Ajax', xhr);
        notify('error', 'Reporte OTIF', 'No se pudo cargar el reporte.');
        clearData();
      });
  }

  function clearData() {
    currentData = [];
    currentSummary = null;
    currentAggregates = {};
    currentLimit = 0;
    currentTruncated = false;
    refreshTable();
    updateSummaryCard();
    renderClientes([]);
    renderClasificaciones([]);
    renderEstados([]);
    updateLimitNotice();
  }

  function refreshTable() {
    if (!dataTable) {
      return;
    }
    dataTable.clear();
    if (currentData.length) {
      dataTable.rows.add(currentData);
    }
    dataTable.draw();
    $('#badgeTotalPedidos').text(formatNumber(currentData.length));
  }

  function updateSummaryCard() {
    if (!currentSummary || !currentSummary.total_pedidos) {
      $('#summaryCard').hide();
      $('#summaryRange, #summaryTolerance').text('');
      $('#summaryHighlight').hide();
      $('#sumTotalPedidos, #sumPedidosEntrega, #sumOtif, #sumOtifPct, #sumOnTime, #sumOnTimePct, #sumInFull, #sumInFullPct, #sumDestinos, #sumDestinosPct, #sumAvgDelta, #sumMedianDelta').text('-');
      return;
    }

    const total = Number(currentSummary.total_pedidos || 0);
    const conEntrega = Number(currentSummary.pedidos_con_entrega || 0);
    const sinEntrega = Number(currentSummary.pedidos_sin_entrega || 0);

    $('#sumTotalPedidos').text(formatNumber(total));
    $('#sumPedidosEntrega').text('Con entrega: ' + formatNumber(conEntrega) + ' · Pendientes: ' + formatNumber(sinEntrega));

    $('#sumOtif').text(formatNumber(currentSummary.otif_total || 0));
    $('#sumOtifPct').text(formatPercent(currentSummary.otif_pct || 0));

    $('#sumOnTime').text(formatNumber(currentSummary.on_time_total || 0));
    $('#sumOnTimePct').text(formatPercent(currentSummary.on_time_pct || 0));

    $('#sumInFull').text(formatNumber(currentSummary.in_full_total || 0));
    $('#sumInFullPct').text(formatPercent(currentSummary.in_full_pct || 0));

    const destinosTotal = Number(currentSummary.destinos_total || 0);
    const destinosEntregados = Number(currentSummary.destinos_entregados || 0);
    $('#sumDestinos').text(formatNumber(destinosEntregados) + ' / ' + formatNumber(destinosTotal));
    $('#sumDestinosPct').text('Cumplimiento: ' + formatPercent(currentSummary.destinos_pct || 0));

    $('#sumAvgDelta').text(formatHours(currentSummary.avg_delta_horas));
    $('#sumMedianDelta').text('Mediana: ' + formatHours(currentSummary.median_delta_horas));

    $('#summaryRange').text(currentSummary.range_label || '');
    $('#summaryTolerance').text('Tolerancia OTIF: ' + formatNumber(currentSummary.tolerancia_horas || 0) + ' h');

    const highlightParts = [];
    if (currentSummary.avg_delta_label) {
      highlightParts.push('Retraso promedio (pedidos con entrega): ' + currentSummary.avg_delta_label);
    }
    if (sinEntrega > 0) {
      highlightParts.push('Pedidos sin entrega registrada: ' + formatNumber(sinEntrega));
    }
    if (highlightParts.length) {
      $('#summaryHighlightText').text(highlightParts.join(' · '));
      $('#summaryHighlight').show();
    } else {
      $('#summaryHighlight').hide();
      $('#summaryHighlightText').text('');
    }

    $('#summaryCard').show();
  }

  function renderClientes(list) {
    const $card = $('#cardClientes');
    const $tbody = $('#tablaClientes tbody');
    const $summary = $('#clientesSummary');

    if (!Array.isArray(list) || list.length === 0) {
      $tbody.html('<tr class="text-muted"><td colspan="5" class="text-center py-4">Sin datos disponibles</td></tr>');
      $summary.text('');
      $card.hide();
      return;
    }

    const totalPedidos = list.reduce(function(acc, item) {
      return acc + Number(item.pedidos || 0);
    }, 0);

    const rowsHtml = list.map(function(item) {
      return (
        '<tr>' +
        '<td>' + escapeHtml(item.label || 'Sin cliente') + '</td>' +
        '<td class="text-end">' + formatNumber(item.pedidos || 0) + '</td>' +
        '<td class="text-end">' + formatPercent(item.otif_pct || 0) + '</td>' +
        '<td class="text-end">' + formatPercent(item.on_time_pct || 0) + '</td>' +
        '<td class="text-end">' + formatPercent(item.in_full_pct || 0) + '</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
    $summary.text('Top ' + list.length + ' · ' + formatNumber(totalPedidos) + ' pedidos');
    $card.show();
  }

  function renderClasificaciones(list) {
    const $card = $('#cardClasificaciones');
    const $tbody = $('#tablaClasificaciones tbody');
    const $summary = $('#clasificacionesSummary');

    if (!Array.isArray(list) || list.length === 0) {
      $tbody.html('<tr class="text-muted"><td colspan="3" class="text-center py-4">Sin datos disponibles</td></tr>');
      $summary.text('');
      $card.hide();
      return;
    }

    const total = list.reduce(function(acc, item) {
      return acc + Number(item.pedidos || 0);
    }, 0);

    const rowsHtml = list.map(function(item) {
      return (
        '<tr>' +
        '<td>' + escapeHtml(item.label || '-') + '</td>' +
        '<td class="text-end">' + formatNumber(item.pedidos || 0) + '</td>' +
        '<td class="text-end">' + formatPercent(item.pct || 0) + '</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
    $summary.text(formatNumber(total) + ' pedidos categorizados');
    $card.show();
  }

  function renderEstados(list) {
    const $card = $('#cardEstados');
    const $tbody = $('#tablaEstados tbody');
    const $summary = $('#estadosSummary');

    if (!Array.isArray(list) || list.length === 0) {
      $tbody.html('<tr class="text-muted"><td colspan="3" class="text-center py-4">Sin datos disponibles</td></tr>');
      $summary.text('');
      $card.hide();
      return;
    }

    const total = list.reduce(function(acc, item) {
      return acc + Number(item.pedidos || 0);
    }, 0);

    const rowsHtml = list.map(function(item) {
      const code = item.code ? '<div class="text-muted small">' + escapeHtml(item.code) + '</div>' : '';
      return (
        '<tr>' +
        '<td>' + escapeHtml(item.label || '-') + code + '</td>' +
        '<td class="text-end">' + formatNumber(item.pedidos || 0) + '</td>' +
        '<td class="text-end">' + formatPercent(item.otif_pct || 0) + '</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
    $summary.text(formatNumber(total) + ' pedidos');
    $card.show();
  }

  function updateLimitNotice() {
    if (currentTruncated && currentLimit) {
      $('#limitValue').text(formatNumber(currentLimit));
      $('#limitNotice').show();
    } else {
      $('#limitNotice').hide();
    }
  }

  function notify(type, title, text) {
    if (typeof Swal !== 'undefined') {
      Swal.fire({ icon: type, title, text, timer: 3200, showConfirmButton: false });
      return;
    }
    const message = '[' + title + '] ' + text;
    if (type === 'error') {
      console.error(message);
    } else if (type === 'warning') {
      console.warn(message);
    } else {
      console.info(message);
    }
  }

})(window, jQuery);
