/**
 * SOL · JavaScript · Reporte Cobertura por SKU
 */

(function(window, $) {
  'use strict';

  if (!$ || typeof $.ajax !== 'function') {
    console.error('[Cobertura SKU] jQuery es requerido.');
    return;
  }

  const baseUrl = document.querySelector('meta[name="base-url"]')?.getAttribute('content') ||
                  (typeof window.BASE_URL !== 'undefined' ? window.BASE_URL : '/');

  function joinUrl(path) {
    return baseUrl.replace(/\/$/, '') + '/' + String(path || '').replace(/^\//, '');
  }

  const intFormatter = new Intl.NumberFormat('es-AR', { minimumFractionDigits: 0, maximumFractionDigits: 0 });
  const decimalFormatter = new Intl.NumberFormat('es-AR', { minimumFractionDigits: 2, maximumFractionDigits: 2 });

  function formatInt(value, fallback) {
    const numeric = Number(value);
    if (Number.isFinite(numeric)) {
      return intFormatter.format(numeric);
    }
    return fallback !== undefined ? fallback : '0';
  }

  function formatDecimal(value, fallback) {
    const numeric = Number(value);
    if (Number.isFinite(numeric)) {
      return decimalFormatter.format(numeric);
    }
    return fallback !== undefined ? fallback : '0,00';
  }

  function formatDays(value) {
    const numeric = Number(value);
    if (!Number.isFinite(numeric)) {
      return '—';
    }
    if (numeric > 3650) {
      return '∞';
    }
    return decimalFormatter.format(numeric).replace(/,00$/, '');
  }

  function formatDate(value) {
    if (!value) { return '—'; }
    const str = String(value).slice(0, 10);
    const parts = str.split('-');
    if (parts.length !== 3) { return str; }
    return parts[2] + '/' + parts[1] + '/' + parts[0];
  }

  function showAlert(type, message) {
    if (window.Swal && typeof window.Swal.fire === 'function') {
      window.Swal.fire({ icon: type, title: message, confirmButtonText: 'Cerrar' });
      return;
    }
    window.alert(message);
  }

  const $form = $('#frmCoberturaSku');
  const $sku = $('#sku');
  const $productoId = $('#producto_id');
  const $btnCalcular = $('#btnCalcular');
  const $btnLimpiar = $('#btnLimpiar');
  const $alertIndicaciones = $('#alertIndicaciones');
  const $alertSinConsumo = $('#alertSinConsumo');
  const $cardResultado = $('#cardResultado');
  const $tablaHorizontes = $('#tablaHorizontes tbody');
  const $bloqueSerie = $('#bloqueSerie');
  const $listaConsumo = $('#listaConsumo');
  const $listaNotas = $('#listaNotas');

  const productoTitulo = document.getElementById('productoTitulo');
  const productoSubtitulo = document.getElementById('productoSubtitulo');
  const lblRango = document.getElementById('lblRangoAnalizado');
  const lblUltimo = document.getElementById('lblUltimoMovimiento');

  const dataFields = {
    stock_disponible: document.querySelector('[data-field="stock_disponible"]'),
    stock_reservado: document.querySelector('[data-field="stock_reservado"]'),
    stock_total: document.querySelector('[data-field="stock_total"]'),
    consumo_promedio: document.querySelector('[data-field="consumo_promedio"]'),
    consumo_total: document.querySelector('[data-field="consumo_total"]'),
    dias_cobertura: document.querySelector('[data-field="dias_cobertura"]'),
    dias_sin_consumo: document.querySelector('[data-field="dias_sin_consumo"]'),
    stock_minimo: document.querySelector('[data-field="stock_minimo"]'),
    stock_maximo: document.querySelector('[data-field="stock_maximo"]'),
    documentos_total: document.querySelector('[data-field="documentos_total"]'),
    documentos_promedio: document.querySelector('[data-field="documentos_promedio"]'),
  };

  const skuSuggestions = document.getElementById('skuSuggestions');
  const suggestionMap = new Map();

  $(document).ready(function() {
    setDefaultRange();
    bindEvents();
  });

  function setDefaultRange() {
    const today = new Date();
    const past = new Date(today.getTime() - (90 * 24 * 60 * 60 * 1000));
    $('#fecha_hasta').val(today.toISOString().slice(0, 10));
    $('#fecha_desde').val(past.toISOString().slice(0, 10));
  }

  function bindEvents() {
    $form.on('submit', function(evt) {
      evt.preventDefault();
      ejecutarReporte();
    });

    $btnLimpiar.on('click', function() {
      $form.trigger('reset');
      setDefaultRange();
      $productoId.val('');
      suggestionMap.clear();
      if (skuSuggestions) {
        skuSuggestions.innerHTML = '';
      }
      $cardResultado.hide();
      $alertSinConsumo.hide();
      $alertIndicaciones.show();
    });

    $sku.on('input', debounce(function() {
      const query = String(this.value || '').trim();
      $productoId.val('');
      if (query.length < 3) {
        suggestionMap.clear();
        if (skuSuggestions) { skuSuggestions.innerHTML = ''; }
        return;
      }
      fetchSkuSuggestions(query);
    }, 250));

    $sku.on('change', function() {
      const value = String(this.value || '').trim().toUpperCase();
      if (!value || !skuSuggestions) {
        $productoId.val('');
        return;
      }
      const option = Array.from(skuSuggestions.options || []).find(function(opt) {
        return String(opt.value || '').trim().toUpperCase() === value;
      });
      if (option && option.dataset.id) {
        $productoId.val(option.dataset.id);
      }
    });
  }

  function debounce(fn, delay) {
    let timer = null;
    return function() {
      const ctx = this;
      const args = arguments;
      clearTimeout(timer);
      timer = setTimeout(function() { fn.apply(ctx, args); }, delay);
    };
  }

  function fetchSkuSuggestions(term) {
    const url = joinUrl('api/reportes/cobertura_sku.php');
    $.ajax({
      url: url,
      method: 'GET',
      dataType: 'json',
      data: { meta: 'productos', term: term },
    }).done(function(resp) {
      suggestionMap.clear();
      if (!skuSuggestions) { return; }
      skuSuggestions.innerHTML = '';
      if (!resp || !Array.isArray(resp.productos)) {
        return;
      }
      resp.productos.forEach(function(prod) {
        const sku = String(prod.sku || '').toUpperCase();
        const label = sku + ' · ' + String(prod.nombre || '');
        suggestionMap.set(sku, prod);
        const option = document.createElement('option');
        option.value = sku;
        option.textContent = label;
        option.dataset.id = String(prod.id || '');
        skuSuggestions.appendChild(option);
      });
    }).fail(function() {
      // Sugerencias no críticas, silencioso
    });
  }

  function ejecutarReporte() {
    const formEl = $form.get(0);
    if (!formEl) { return; }

    const formData = new FormData(formEl);
    const params = new URLSearchParams();
    formData.forEach(function(value, key) {
      if (value !== null && value !== undefined) {
        params.append(key, String(value));
      }
    });

    toggleLoading(true);
    $alertIndicaciones.hide();
    $alertSinConsumo.hide();

    $.ajax({
      url: joinUrl('api/reportes/cobertura_sku.php') + '?' + params.toString(),
      method: 'GET',
      dataType: 'json',
    }).done(function(resp) {
      if (!resp || resp.ok !== true) {
        showAlert('error', resp && resp.error ? resp.error : 'No se pudo generar el reporte.');
        return;
      }
      renderResultado(resp);
    }).fail(function(xhr) {
      let message = 'No se pudo generar el reporte.';
      if (xhr && xhr.responseJSON && xhr.responseJSON.error) {
        message = xhr.responseJSON.error;
      }
      showAlert('error', message);
    }).always(function() {
      toggleLoading(false);
    });
  }

  function toggleLoading(isLoading) {
    if (isLoading) {
      $btnCalcular.prop('disabled', true).data('original-text', $btnCalcular.html()).html('<span class="spinner-border spinner-border-sm me-1" role="status"></span>Calculando');
    } else {
      const original = $btnCalcular.data('original-text');
      $btnCalcular.prop('disabled', false).html(original || '<i class="bi bi-graph-up-arrow me-1"></i>Calcular');
    }
  }

  function renderResultado(data) {
    if (!data || !data.producto) {
      showAlert('warning', 'No se recibió información del producto.');
      return;
    }

    productoTitulo.textContent = data.producto.nombre || 'Producto sin nombre';
    const subtitleParts = [];
    if (data.producto.sku) {
      subtitleParts.push('SKU ' + data.producto.sku);
    }
    if (data.producto.cliente) {
      subtitleParts.push('Cliente: ' + data.producto.cliente);
    }
    if (data.producto.operativa) {
      subtitleParts.push('Operativa: ' + data.producto.operativa);
    }
    productoSubtitulo.textContent = subtitleParts.join(' · ') || '—';

    if (data.filters) {
      lblRango.textContent = 'Período: ' + formatDate(data.filters.fecha_desde) + ' al ' + formatDate(data.filters.fecha_hasta);
    }

    if (data.consumo && data.consumo.ultima_salida) {
      lblUltimo.textContent = 'Última salida registrada: ' + formatDate(data.consumo.ultima_salida);
    } else {
      lblUltimo.textContent = 'Última salida registrada: —';
    }

    updateMetrics(data);
    renderHorizontes(data);
    renderSerie(data);

    $cardResultado.show();

    if (!data.consumo || Number(data.consumo.total_salidas_uc || 0) <= 0) {
      $alertSinConsumo.show();
    }
  }

  function updateMetrics(data) {
    const stock = data.stock || {};
    const consumo = data.consumo || {};
    const producto = data.producto || {};

    setFieldText('stock_disponible', formatDecimal(stock.disponibles_uc || 0));
    setFieldText('stock_reservado', formatDecimal(stock.reservados_uc || 0));
    setFieldText('stock_total', formatDecimal(stock.stock_uc || 0));

    setFieldText('consumo_promedio', formatDecimal(consumo.promedio_diario_uc || 0));
    setFieldText('consumo_total', formatDecimal(consumo.total_salidas_uc || 0));
  setFieldText('documentos_total', formatInt(consumo.total_documentos || 0));
  setFieldText('documentos_promedio', formatDecimal(consumo.promedio_diario_documentos || 0));

    if (consumo.dias_cobertura === null || consumo.dias_cobertura === undefined) {
      setFieldText('dias_cobertura', Number(consumo.total_salidas_uc || 0) > 0 ? 'No definido' : 'Sin consumo');
    } else {
      setFieldText('dias_cobertura', formatDays(consumo.dias_cobertura));
    }

    setFieldText('dias_sin_consumo', formatInt(consumo.dias_sin_consumo || 0));

    setFieldText('stock_minimo', producto.stock_min !== undefined ? formatDecimal(producto.stock_min || 0) : '—');
    setFieldText('stock_maximo', producto.stock_max !== undefined && producto.stock_max !== null ? formatDecimal(producto.stock_max) : '—');
  }

  function setFieldText(field, text) {
    if (dataFields[field]) {
      dataFields[field].textContent = text;
    }
  }

  function renderHorizontes(data) {
    const horizons = data.cobertura && Array.isArray(data.cobertura.horizontes)
      ? data.cobertura.horizontes
      : [];

    if (!horizons.length) {
      $tablaHorizontes.html('<tr class="text-muted"><td colspan="6" class="text-center py-4">Sin datos calculados.</td></tr>');
      return;
    }

    const rows = horizons.map(function(h) {
      const dias = Number(h.dias || 0);
      const consumoEstimado = formatDecimal(h.consumo_estimado_uc || 0);
      const stockActual = formatDecimal(h.stock_actual_uc || 0);
      const faltante = Number(h.faltante_uc || 0);
      const sobrante = Number(h.sobrante_uc || 0);
      let balanceLabel = 'Equilibrado';
      if (faltante > 0.001) {
        balanceLabel = 'Faltan ' + formatDecimal(faltante) + ' UC';
      } else if (sobrante > 0.001) {
        balanceLabel = 'Sobran ' + formatDecimal(sobrante) + ' UC';
      }
      const diasCubiertos = h.dias_cubiertos !== null && h.dias_cubiertos !== undefined
        ? formatDays(h.dias_cubiertos)
        : '—';
      const alcanza = !!h.alcance;
      const badgeClass = alcanza ? 'bg-success' : 'bg-danger';
      const badgeText = alcanza ? 'Alcanza' : 'Faltante';
      const rowClass = alcanza ? 'table-success' : 'table-danger';

      return '<tr class="' + rowClass + '">' +
        '<td>' + dias + ' días</td>' +
        '<td class="text-end">' + consumoEstimado + '</td>' +
        '<td class="text-end">' + stockActual + '</td>' +
        '<td class="text-end">' + diasCubiertos + '</td>' +
        '<td class="text-end">' + balanceLabel + '</td>' +
        '<td class="text-center"><span class="badge ' + badgeClass + '">' + badgeText + '</span></td>' +
      '</tr>';
    }).join('');

    $tablaHorizontes.html(rows);
  }

  function renderSerie(data) {
    const serie = data.consumo && Array.isArray(data.consumo.serie_diaria)
      ? data.consumo.serie_diaria
      : [];

    if (!serie.length) {
      $bloqueSerie.hide();
      $listaConsumo.empty();
      return;
    }

    const recientes = serie.slice(-12).reverse();
    const items = recientes.map(function(entry) {
      const fecha = formatDate(entry.fecha);
      const cantidad = formatDecimal(entry.salidas_uc || 0);
      const docs = Number(entry.documentos || 0);
      return '<div class="border rounded px-3 py-2 d-flex justify-content-between align-items-center">' +
        '<div><span class="badge badge-dot text-primary">&nbsp;</span>' + fecha + (docs ? '<div class="small text-muted mt-1">' + formatInt(docs) + ' pedidos</div>' : '') + '</div>' +
        '<div class="fw-semibold text-end">' + cantidad + ' UC</div>' +
      '</div>';
    }).join('');

    $listaConsumo.html(items);

    if ($listaNotas.length) {
      const consumo = data.consumo || {};
      const stock = data.stock || {};
      const promedio = formatDecimal(consumo.promedio_diario_uc || 0);
      const disponible = formatDecimal(stock.disponibles_uc || 0);
      const alcance = consumo.dias_cobertura === null || consumo.dias_cobertura === undefined
        ? (Number(consumo.total_salidas_uc || 0) > 0 ? 'Sin determinar por baja rotación.' : 'No hay salidas registradas en el período.')
        : formatDays(consumo.dias_cobertura) + ' días estimados con stock actual.';
      $listaNotas.html([
        '<li class="text-muted">Promedio diario considerado: <strong>' + promedio + ' UC</strong>.</li>',
        '<li class="text-muted">Consumo calculado desde pedidos y embarques despachados del SKU.</li>',
        '<li class="text-muted">Stock disponible al momento del cálculo: <strong>' + disponible + ' UC</strong>.</li>',
        '<li class="text-muted">Cobertura estimada: <strong>' + alcance + '</strong></li>'
      ].join(''));
    }

    $bloqueSerie.show();
  }

})(window, window.jQuery);
