/**
 * SOL · Reporte de Distribución con semáforo
 */
(function (window, $) {
  'use strict';

  const baseUrl =
    document.querySelector('meta[name="base-url"]')?.getAttribute('content') ||
    (typeof window.BASE_URL !== 'undefined' ? window.BASE_URL : '/');
  const joinUrl = (path) => `${baseUrl.replace(/\/$/, '')}/${String(path || '').replace(/^\//, '')}`;
  const apiUrl = joinUrl('api/reportes/distribucion.php');

  const $form = $('#frmDistribucion');
  const $indicadores = $('#indicadoresContainer');
  const $periodoLabel = $('#periodoLabel');
  const $badgeEntregas = $('#badgeEntregas');

  $(function () {
    bindEvents();
    loadFilters().always(loadData);
  });

  function bindEvents() {
    $form.on('submit', function (event) {
      event.preventDefault();
      loadData();
    });

    $('#btnLimpiar').on('click', function () {
      const formEl = $form.get(0);
      if (formEl) {
        formEl.reset();
      }
      $('#deposito_id').val('');
      $('#cliente_id').val('');
      $('#movil_id').val('');
      loadData();
    });
  }

  function loadFilters() {
    return $.getJSON(apiUrl, { meta: 'filters' })
      .done(function (response) {
        if (!response || response.ok === false) {
          notifier('warning', response?.error || 'No se pudieron cargar los filtros');
          return;
        }
        fillSelect('#deposito_id', response.depositos, 'Todos');
        fillSelect('#cliente_id', response.clientes, 'Todos');
        fillSelect('#movil_id', response.moviles, 'Todos');
      })
      .fail(function () {
        notifier('error', 'Error de conexión al cargar filtros');
      });
  }

  function fillSelect(selector, items, placeholder) {
    const $select = $(selector);
    if (!$select.length) return;
    const current = $select.val();
    $select.find('option').not(':first').remove();
    if (Array.isArray(items)) {
      items.forEach(function (item) {
        const id = item.id ?? '';
        const label = item.label ?? item.nombre ?? item.descripcion ?? item.codigo ?? `ID ${id}`;
        $select.append(new Option(label, id));
      });
    }
    if (current && $select.find(`option[value="${current}"]`).length) {
      $select.val(current);
    } else {
      $select.val('');
    }
  }

  function loadData() {
    const formEl = $form.get(0);
    if (!formEl) {
      return;
    }
    const params = Object.fromEntries(new FormData(formEl).entries());
    setLoading(true);

    $.ajax({
      url: apiUrl,
      method: 'GET',
      data: params,
      dataType: 'json',
    })
      .done(function (response) {
        if (!response || response.ok === false) {
          notifier('error', response?.error || 'No se pudo generar el reporte');
          resetView();
          return;
        }
        updateFormFilters(response.filters || {});
        updateIndicators(response.metrics || []);
        updateSummary(response.summary || {});
        updatePeriodo(response.periodo || '');
      })
      .fail(function () {
        notifier('error', 'Error de conexión con el servidor');
        resetView();
      })
      .always(function () {
        setLoading(false);
      });
  }

  function updateFormFilters(filters) {
    if (!filters) {
      return;
    }
    $('#fecha_desde').val(filters.fecha_desde ?? '');
    $('#fecha_hasta').val(filters.fecha_hasta ?? '');
    $('#tolerancia_min').val(filters.tolerancia_min ?? 60);
    if (filters.cliente_id) {
      $('#cliente_id').val(String(filters.cliente_id));
    }
    if (filters.deposito_id) {
      $('#deposito_id').val(String(filters.deposito_id));
    }
    if (filters.movil_id) {
      $('#movil_id').val(String(filters.movil_id));
    }
    if (filters.combustible_meta !== undefined && filters.combustible_meta !== null) {
      $('#combustible_meta').val(filters.combustible_meta);
    }
  }

  function updateIndicators(metrics) {
    const mapByKey = {};
    if (Array.isArray(metrics)) {
      metrics.forEach(function (metric) {
        if (metric && metric.clave) {
          mapByKey[metric.clave] = metric;
        }
      });
    }

    $indicadores.find('[data-indicador]').each(function () {
      const $wrapper = $(this);
      const clave = $wrapper.data('indicador');
      const metric = mapByKey[clave] || null;
      const $card = $wrapper.find('.card-indicador');
      const estado = metric?.estado || 'SIN_DATOS';
      const estadoClass = mapEstadoToClass(estado);
      const badgeClass = mapEstadoToBadge(estado);
      const valorClass = mapEstadoToValueClass(estado);

      $card.removeClass('ok alerta critico sin-datos').addClass(estadoClass);
      const $title = $card.find('[data-field="titulo"]');
      $title.text(metric?.indicador || $title.text());

      const metaText = metric?.meta !== null && metric?.meta !== undefined
        ? `Meta: ${formatValue(metric.meta, metric.unidad)}`
        : 'Meta: -';
      const $badge = $card.find('[data-field="meta"]');
      $badge.attr('class', `badge badge-indicador ${badgeClass}`).text(metaText);

      const $valor = $card.find('[data-field="valor"]');
      const valorTexto = metric?.valor !== null && metric?.valor !== undefined
        ? formatValue(metric.valor, metric.unidad)
        : '-';
      $valor.removeClass('text-success text-warning text-danger text-muted').addClass(valorClass).text(valorTexto);

      const observacion = metric?.observaciones || 'Sin datos disponibles';
      $card.find('[data-field="observacion"]').text(observacion);
    });
  }

  function updateSummary(summary) {
    const entregas = summary?.entregas ?? 0;
    const enPlazo = summary?.en_plazo ?? 0;
    const conRetraso = summary?.con_retraso ?? 0;
    const kmTotales = summary?.km_totales ?? null;
    const kmPromedio = summary?.km_promedio ?? null;
    const combustible = summary?.combustible_litros ?? null;
    const devoluciones = summary?.devoluciones ?? 0;
    const devolucionesPct = summary?.devoluciones_pct ?? null;
    const costos = summary?.costos ?? null;

    $badgeEntregas.text(`${entregas} entregas`);

    $('#resEntregaTotal').text(formatNumber(entregas, 0));
    $('#resEntregaPlazo').text(`${formatNumber(enPlazo, 0)} (${formatPercentSafe(summary?.puntualidad_pct)})`);
    $('#resEntregaRetraso').text(formatNumber(conRetraso, 0));
    $('#resKmTotales').text(kmTotales !== null ? `${formatNumber(kmTotales, 1)} km` : '-');
    $('#resCombustible').text(combustible !== null ? `${formatNumber(combustible, 1)} L` : '-');
    $('#resDevoluciones').text(`${formatNumber(devoluciones, 0)} (${formatPercentSafe(devolucionesPct)})`);
    $('#resCostos').text(costos !== null ? formatCurrency(costos) : '-');

    if (kmPromedio !== null) {
      $('#resKmTotales').attr('title', `Promedio ${formatNumber(kmPromedio, 2)} km por entrega`);
    } else {
      $('#resKmTotales').removeAttr('title');
    }
  }

  function updatePeriodo(periodo) {
    if (periodo) {
      $periodoLabel.text(`Periodo: ${periodo}`);
    } else {
      $periodoLabel.text('Seleccione filtros para ver resultados');
    }
  }

  function resetView() {
    updateIndicators([]);
    updateSummary({});
    updatePeriodo('');
  }

  function setLoading(isLoading) {
    if (isLoading) {
      $indicadores.addClass('opacity-50 position-relative');
      $indicadores.find('.loading-overlay').remove();
      $indicadores.append('<div class="loading-overlay position-absolute top-50 start-50 translate-middle text-muted"><div class="spinner-border spinner-border-sm me-2" role="status"></div>Procesando...</div>');
    } else {
      $indicadores.removeClass('opacity-50 position-relative');
      $indicadores.find('.loading-overlay').remove();
    }
  }

  function formatValue(value, unidad) {
    if (value === null || value === undefined) {
      return '-';
    }
    const decimals = unidad === '%' ? 2 : unidad === 'km' ? 1 : 1;
    const formatted = formatNumber(value, decimals);
    if (!unidad) {
      return formatted;
    }
    if (unidad === '%') {
      return `${formatted}%`;
    }
    return `${formatted} ${unidad}`.trim();
  }

  function formatNumber(value, decimals = 0) {
    const num = Number(value);
    if (!Number.isFinite(num)) {
      return '-';
    }
    try {
      return new Intl.NumberFormat('es-AR', {
        minimumFractionDigits: decimals,
        maximumFractionDigits: decimals,
      }).format(num);
    } catch (err) {
      return num.toFixed(decimals);
    }
  }

  function formatPercentSafe(value) {
    if (value === null || value === undefined || Number.isNaN(Number(value))) {
      return '-';
    }
    return `${formatNumber(value, 2)}%`;
  }

  function formatCurrency(value) {
    if (value === null || value === undefined || Number.isNaN(Number(value))) {
      return '-';
    }
    const num = Number(value);
    try {
      return `Gs. ${new Intl.NumberFormat('es-AR', {
        minimumFractionDigits: 0,
        maximumFractionDigits: 0,
      }).format(num)}`;
    } catch (err) {
      return `Gs. ${Math.round(num)}`;
    }
  }

  function mapEstadoToClass(estado) {
    switch (estado) {
      case 'OK':
        return 'ok';
      case 'ALERTA':
        return 'alerta';
      case 'CRITICO':
        return 'critico';
      default:
        return 'sin-datos';
    }
  }

  function mapEstadoToBadge(estado) {
    switch (estado) {
      case 'OK':
        return 'bg-success';
      case 'ALERTA':
        return 'bg-warning text-dark';
      case 'CRITICO':
        return 'bg-danger';
      default:
        return 'bg-secondary';
    }
  }

  function mapEstadoToValueClass(estado) {
    switch (estado) {
      case 'OK':
        return 'text-success';
      case 'ALERTA':
        return 'text-warning';
      case 'CRITICO':
        return 'text-danger';
      default:
        return 'text-muted';
    }
  }

  function notifier(type, message) {
    if (typeof Swal !== 'undefined' && typeof Swal.fire === 'function') {
      Swal.fire({ icon: type, title: message, timer: 2600, showConfirmButton: false });
      return;
    }
    if (type === 'error') {
      console.error('[Distribución]', message);
    } else if (type === 'warning') {
      console.warn('[Distribución]', message);
    } else {
      console.info('[Distribución]', message);
    }
  }
})(window, window.jQuery);
