/**
 * SOL · JavaScript · Reporte de Cumplimiento de Entregas
 * Ruta: public/page-scripts/pages/reportes/embarques_cumplimiento.js
 */

(function(window, $) {
  'use strict';

  // Base URL
  const baseUrl = document.querySelector('meta[name="base-url"]')?.getAttribute('content') || 
                 (typeof window.BASE_URL !== 'undefined' ? window.BASE_URL : '/');

  function joinUrl(path) {
    return baseUrl.replace(/\/$/, '') + '/' + path.replace(/^\//, '');
  }

  // Utility functions
  function notify(type, title, text) {
    if (type === 'success') {
      // Avoid intrusive alerts on regular flows; log instead.
      console.info(`[Cumplimiento] ${title}: ${text}`);
      return;
    }

    if (typeof Swal !== 'undefined') {
      Swal.fire({ icon: type, title, text, timer: 3500, showConfirmButton: false });
      return;
    }

    const message = `${title}: ${text}`;
    if (type === 'error') {
      console.error(message);
    } else {
      console.warn(message);
    }
  }

  function formatNumber(num) {
    return new Intl.NumberFormat('es-AR').format(num || 0);
  }

  function formatPercent(num) {
    if (num === null || num === undefined || Number.isNaN(num)) {
      return '0%';
    }
    const formatter = new Intl.NumberFormat('es-AR', {
      minimumFractionDigits: 1,
      maximumFractionDigits: 1
    });
    return `${formatter.format(num)}%`;
  }

  function formatDate(dateStr) {
    if (!dateStr) return '-';
    return new Date(dateStr).toLocaleDateString('es-AR');
  }

  function formatTime(timeStr) {
    if (!timeStr) return '-';
    return new Date('1970-01-01T' + timeStr).toLocaleTimeString('es-AR', {
      hour: '2-digit',
      minute: '2-digit'
    });
  }

  // Global variables
  let dataTable = null;
  let currentData = [];
  let filterOptions = {};
  let filtersLoaded = false;
  let summaryMetrics = null;

  // Initialize page
  $(document).ready(function() {
    initializePage();
    bindEvents();
    loadInitialData();
  });

  function initializePage() {
    // Set default date range (last 30 days)
    const today = new Date();
    const thirtyDaysAgo = new Date(today.getTime() - (30 * 24 * 60 * 60 * 1000));
    
    $('#fecha_hasta').val(today.toISOString().split('T')[0]);
    $('#fecha_desde').val(thirtyDaysAgo.toISOString().split('T')[0]);

    // Initialize DataTable
    if (!$('#tblCumplimiento').length) {
      console.error('Table element #tblCumplimiento not found');
      return;
    }

    initializeDataTable();
  }

  function initializeDataTable() {
    try {
      dataTable = $('#tblCumplimiento').DataTable({
        processing: true,
        serverSide: false,
        responsive: true,
        language: {
          url: "https://cdn.datatables.net/plug-ins/1.13.7/i18n/es-ES.json"
        },
        order: [[1, 'desc']], // Order by date desc
        columnDefs: [
          { targets: [4, 5, 6, 7, 10, 11], className: 'text-center' },
          { targets: [8, 9], className: 'text-end' }
        ],
        columns: [
          { data: 'codigo', title: 'Código' },
          { 
            data: 'fecha_embarque', 
            title: 'Fecha',
            render: function(data) { return formatDate(data); }
          },
          { data: 'estado_nombre', title: 'Estado' },
          { 
            data: 'movil', 
            title: 'Móvil',
            render: function(data) { return data || '-'; }
          },
          { 
            data: 'total_destinos', 
            title: 'Destinos',
            render: function(data) { return formatNumber(data); }
          },
          { 
            data: 'pedidos_esperados', 
            title: 'Pedidos Esp.',
            render: function(data) { return formatNumber(data); }
          },
          { 
            data: 'pedidos_despachados', 
            title: 'Pedidos Desp.',
            render: function(data) { return formatNumber(data); }
          },
          { 
            data: 'destinos_entregados', 
            title: 'Destinos Entr.',
            render: function(data) { return formatNumber(data); }
          },
          { 
            data: null,
            title: 'Items UV',
            render: function(data, type, row) {
              const desp = formatNumber(row.items_despachados_uv);
              const esp = formatNumber(row.items_esperados_uv);
              return `${desp} / ${esp}`;
            }
          },
          { 
            data: null,
            title: 'Items UC',
            render: function(data, type, row) {
              const desp = formatNumber(row.items_despachados_uc);
              const esp = formatNumber(row.items_esperados_uc);
              return `${desp} / ${esp}`;
            }
          },
          { 
            data: null,
            title: 'Rate Cumpl.',
            render: function(data, type, row) {
              const rate = Math.round(Number(row.fulfillment_rate ?? 0));
              let badgeClass = 'bg-secondary';
              if (rate >= 95) badgeClass = 'bg-success';
              else if (rate >= 80) badgeClass = 'bg-warning';
              else if (rate > 0) badgeClass = 'bg-danger';
              
              return `<span class="badge ${badgeClass}">${rate}%</span>`;
            }
          },
          { 
            data: 'delay_days',
            title: 'Delay (días)',
            render: function(data) {
              if (data === null || data === undefined || data === '') return '-';
              const days = parseInt(data, 10);
              if (!Number.isFinite(days)) {
                return '-';
              }
              let badgeClass = 'bg-success';
              if (days > 3) badgeClass = 'bg-danger';
              else if (days > 1) badgeClass = 'bg-warning';
              return `<span class="badge ${badgeClass}">${days}</span>`;
            }
          },
          { 
            data: 'delivery_status',
            title: 'Status',
            render: function(data) {
              const statusMap = {
                COMPLETADO: '<span class="badge bg-success">Completado</span>',
                COMPLETADO_TARDE: '<span class="badge bg-primary">Completado (Tarde)</span>',
                EN_TRANSITO: '<span class="badge bg-info">En Tránsito</span>',
                PARCIAL: '<span class="badge bg-warning text-dark">Parcial</span>',
                PENDIENTE: '<span class="badge bg-secondary">Pendiente</span>',
                FALLIDO: '<span class="badge bg-danger">Fallido</span>'
              };
              return statusMap[data] || '<span class="badge bg-secondary">-</span>';
            }
          }
        ]
      });
    } catch (error) {
      console.error('Error initializing DataTable:', error);
      notify('error', 'Error', 'Error al inicializar la tabla');
    }
  }

  function bindEvents() {
    // Form submission
    $('#frmFiltros').on('submit', function(e) {
      e.preventDefault();
      loadData();
    });

    // Clear filters
    $('#btnLimpiar').on('click', function() {
      $('#frmFiltros')[0].reset();
      
      // Reset default dates
      const today = new Date();
      const thirtyDaysAgo = new Date(today.getTime() - (30 * 24 * 60 * 60 * 1000));
      $('#fecha_hasta').val(today.toISOString().split('T')[0]);
      $('#fecha_desde').val(thirtyDaysAgo.toISOString().split('T')[0]);
      
      loadData();
    });

    // Export buttons
    $('#btnExportPDF').on('click', function() {
      exportPDF();
    });

    $('#btnExportCSV').on('click', function() {
      exportCSV();
    });

    // Cliente change - filter destinatarios
    $('#cliente_id').on('change', function() {
      filterDestinatarios();
    });
  }

  function loadInitialData() {
    loadData();
  }

  function loadData() {
    const formData = $('#frmFiltros').serialize();
    
    $.ajax({
      url: joinUrl('api/reportes/embarque_cumplimiento.php'),
      method: 'GET',
      data: formData,
      dataType: 'json',
      beforeSend: function() {
        $('#btnExportPDF, #btnExportCSV').prop('disabled', true);
        $('#summarySection').hide();
        $('#destinatariosCard').hide();
        $('#periodRatesCard').hide();
      }
    })
    .done(function(response) {
      if (response.ok) {
        currentData = response.data;
        filterOptions = response.filter_options || {};
        summaryMetrics = response.metrics || null;

        if (!filtersLoaded && Object.keys(filterOptions).length) {
          updateFilterOptions(filterOptions);
          filterDestinatarios();
          filtersLoaded = true;
        }

        // Update DataTable
        if (dataTable) {
          dataTable.clear();
          dataTable.rows.add(currentData);
          dataTable.draw();
        }

        updateSummaryMetrics(summaryMetrics);
        renderDestinatarioPerformance(response.destinatarios_top || []);
        renderPeriodRates(response.period_rates || []);

        $('#btnExportPDF, #btnExportCSV').prop('disabled', currentData.length === 0);
      } else {
        notify('error', 'Error', response.error || 'Error al cargar datos');
        currentData = [];
        if (dataTable) {
          dataTable.clear().draw();
        }
        updateSummaryMetrics(null);
        renderDestinatarioPerformance([]);
        renderPeriodRates([]);
      }
    })
    .fail(function(xhr) {
      console.error('Error loading data:', xhr);
      notify('error', 'Error', 'Error de conexión al cargar datos');
      currentData = [];
      if (dataTable) {
        dataTable.clear().draw();
      }
      updateSummaryMetrics(null);
      renderDestinatarioPerformance([]);
      renderPeriodRates([]);
    })
    .always(function() {
      $('#btnExportPDF, #btnExportCSV').prop('disabled', currentData.length === 0);
    });
  }

  function updateSummaryMetrics(metrics) {
    if (!metrics || !metrics.total_embarques) {
      $('#summarySection').hide();
      $('#rateCumplimiento, #rateEntregaParcial, #rateDespacho, #ratePuntualidad').text('-');
      $('#rateCumplimientoDesc, #rateEntregaParcialDesc, #rateDespachoDesc, #ratePuntualidadDesc').text('-');
      return;
    }

    const fulfillment = metrics.fulfillment_rates || {};
    const tiempos = metrics.tiempos || {};
    const volumenes = metrics.volumenes || {};

    $('#rateCumplimiento').text(formatPercent(fulfillment.rate_cumplimiento ?? 0));
    $('#rateCumplimientoDesc').text(
      `${formatNumber(fulfillment.pedidos_completos ?? 0)} de ${formatNumber(metrics.total_embarques)} embarques`
    );

    $('#rateEntregaParcial').text(formatPercent(fulfillment.rate_entrega_parcial ?? 0));
    $('#rateEntregaParcialDesc').text(
      `${formatNumber(fulfillment.entregas_parciales ?? 0)} embarques con entrega parcial`
    );

    $('#rateDespacho').text(formatPercent(volumenes.rate_despacho_uv ?? 0));
    $('#rateDespachoDesc').text(
      `${formatNumber(volumenes.items_despachados_uv ?? 0)} / ${formatNumber(volumenes.items_esperados_uv ?? 0)} items UV`
    );

    $('#ratePuntualidad').text(formatPercent(tiempos.rate_puntualidad ?? 0));
    const atraso = formatNumber(tiempos.embarques_con_retraso ?? 0);
    $('#ratePuntualidadDesc').text(
      `${formatNumber(tiempos.embarques_a_tiempo ?? 0)} embarques a tiempo · ${atraso} con retraso`
    );

    $('#summarySection').show();
  }

  function updateFilterOptions(options) {
    if (!options) {
      return;
    }

    const $estado = $('#estado');
    $estado.find('option:not(:first)').remove();
    (options.estados || []).forEach(function(estado) {
      $estado.append(`<option value="${estado.code}">${estado.nombre}</option>`);
    });

    const $cliente = $('#cliente_id');
    $cliente.find('option:not(:first)').remove();
    (options.clientes || []).forEach(function(cliente) {
      $cliente.append(`<option value="${cliente.id}">${cliente.razon_social}</option>`);
    });

    const $destinatario = $('#destinatario_id');
    $destinatario.find('option:not(:first)').remove();
    (options.destinatarios || []).forEach(function(dest) {
      $destinatario.append(
        `<option value="${dest.id}" data-cliente="${dest.cliente_id}">${dest.cod} - ${dest.nombre}</option>`
      );
    });

    const $movil = $('#movil_id');
    $movil.find('option:not(:first)').remove();
    (options.moviles || []).forEach(function(movil) {
      $movil.append(`<option value="${movil.id}">${movil.label}</option>`);
    });
  }

  function filterDestinatarios() {
    const clienteId = $('#cliente_id').val();
    const $destinatario = $('#destinatario_id');
    
    $destinatario.find('option:not(:first)').each(function() {
      const $option = $(this);
      if (!clienteId || $option.data('cliente') == clienteId) {
        $option.show();
      } else {
        $option.hide();
      }
    });
    
    // Reset selection if hidden
    if ($destinatario.find('option:selected').is(':hidden')) {
      $destinatario.val('');
    }
  }

  function exportPDF() {
    if (!currentData || currentData.length === 0) {
      notify('warning', 'Sin datos', 'No hay datos para exportar');
      return;
    }

    const formData = $('#frmFiltros').serialize();
    const url = joinUrl('api/reportes/embarque_cumplimiento_pdf.php') + '?' + formData;
    window.open(url, '_blank');
  }

  function exportCSV() {
    if (!currentData || currentData.length === 0) {
      notify('warning', 'Sin datos', 'No hay datos para exportar');
      return;
    }

    // Simple CSV export
    let csvContent = "data:text/csv;charset=utf-8,";
    
    // Headers
    const headers = [
      'Código', 'Fecha', 'Estado', 'Móvil', 'Destinos', 'Pedidos Esp.', 'Pedidos Desp.',
      'Destinos Entr.', 'Items UV Esp.', 'Items UV Desp.', 'Items UC Esp.', 'Items UC Desp.',
      'Rate Cumpl.', 'Delay (días)', 'Status'
    ];
    csvContent += headers.join(',') + '\r\n';
    
    // Data rows
    currentData.forEach(row => {
      const esperados = parseInt(row.pedidos_esperados) || 0;
      const despachados = parseInt(row.pedidos_despachados) || 0;
      const entregados = parseInt(row.destinos_entregados) || 0;
      
      let rate = 0;
      if (esperados > 0) {
        rate = Math.round((Math.min(despachados, entregados) / esperados) * 100);
      }
      
      const values = [
        row.codigo || '',
        row.fecha_embarque || '',
        row.estado_nombre || '',
        row.movil || '',
        row.total_destinos || '',
        row.pedidos_esperados || '',
        row.pedidos_despachados || '',
        row.destinos_entregados || '',
        row.items_esperados_uv || '',
        row.items_despachados_uv || '',
        row.items_esperados_uc || '',
        row.items_despachados_uc || '',
        rate + '%',
        row.delay_days || '',
        row.delivery_status || ''
      ];
      csvContent += values.join(',') + '\r\n';
    });
    
    // Download
    const encodedUri = encodeURI(csvContent);
    const link = document.createElement('a');
    link.setAttribute('href', encodedUri);
    link.setAttribute('download', `cumplimiento_entregas_${new Date().toISOString().split('T')[0]}.csv`);
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  }

  function renderDestinatarioPerformance(destinatarios) {
    const $card = $('#destinatariosCard');
    const $summary = $('#destinatariosCardSummary');
    const $tbody = $('#tablaDestinatarios tbody');

    if (!Array.isArray(destinatarios) || destinatarios.length === 0) {
      $summary.text('');
      $tbody.html('<tr class="text-muted"><td colspan="7" class="text-center py-4">Sin datos disponibles</td></tr>');
      $card.hide();
      return;
    }

    const rowsHtml = destinatarios.map((item) => {
      const visitas = item.total_visitas || 0;
      const completas = item.entregas_completas || 0;
      const parciales = item.entregas_parciales || 0;
      const pendientes = item.entregas_pendientes || 0;
      const rate = visitas > 0 ? (completas / visitas) * 100 : 0;

      return `
        <tr>
          <td>${item.destinatario || '-'}</td>
          <td>${item.cliente || '-'}</td>
          <td class="text-end">${formatNumber(visitas)}</td>
          <td class="text-end text-success">${formatNumber(completas)}</td>
          <td class="text-end text-warning">${formatNumber(parciales)}</td>
          <td class="text-end text-muted">${formatNumber(pendientes)}</td>
          <td class="text-end fw-semibold">${formatPercent(rate)}</td>
        </tr>
      `;
    }).join('');

    const totVisitas = destinatarios.reduce((acc, cur) => acc + (cur.total_visitas || 0), 0);
    $summary.text(`Top ${destinatarios.length} · ${formatNumber(totVisitas)} visitas`);
    $tbody.html(rowsHtml);
    $card.show();
  }

  function renderPeriodRates(periodos) {
    const $card = $('#periodRatesCard');
    const $summary = $('#periodRatesSummary');
    const $tbody = $('#tablaPeriodos tbody');

    if (!Array.isArray(periodos) || periodos.length === 0) {
      $summary.text('');
      $tbody.html('<tr class="text-muted"><td colspan="5" class="text-center py-4">Sin datos disponibles</td></tr>');
      $card.hide();
      return;
    }

    const sorted = [...periodos].sort((a, b) => a.fecha.localeCompare(b.fecha));
    const rowsHtml = sorted.map((item) => {
      const rateDesp = item.rate_despacho ?? 0;
      const rateEntrega = item.rate_entrega ?? 0;
      const pedidosTexto = `${formatNumber(item.pedidos_despachados ?? 0)} / ${formatNumber(item.pedidos_esperados ?? 0)}`;

      return `
        <tr>
          <td>${formatDate(item.fecha)}</td>
          <td class="text-end">${formatNumber(item.embarques ?? 0)}</td>
          <td class="text-end">${pedidosTexto}</td>
          <td class="text-end">${formatPercent(rateDesp)}</td>
          <td class="text-end">${formatPercent(rateEntrega)}</td>
        </tr>
      `;
    }).join('');

    const totalEmbarques = sorted.reduce((acc, cur) => acc + (cur.embarques || 0), 0);
    $summary.text(`${formatNumber(sorted.length)} períodos · ${formatNumber(totalEmbarques)} embarques`);
    $tbody.html(rowsHtml);
    $card.show();
  }

})(window, jQuery);