/**
 * SOL · JavaScript · Reporte de Devoluciones y Retornos
 * Ruta: public/page-scripts/pages/reportes/embarques_devoluciones.js
 */

(function(window, $) {
  'use strict';

  if (!$ || typeof $.ajax !== 'function') {
    console.error('[Reporte Devoluciones] jQuery es requerido.');
    return;
  }

  // ---------------------------------------------
  // Utilidades
  // ---------------------------------------------
  const baseUrl = document.querySelector('meta[name="base-url"]')?.getAttribute('content') ||
                  (typeof window.BASE_URL !== 'undefined' ? window.BASE_URL : '/');

  function joinUrl(path) {
    return baseUrl.replace(/\/$/, '') + '/' + String(path || '').replace(/^\//, '');
  }

  const numberFormatter = new Intl.NumberFormat('es-AR');
  const decimalFormatter = new Intl.NumberFormat('es-AR', { minimumFractionDigits: 1, maximumFractionDigits: 1 });
  const dateFormatter = new Intl.DateTimeFormat('es-AR');
  const dateTimeFormatter = new Intl.DateTimeFormat('es-AR', {
    dateStyle: 'short',
    timeStyle: 'short'
  });

  function formatNumber(value) {
    const numeric = Number(value);
    return numberFormatter.format(Number.isFinite(numeric) ? numeric : 0);
  }

  function formatDecimal(value, digits = 1) {
    const formatter = digits === 1 ? decimalFormatter : new Intl.NumberFormat('es-AR', {
      minimumFractionDigits: digits,
      maximumFractionDigits: digits
    });
    const numeric = Number(value);
    return formatter.format(Number.isFinite(numeric) ? numeric : 0);
  }

  function toDate(value) {
    if (!value) return null;
    const iso = typeof value === 'string' ? value.replace(' ', 'T') : value;
    const date = new Date(iso);
    return Number.isNaN(date.getTime()) ? null : date;
  }

  function formatDate(value) {
    const date = toDate(value);
    return date ? dateFormatter.format(date) : '-';
  }

  function formatDateTime(value) {
    const date = toDate(value);
    return date ? dateTimeFormatter.format(date) : '-';
  }

  function escapeHtml(value) {
    return String(value ?? '')
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;')
      .replace(/'/g, '&#39;');
  }

  function notify(type, title, text) {
    if (typeof Swal !== 'undefined') {
      Swal.fire({ icon: type, title, text, timer: 3200, showConfirmButton: false });
      return;
    }
    const message = `[${title}] ${text}`;
    if (type === 'error') {
      console.error(message);
    } else if (type === 'warning') {
      console.warn(message);
    } else {
      console.info(message);
    }
  }

  function average(total, count) {
    if (!count) return 0;
    const numericTotal = Number(total);
    return Number.isFinite(numericTotal) ? numericTotal / count : 0;
  }

  // ---------------------------------------------
  // Estado de la página
  // ---------------------------------------------
  let dataTable = null;
  let currentData = [];
  let currentSummary = null;
  let currentAggregates = {};
  let filtersLoaded = false;
  let filterOptions = {};

  $(document).ready(function() {
    initFormDefaults();
    initDataTable();
    bindEvents();
    loadData();
  });

  function initFormDefaults() {
    const today = new Date();
    const start = new Date(today.getTime() - (29 * 24 * 60 * 60 * 1000));
    $('#fecha_hasta').val(today.toISOString().slice(0, 10));
    $('#fecha_desde').val(start.toISOString().slice(0, 10));
  }

  function bindEvents() {
    $('#frmDevoluciones').on('submit', function(evt) {
      evt.preventDefault();
      loadData();
    });

    $('#btnLimpiar').on('click', function() {
      const form = document.getElementById('frmDevoluciones');
      if (form) {
        form.reset();
      }
      initFormDefaults();
      filterDestinatarios();
      loadData();
    });

    $('#btnExportCSV').on('click', function() {
      exportCsv();
    });

    $('#cliente_id').on('change', function() {
      filterDestinatarios();
    });
  }

  function initDataTable() {
    if (!$('#tblDevoluciones').length) {
      console.error('[Reporte Devoluciones] Tabla #tblDevoluciones no encontrada');
      return;
    }

    dataTable = $('#tblDevoluciones').DataTable({
      processing: true,
      serverSide: false,
      responsive: true,
      language: {
        url: 'https://cdn.datatables.net/plug-ins/1.13.7/i18n/es-ES.json'
      },
      order: [[0, 'desc']],
      pageLength: 25,
      columns: [
        {
          data: 'retorno_at',
          title: 'Retorno',
          render: function(data, type) {
            if (type === 'display' || type === 'filter') {
              return formatDateTime(data);
            }
            return data || '';
          }
        },
        { data: 'embarque_codigo', title: 'Embarque', defaultContent: '-' },
        { data: 'deposito_nombre', title: 'Depósito', defaultContent: '-' },
        { data: 'cliente_nombre', title: 'Cliente', defaultContent: '-' },
        { data: 'destinatario_label', title: 'Destinatario', defaultContent: '-' },
        { data: 'motivo_nombre', title: 'Motivo', defaultContent: '-' },
        { data: 'producto_label', title: 'Producto', defaultContent: '-' },
        {
          data: 'qty_pallets',
          title: 'Pallets',
          className: 'text-end',
          render: function(data) { return formatNumber(data); }
        },
        {
          data: 'qty_uv',
          title: 'UV',
          className: 'text-end',
          render: function(data) { return formatNumber(data); }
        },
        {
          data: 'qty_uc',
          title: 'Unidades',
          className: 'text-end',
          render: function(data) { return formatNumber(data); }
        },
        { data: 'doc_referencia', title: 'Documento', defaultContent: '-' },
        { data: 'lote_codigo', title: 'Lote', defaultContent: '-' },
        { data: 'responsable_nombre', title: 'Responsable', defaultContent: '-' },
        {
          data: 'retorno_observacion',
          title: 'Observaciones',
          render: function(data, type) {
            if (type === 'display') {
              if (!data || data === '-') {
                return '<span class="text-muted">-</span>';
              }
              return escapeHtml(data);
            }
            return data || '';
          }
        }
      ]
    });
  }

  function loadData() {
    const formData = $('#frmDevoluciones').serialize();

    $.ajax({
      url: joinUrl('api/reportes/embarques_devoluciones.php'),
      method: 'GET',
      data: formData,
      dataType: 'json',
      beforeSend: function() {
        $('#btnExportCSV').prop('disabled', true);
      }
    })
      .done(function(response) {
        if (!response || !response.ok) {
          const errorMsg = response && response.error ? response.error : 'No fue posible obtener los datos.';
          notify('error', 'Error', errorMsg);
          clearData();
          return;
        }

        currentData = Array.isArray(response.data) ? response.data : [];
        currentSummary = response.summary || null;
        currentAggregates = response.aggregates || {};

        if (!filtersLoaded && response.filter_options) {
          filterOptions = response.filter_options;
          applyFilterOptions(filterOptions);
          filtersLoaded = true;
        }

        refreshTable();
        updateSummaryCard();
        renderAggregates();

        $('#btnExportCSV').prop('disabled', currentData.length === 0);
      })
      .fail(function(xhr) {
        console.error('[Reporte Devoluciones] Error Ajax', xhr);
        notify('error', 'Error', 'No se pudo cargar el reporte.');
        clearData();
      })
      .always(function() {
        $('#btnExportCSV').prop('disabled', currentData.length === 0);
      });
  }

  function clearData() {
    currentData = [];
    currentSummary = null;
    currentAggregates = {};
    refreshTable();
    updateSummaryCard();
    renderAggregates();
  }

  function refreshTable() {
    if (!dataTable) return;
    dataTable.clear();
    if (currentData.length) {
      dataTable.rows.add(currentData);
    }
    dataTable.draw();
  }

  function applyFilterOptions(options) {
    if (!options) return;

    populateSelect($('#deposito_id'), options.depositos, function(item) {
      const code = item.code ? ' (' + item.code + ')' : '';
      return item.nombre + code;
    });

    populateSelect($('#motivo_id'), options.motivos, function(item) {
      return item.nombre + (item.code ? ' [' + item.code + ']' : '');
    });

    populateSelect($('#cliente_id'), options.clientes, function(item) {
      return item.nombre;
    });

    const $dest = $('#destinatario_id');
    $dest.find('option:not(:first)').remove();
    (options.destinatarios || []).forEach(function(dest) {
      const option = $('<option></option>')
        .val(dest.id)
        .text(dest.label)
        .attr('data-cliente', dest.cliente_id ?? '');
      $dest.append(option);
    });

    populateSelect($('#movil_id'), options.moviles, function(item) {
      return item.label;
    });

    filterDestinatarios();
  }

  function populateSelect($select, items, textBuilder) {
    if (!$select || !$select.length) return;
    $select.find('option:not(:first)').remove();
    (items || []).forEach(function(item) {
      const label = typeof textBuilder === 'function' ? textBuilder(item) : item.nombre || item.label || item.id;
      const option = $('<option></option>').val(item.id).text(label);
      $select.append(option);
    });
  }

  function filterDestinatarios() {
    const clienteId = $('#cliente_id').val();
    const $dest = $('#destinatario_id');
    $dest.find('option:not(:first)').each(function() {
      const $opt = $(this);
      const optCliente = $opt.attr('data-cliente');
      if (!clienteId || clienteId === '' || optCliente === clienteId) {
        $opt.show();
      } else {
        $opt.hide();
      }
    });
    const selected = $dest.find('option:selected');
    if (selected.length && selected.is(':hidden')) {
      $dest.val('');
    }
  }

  function updateSummaryCard() {
    if (!currentSummary || !currentSummary.total_registros) {
      $('#summaryCard').hide();
      $('#sumRegistros, #sumEmbarques, #sumUV, #sumUVDesc, #sumUC, #sumUCDesc, #sumPallets, #sumDestinatarios').text('-');
      $('#summaryRange').text('');
      return;
    }

    const totalRegistros = currentSummary.total_registros || 0;
    const totalEmbarques = currentSummary.total_embarques || 0;
    const totalDestinatarios = currentSummary.total_destinatarios || 0;
    const totalClientes = currentSummary.total_clientes || 0;
    const totalMoviles = currentSummary.total_moviles || 0;
    const totalPallets = currentSummary.total_pallets || 0;
    const totalUv = currentSummary.total_uv || 0;
    const totalUc = currentSummary.total_uc || 0;

    const avgUv = average(totalUv, totalRegistros);
    const avgUc = average(totalUc, totalRegistros);

    $('#sumRegistros').text(formatNumber(totalRegistros));
    $('#sumEmbarques').text(
      formatNumber(totalEmbarques) + ' embarques · ' + formatNumber(totalDestinatarios) + ' destinatarios'
    );
    $('#sumUV').text(formatNumber(totalUv));
    $('#sumUVDesc').text('Promedio: ' + formatDecimal(avgUv, 1) + ' por registro');
    $('#sumUC').text(formatNumber(totalUc));
    $('#sumUCDesc').text('Promedio: ' + formatDecimal(avgUc, 1) + ' por registro');
    $('#sumPallets').text(formatNumber(totalPallets));
    $('#sumDestinatarios').text(
      formatNumber(totalClientes) + ' clientes · ' + formatNumber(totalMoviles) + ' móviles'
    );
    $('#summaryRange').text(currentSummary.range_label || '');

    $('#summaryCard').show();
  }

  function renderAggregates() {
    renderMotivos(currentAggregates.motivos || []);
    renderDestinatariosTotals(currentAggregates.destinatarios || []);
    renderDestinatarios(currentAggregates.destinatarios || []);
    renderProductos(currentAggregates.productos || []);
  }

  function renderDestinatariosTotals(list) {
    const $card = $('#cardDestinatariosTotales');
    if (!Array.isArray(list) || list.length === 0) {
      $('#destTotalRows, #destTotalUv, #destTotalUc, #destTotalCost').text('-');
      $card.hide();
      return;
    }

    const totals = list.reduce(function(acc, item) {
      const rowsVal = Number(item.rows || 0);
      const uvVal = Number(item.uv || 0);
      const ucVal = Number(item.uc || 0);
      const costVal = Number(item.cost || 0);
      return {
        rows: acc.rows + (Number.isFinite(rowsVal) ? rowsVal : 0),
        uv: acc.uv + (Number.isFinite(uvVal) ? uvVal : 0),
        uc: acc.uc + (Number.isFinite(ucVal) ? ucVal : 0),
        cost: acc.cost + (Number.isFinite(costVal) ? costVal : 0)
      };
    }, { rows: 0, uv: 0, uc: 0, cost: 0 });

    $('#destTotalRows').text(formatNumber(totals.rows));
    $('#destTotalUv').text(formatNumber(totals.uv));
    $('#destTotalUc').text(formatNumber(totals.uc));
    $('#destTotalCost').text('Gs ' + formatNumber(Math.round(totals.cost)));

    $card.show();
  }

  function renderMotivos(list) {
    const $card = $('#cardMotivos');
    const $summary = $('#motivosSummary');
    const $tbody = $('#tablaMotivos tbody');

    if (!Array.isArray(list) || list.length === 0) {
      $tbody.html('<tr class="text-muted"><td colspan="4" class="text-center py-4">Sin datos disponibles</td></tr>');
      $summary.text('');
      $card.hide();
      return;
    }

    const totalUc = list.reduce(function(acc, item) { return acc + (item.uc || 0); }, 0);

    const rowsHtml = list.map(function(item) {
      const code = item.code ? '<div class="text-muted small">' + escapeHtml(item.code) + '</div>' : '';
      return (
        '<tr>' +
        '<td>' + escapeHtml(item.label) + code + '</td>' +
        '<td class="text-end">' + formatNumber(item.rows || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.uv || 0) + '</td>' +
        '<td class="text-end fw-semibold">' + formatNumber(item.uc || 0) + '</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
    $summary.text('Top ' + list.length + ' · ' + formatNumber(totalUc) + ' unidades');
    $card.show();
  }

  function renderDestinatarios(list) {
    const $card = $('#cardDestinatarios');
    const $summary = $('#destinatariosSummary');
    const $tbody = $('#tablaDestinatarios tbody');

    if (!Array.isArray(list) || list.length === 0) {
      $tbody.html('<tr class="text-muted"><td colspan="5" class="text-center py-4">Sin datos disponibles</td></tr>');
      $summary.text('');
      $card.hide();
      return;
    }

  const totalRows = list.reduce(function(acc, item) { return acc + (item.rows || 0); }, 0);
    const totalCost = list.reduce(function(acc, item) {
      const cost = Number(item.cost || 0);
      return acc + (Number.isFinite(cost) ? cost : 0);
    }, 0);

    const rowsHtml = list.map(function(item) {
      const cliente = item.cliente ? escapeHtml(item.cliente) : '-';
      const cost = Number(item.cost || 0);
      return (
        '<tr>' +
        '<td>' + escapeHtml(item.label) + '</td>' +
        '<td>' + cliente + '</td>' +
        '<td class="text-end">' + formatNumber(item.rows || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.uv || 0) + '</td>' +
  '<td class="text-end fw-semibold">' + formatNumber(item.uc || 0) + '</td>' +
  '<td class="text-end fw-semibold">' + formatNumber(Math.round(Number.isFinite(cost) ? cost : 0)) + '</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
    $summary.text(
      'Top ' + list.length + ' · ' + formatNumber(totalRows) + ' registros · Gs ' +
      formatNumber(Math.round(totalCost))
    );
    $card.show();
  }

  function renderProductos(list) {
    const $card = $('#cardProductos');
    const $summary = $('#productosSummary');
    const $tbody = $('#tablaProductos tbody');

    if (!Array.isArray(list) || list.length === 0) {
      $tbody.html('<tr class="text-muted"><td colspan="4" class="text-center py-4">Sin datos disponibles</td></tr>');
      $summary.text('');
      $card.hide();
      return;
    }

    const totalUc = list.reduce(function(acc, item) { return acc + (item.uc || 0); }, 0);

    const rowsHtml = list.map(function(item) {
      const extra = item.sku ? '<div class="text-muted small">SKU: ' + escapeHtml(item.sku) + '</div>' : '';
      return (
        '<tr>' +
        '<td>' + escapeHtml(item.label) + extra + '</td>' +
        '<td class="text-end">' + formatNumber(item.rows || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.uv || 0) + '</td>' +
        '<td class="text-end fw-semibold">' + formatNumber(item.uc || 0) + '</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
    $summary.text('Top ' + list.length + ' · ' + formatNumber(totalUc) + ' unidades');
    $card.show();
  }

  function exportCsv() {
    if (!Array.isArray(currentData) || currentData.length === 0) {
      notify('warning', 'Exportar', 'No hay datos para exportar.');
      return;
    }

    const headers = [
      'Retorno', 'Embarque', 'Deposito', 'Cliente', 'Destinatario', 'Motivo', 'Producto',
      'Pallets', 'UV', 'Unidades', 'Documento', 'Lote', 'Responsable', 'Observaciones'
    ];

    const lines = [headers];

    currentData.forEach(function(row) {
      lines.push([
        formatDateTime(row.retorno_at),
        row.embarque_codigo || '',
        row.deposito_nombre || '',
        row.cliente_nombre || '',
        row.destinatario_label || '',
        row.motivo_nombre || '',
        row.producto_label || '',
        Number(row.qty_pallets || 0),
        Number(row.qty_uv || 0),
        Number(row.qty_uc || 0),
        row.doc_referencia || '',
        row.lote_codigo || '',
        row.responsable_nombre || '',
        row.retorno_observacion && row.retorno_observacion !== '-' ? row.retorno_observacion : ''
      ]);
    });

    const csvContent = lines.map(function(row) {
      return row.map(function(field) {
        const value = field === null || field === undefined ? '' : String(field);
        if (/[,"\n]/.test(value)) {
          return '"' + value.replace(/"/g, '""') + '"';
        }
        return value;
      }).join(',');
    }).join('\r\n');

    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const url = URL.createObjectURL(blob);

    const link = document.createElement('a');
    link.href = url;
    const today = new Date().toISOString().slice(0, 10);
    link.download = 'reporte_devoluciones_' + today + '.csv';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    URL.revokeObjectURL(url);
  }

})(window, jQuery);
