// public/page-scripts/pages/reportes/embarques_monitoreo.js
(function (window, $) {
  "use strict";

  const baseUrl = document.querySelector('meta[name="base-url"]')?.getAttribute("content") ||
    (typeof window.BASE_URL !== "undefined" ? window.BASE_URL : "/");

  function joinUrl(p) {
    return (baseUrl.endsWith('/') ? baseUrl : baseUrl + '/') + p.replace(/^\/+/, '');
  }

  function notify(type, title, text) {
    if (window.Swal && Swal.fire) {
      return Swal.fire({
        icon: type,
        title: title,
        text: text,
        confirmButtonText: 'OK'
      });
    }
    alert((title ? title + ': ' : '') + (text || ''));
  }

  function formatNumber(n) {
    if (n === null || n === undefined || n === '') return '-';
    if (typeof n === 'string') {
      const parsed = parseFloat(n);
      return isNaN(parsed) ? n : parsed.toLocaleString('es-AR');
    }
    return Number(n).toLocaleString('es-AR');
  }

  function formatDateTime(datetime) {
    if (!datetime) return '-';
    try {
      const date = new Date(datetime);
      return date.toLocaleDateString('es-AR') + ' ' + date.toLocaleTimeString('es-AR', { hour: '2-digit', minute: '2-digit' });
    } catch (e) {
      return datetime;
    }
  }

  function formatTime(datetime) {
    if (!datetime) return '-';
    try {
      const date = new Date(datetime);
      return date.toLocaleTimeString('es-AR', { hour: '2-digit', minute: '2-digit' });
    } catch (e) {
      return datetime;
    }
  }

  function formatMinutes(minutes) {
    if (minutes === null || minutes === undefined) return '-';
    const mins = parseInt(minutes);
    if (isNaN(mins)) return '-';
    
    if (mins >= 60) {
      const hours = Math.floor(mins / 60);
      const remainingMins = mins % 60;
      return `${hours}h ${remainingMins}m`;
    }
    return `${mins}m`;
  }

  function showLoading() {
    $('#loadingIndicator').show();
  }

  function hideLoading() {
    $('#loadingIndicator').hide();
  }

  function getEstadoBadgeClass(estadoCode) {
    const classes = {
      'EN_COLA': 'bg-secondary',
      'EN_CARGA': 'bg-warning text-dark',
      'CARGADO': 'bg-primary',
      'EN_RUTA': 'bg-info',
      'ENTREGADO': 'bg-success',
      'FINALIZADO': 'bg-dark',
      'CANCELADO': 'bg-danger'
    };
    return classes[estadoCode] || 'bg-secondary';
  }

  let dataTable = null;
  let currentData = null;

  function initializeFilters() {
    // Set default date range (last 30 days)
    const today = new Date();
    const thirtyDaysAgo = new Date(today);
    thirtyDaysAgo.setDate(today.getDate() - 30);
    
    $('#filtroFechaDesde').val(thirtyDaysAgo.toISOString().split('T')[0]);
    $('#filtroFechaHasta').val(today.toISOString().split('T')[0]);

    // Bind filter change events
    $('#filtroFechaDesde, #filtroFechaHasta, #filtroEstado, #filtroMovil, #filtroChofer').on('change', function() {
      loadData();
    });
  }

  function populateFilterOptions(filters) {
    // Estados
    const $filtroEstado = $('#filtroEstado');
    $filtroEstado.empty().append('<option value="">Todos</option>');
    filters.estados.forEach(estado => {
      $filtroEstado.append(`<option value="${estado.code}">${estado.nombre}</option>`);
    });

    // Móviles
    const $filtroMovil = $('#filtroMovil');
    $filtroMovil.empty().append('<option value="">Todos</option>');
    filters.moviles.forEach(movil => {
      $filtroMovil.append(`<option value="${movil.id}">${movil.label}</option>`);
    });

    // Choferes
    const $filtroChofer = $('#filtroChofer');
    $filtroChofer.empty().append('<option value="">Todos</option>');
    filters.choferes.forEach(chofer => {
      $filtroChofer.append(`<option value="${chofer.id}">${chofer.nombre}</option>`);
    });
  }

  function updateSummary(summary) {
    $('#totalEmbarques').text(formatNumber(summary.total_embarques));
    $('#totalDestinos').text(formatNumber(summary.total_destinos));
    $('#promedioEspera').text(summary.tiempos_promedio.espera_min ? formatNumber(summary.tiempos_promedio.espera_min) : '-');
    $('#promedioCarga').text(summary.tiempos_promedio.carga_min ? formatNumber(summary.tiempos_promedio.carga_min) : '-');
    $('#promedioPlanta').text(summary.tiempos_promedio.planta_min ? formatNumber(summary.tiempos_promedio.planta_min) : '-');
    
    // Find most frequent estado
    let maxCount = 0;
    let mostFrequentEstado = '-';
    for (const [estado, count] of Object.entries(summary.por_estado)) {
      if (count > maxCount) {
        maxCount = count;
        mostFrequentEstado = estado;
      }
    }
    $('#estadoMasFrecuente').text(mostFrequentEstado);

    // Update estados distribution
    updateEstadosDistribution(summary.por_estado);
  }

  function updateEstadosDistribution(porEstado) {
    const $container = $('#estadosDistribution');
    $container.empty();

    for (const [estado, count] of Object.entries(porEstado)) {
      const badgeClass = getEstadoBadgeClass(estado);
      const card = `
        <div class="col-md-2 mb-2">
          <div class="card ${badgeClass} text-white">
            <div class="card-body text-center py-2">
              <h6 class="card-title mb-1">${count}</h6>
              <p class="card-text small mb-0">${estado}</p>
            </div>
          </div>
        </div>
      `;
      $container.append(card);
    }
  }

  function initializeDataTable() {
    if (dataTable) {
      dataTable.destroy();
    }

    // Check if table exists
    const $table = $('#tblEmbarques');
    if ($table.length === 0) {
      console.error('Table #tblEmbarques not found');
      return;
    }

    dataTable = $table.DataTable({
      data: [],
      columns: [
        { 
          data: 'codigo',
          render: function(data, type, row) {
            if (type === 'display') {
              return `<a href="${joinUrl('salidas/embarque/detalle')}?id=${row.id}" class="text-decoration-none">${data}</a>`;
            }
            return data;
          }
        },
        { 
          data: 'fecha_embarque',
          render: function(data) {
            if (!data) return '-';
            return new Date(data).toLocaleDateString('es-AR');
          }
        },
        { 
          data: 'estado_code',
          render: function(data, type, row) {
            const badgeClass = getEstadoBadgeClass(data);
            return `<span class="badge ${badgeClass}">${row.estado}</span>`;
          }
        },
        { data: 'deposito', defaultContent: '-' },
        { data: 'movil', defaultContent: '-' },
        { data: 'chofer', defaultContent: '-' },
        { data: 'total_destinos', className: 'text-end', render: formatNumber },
        { data: 'total_preembarques', className: 'text-end', render: formatNumber },
        { data: 'llegada_at', render: formatTime },
        { data: 'carga_inicio_at', render: formatTime },
        { data: 'carga_fin_at', render: formatTime },
        { data: 'salida_at', render: formatTime },
        { data: 'tiempo_espera_min', className: 'text-end', render: formatMinutes },
        { data: 'tiempo_carga_min', className: 'text-end', render: formatMinutes },
        { data: 'tiempo_total_planta_min', className: 'text-end', render: formatMinutes },
        { data: 'ayudantes_cant', className: 'text-end', render: formatNumber },
        { 
          data: 'temp_salida_c', 
          className: 'text-end',
          render: function(data) {
            return data ? `${formatNumber(data)}°C` : '-';
          }
        },
        { data: 'observacion', defaultContent: '-' }
      ],
      order: [[1, 'desc']], // Order by date descending
      pageLength: 25,
      lengthMenu: [[10, 25, 50, 100], [10, 25, 50, 100]],
      language: {
        url: "https://cdn.datatables.net/plug-ins/1.13.7/i18n/es-ES.json"
      },
      responsive: true,
      scrollX: true,
      processing: true
    });
  }

  function loadData() {
    showLoading();

    const params = {
      fecha_desde: $('#filtroFechaDesde').val(),
      fecha_hasta: $('#filtroFechaHasta').val(),
      estado: $('#filtroEstado').val(),
      movil_id: $('#filtroMovil').val(),
      chofer_id: $('#filtroChofer').val()
    };

    // Remove empty parameters
    Object.keys(params).forEach(key => {
      if (params[key] === '' || params[key] === null) {
        delete params[key];
      }
    });

    $.getJSON(joinUrl('api/reportes/embarque_monitoreo.php'), params)
      .done(function(response) {
        if (response.ok) {
          currentData = response.data;
          
          // Update filters if first load
          if (response.filters) {
            populateFilterOptions(response.filters);
          }
          
          // Update summary
          updateSummary(response.summary);
          
          // Update DataTable
          if (dataTable) {
            dataTable.clear().rows.add(response.data).draw();
          }
        } else {
          notify('error', 'Error', response.error || 'Error al cargar los datos');
        }
      })
      .fail(function(xhr) {
        let errorMsg = 'Error al cargar los datos';
        if (xhr.responseJSON && xhr.responseJSON.error) {
          errorMsg = xhr.responseJSON.error;
        } else if (xhr.statusText) {
          errorMsg = xhr.statusText;
        }
        notify('error', 'Error', errorMsg);
      })
      .always(function() {
        hideLoading();
      });
  }

  function exportPDF() {
    if (!currentData || currentData.length === 0) {
      notify('warning', 'Sin datos', 'No hay datos para exportar');
      return;
    }

    const params = new URLSearchParams({
      fecha_desde: $('#filtroFechaDesde').val() || '',
      fecha_hasta: $('#filtroFechaHasta').val() || '',
      estado: $('#filtroEstado').val() || '',
      movil_id: $('#filtroMovil').val() || '',
      chofer_id: $('#filtroChofer').val() || ''
    });

    window.open(joinUrl(`api/reportes/embarque_monitoreo_pdf.php?${params.toString()}`), '_blank');
  }

  function exportExcel() {
    if (!currentData || currentData.length === 0) {
      notify('warning', 'Sin datos', 'No hay datos para exportar');
      return;
    }

    // Simple CSV export since DataTables buttons might not be available
    let csvContent = "data:text/csv;charset=utf-8,";
    
    // Headers
    const headers = [
      'Código', 'Fecha', 'Estado', 'Depósito', 'Móvil', 'Chofer', 'Destinos', 'PREs',
      'Llegada', 'Inicio Carga', 'Fin Carga', 'Salida', 'Espera (min)', 'Carga (min)', 
      'Total Planta (min)', 'Ayudantes', 'Temp °C', 'Observación'
    ];
    csvContent += headers.join(',') + '\r\n';
    
    // Data rows
    currentData.forEach(row => {
      const values = [
        row.codigo || '',
        row.fecha_embarque || '',
        row.estado || '',
        row.deposito || '',
        row.movil || '',
        row.chofer || '',
        row.total_destinos || '',
        row.total_preembarques || '',
        formatTime(row.llegada_at),
        formatTime(row.carga_inicio_at),
        formatTime(row.carga_fin_at),
        formatTime(row.salida_at),
        row.tiempo_espera_min || '',
        row.tiempo_carga_min || '',
        row.tiempo_total_planta_min || '',
        row.ayudantes_cant || '',
        row.temp_salida_c || '',
        (row.observacion || '').replace(/,/g, ';') // Replace commas to avoid CSV issues
      ];
      csvContent += values.join(',') + '\r\n';
    });
    
    // Download
    const encodedUri = encodeURI(csvContent);
    const link = document.createElement('a');
    link.setAttribute('href', encodedUri);
    link.setAttribute('download', `embarques_monitoreo_${new Date().toISOString().split('T')[0]}.csv`);
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  }

  // Initialize everything
  $(document).ready(function() {
    console.log('Initializing embarques monitoreo...');
    
    // Check if required elements exist
    if ($('#tblEmbarques').length === 0) {
      console.error('Table #tblEmbarques not found in DOM');
      return;
    }
    
    initializeFilters();
    
    // Initialize DataTable with error handling
    try {
      initializeDataTable();
      loadData();
    } catch (error) {
      console.error('Error initializing DataTable:', error);
      notify('error', 'Error', 'Error al inicializar la tabla de datos');
    }

    // Bind export buttons
    $('#btnExportPDF').on('click', exportPDF);
    $('#btnExportExcel').on('click', exportExcel);
    $('#btnActualizar').on('click', loadData);
  });

})(window, jQuery);