// public/page-scripts/pages/reportes/embarques_tiempos.js
(function (window, $) {
  "use strict";

  const baseUrl = document.querySelector('meta[name="base-url"]')?.getAttribute("content") ||
    (typeof window.BASE_URL !== "undefined" ? window.BASE_URL : "/");

  function joinUrl(p) {
    return (baseUrl.endsWith('/') ? baseUrl : baseUrl + '/') + p.replace(/^\/+/, '');
  }

  function notify(type, title, text) {
    if (window.Swal && Swal.fire) {
      return Swal.fire({
        icon: type,
        title: title,
        text: text,
        confirmButtonText: 'OK'
      });
    }
    alert((title ? title + ': ' : '') + (text || ''));
  }

  function formatNumber(n) {
    if (n === null || n === undefined || n === '') return '-';
    if (typeof n === 'string') {
      const parsed = parseFloat(n);
      return isNaN(parsed) ? n : parsed.toLocaleString('es-AR');
    }
    return Number(n).toLocaleString('es-AR');
  }

  function formatTime(datetime) {
    if (!datetime) return '-';
    try {
      const date = new Date(datetime);
      return date.toLocaleTimeString('es-AR', { hour: '2-digit', minute: '2-digit' });
    } catch (e) {
      return datetime;
    }
  }

  function formatMinutes(minutes) {
    if (minutes === null || minutes === undefined) return '-';
    const mins = parseInt(minutes);
    if (isNaN(mins)) return '-';
    
    if (mins >= 60) {
      const hours = Math.floor(mins / 60);
      const remainingMins = mins % 60;
      return `${hours}h ${remainingMins}m`;
    }
    return `${mins}m`;
  }

  function formatDecimal(value, decimals = 1) {
    if (value === null || value === undefined) return '-';
    const num = parseFloat(value);
    if (isNaN(num)) return '-';
    return num.toFixed(decimals);
  }

  function showLoading() {
    $('#loadingIndicator').show();
  }

  function hideLoading() {
    $('#loadingIndicator').hide();
  }

  function getEstadoBadgeClass(estadoCode) {
    const classes = {
      'EN_COLA': 'bg-secondary',
      'EN_CARGA': 'bg-warning text-dark',
      'CARGADO': 'bg-primary',
      'EN_RUTA': 'bg-info',
      'ENTREGADO': 'bg-success',
      'FINALIZADO': 'bg-dark',
      'CANCELADO': 'bg-danger'
    };
    return classes[estadoCode] || 'bg-secondary';
  }

  let dataTable = null;
  let currentData = null;
  let currentStats = null;

  function initializeFilters() {
    // Set default date range (last 30 days)
    const today = new Date();
    const thirtyDaysAgo = new Date(today);
    thirtyDaysAgo.setDate(today.getDate() - 30);
    
    $('#filtroFechaDesde').val(thirtyDaysAgo.toISOString().split('T')[0]);
    $('#filtroFechaHasta').val(today.toISOString().split('T')[0]);

    // Bind filter change events
    $('#filtroFechaDesde, #filtroFechaHasta, #filtroEstado, #filtroMovil, #filtroDeposito').on('change', function() {
      loadData();
    });
  }

  function populateFilterOptions(filters) {
    // Estados
    const $filtroEstado = $('#filtroEstado');
    $filtroEstado.empty().append('<option value="">Todos</option>');
    filters.estados.forEach(estado => {
      $filtroEstado.append(`<option value="${estado.code}">${estado.nombre}</option>`);
    });

    // Móviles
    const $filtroMovil = $('#filtroMovil');
    $filtroMovil.empty().append('<option value="">Todos</option>');
    filters.moviles.forEach(movil => {
      $filtroMovil.append(`<option value="${movil.id}">${movil.label}</option>`);
    });

    // Depósitos
    const $filtroDeposito = $('#filtroDeposito');
    $filtroDeposito.empty().append('<option value="">Todos</option>');
    filters.depositos.forEach(deposito => {
      $filtroDeposito.append(`<option value="${deposito.id}">${deposito.label}</option>`);
    });
  }

  function updateStatistics(stats) {
    // Tiempo de espera
    $('#esperaPromedio').text(formatMinutes(stats.tiempo_espera.promedio));
    $('#esperaMinimo').text(formatMinutes(stats.tiempo_espera.minimo));
    $('#esperaMaximo').text(formatMinutes(stats.tiempo_espera.maximo));

    // Tiempo de carga
    $('#cargaPromedio').text(formatMinutes(stats.tiempo_carga.promedio));
    $('#cargaMinimo').text(formatMinutes(stats.tiempo_carga.minimo));
    $('#cargaMaximo').text(formatMinutes(stats.tiempo_carga.maximo));

    // Tiempo total en planta
    $('#plantaPromedio').text(formatMinutes(stats.tiempo_planta.promedio));
    $('#plantaMinimo').text(formatMinutes(stats.tiempo_planta.minimo));
    $('#plantaMaximo').text(formatMinutes(stats.tiempo_planta.maximo));

    // Tiempo de ruta
    $('#rutaPromedio').text(formatMinutes(stats.tiempo_ruta.promedio));
    $('#rutaMinimo').text(formatMinutes(stats.tiempo_ruta.minimo));
    $('#rutaMaximo').text(formatMinutes(stats.tiempo_ruta.maximo));

    // Eficiencia
    $('#totalEmbarques').text(formatNumber(stats.total_embarques));
    $('#embarquesConTiempos').text(formatNumber(stats.embarques_con_tiempos));
    
    // Eficiencia carga por destino
    $('#eficienciaPromedio').text(formatDecimal(stats.eficiencia_carga.promedio));
    $('#eficienciaMinimo').text(formatDecimal(stats.eficiencia_carga.minimo));
    $('#eficienciaMaximo').text(formatDecimal(stats.eficiencia_carga.maximo));
  }

  function updateAnalysisByMovil(analisis) {
    const $tbody = $('#tblMoviles tbody');
    $tbody.empty();

    for (const [movil, data] of Object.entries(analisis)) {
      const row = `
        <tr>
          <td>${movil}</td>
          <td class="text-end">${formatNumber(data.total_embarques)}</td>
          <td class="text-end">${formatDecimal(data.tiempo_promedio_carga)}</td>
          <td class="text-end">${formatDecimal(data.tiempo_promedio_planta)}</td>
        </tr>
      `;
      $tbody.append(row);
    }
  }

  function updateAnalysisByDeposito(analisis) {
    const $tbody = $('#tblDepositos tbody');
    $tbody.empty();

    for (const [deposito, data] of Object.entries(analisis)) {
      const row = `
        <tr>
          <td>${deposito}</td>
          <td class="text-end">${formatNumber(data.total_embarques)}</td>
          <td class="text-end">${formatDecimal(data.tiempo_promedio_espera)}</td>
          <td class="text-end">${formatDecimal(data.tiempo_promedio_carga)}</td>
        </tr>
      `;
      $tbody.append(row);
    }
  }

  function updateAyudantesDistribution(distribucion) {
    const $tbody = $('#tblAyudantes tbody');
    $tbody.empty();

    const total = Object.values(distribucion).reduce((sum, count) => sum + count, 0);

    for (const [ayudantes, count] of Object.entries(distribucion)) {
      const percentage = total > 0 ? ((count / total) * 100).toFixed(1) : 0;
      const row = `
        <tr>
          <td class="text-center">${ayudantes}</td>
          <td class="text-end">${formatNumber(count)}</td>
          <td class="text-end">${percentage}%</td>
        </tr>
      `;
      $tbody.append(row);
    }
  }

  function initializeDataTable() {
    if (dataTable) {
      dataTable.destroy();
    }

    // Check if table exists
    const $table = $('#tblEmbarques');
    if ($table.length === 0) {
      console.error('Table #tblEmbarques not found');
      return;
    }

    dataTable = $table.DataTable({
      data: [],
      columns: [
        { 
          data: 'codigo',
          render: function(data, type, row) {
            if (type === 'display') {
              return `<a href="${joinUrl('salidas/embarque/detalle')}?id=${row.id}" class="text-decoration-none">${data}</a>`;
            }
            return data;
          }
        },
        { 
          data: 'fecha_embarque',
          render: function(data) {
            if (!data) return '-';
            return new Date(data).toLocaleDateString('es-AR');
          }
        },
        { 
          data: 'estado_code',
          render: function(data, type, row) {
            const badgeClass = getEstadoBadgeClass(data);
            return `<span class="badge ${badgeClass}">${row.estado}</span>`;
          }
        },
        { data: 'movil', defaultContent: '-' },
        { data: 'deposito_code', defaultContent: '-' },
        { data: 'total_destinos', className: 'text-end', render: formatNumber },
        { data: 'ayudantes_cant', className: 'text-end', render: formatNumber },
        { data: 'llegada_at', render: formatTime },
        { data: 'carga_inicio_at', render: formatTime },
        { data: 'carga_fin_at', render: formatTime },
        { data: 'salida_at', render: formatTime },
        { data: 'tiempo_espera_min', className: 'text-end', render: formatMinutes },
        { data: 'tiempo_carga_min', className: 'text-end', render: formatMinutes },
        { data: 'tiempo_total_planta_min', className: 'text-end', render: formatMinutes },
        { data: 'tiempo_ruta_total_min', className: 'text-end', render: formatMinutes },
        { 
          data: 'tiempo_carga_por_destino_min', 
          className: 'text-end',
          render: function(data) {
            return data ? formatDecimal(data) + ' min' : '-';
          }
        }
      ],
      order: [[1, 'desc']], // Order by date descending
      pageLength: 25,
      lengthMenu: [[10, 25, 50, 100], [10, 25, 50, 100]],
      language: {
        url: "https://cdn.datatables.net/plug-ins/1.13.7/i18n/es-ES.json"
      },
      responsive: true,
      scrollX: true,
      processing: true
    });
  }

  function loadData() {
    showLoading();

    const params = {
      fecha_desde: $('#filtroFechaDesde').val(),
      fecha_hasta: $('#filtroFechaHasta').val(),
      estado: $('#filtroEstado').val(),
      movil_id: $('#filtroMovil').val(),
      deposito_id: $('#filtroDeposito').val()
    };

    // Remove empty parameters
    Object.keys(params).forEach(key => {
      if (params[key] === '' || params[key] === null) {
        delete params[key];
      }
    });

    $.getJSON(joinUrl('api/reportes/embarque_tiempos.php'), params)
      .done(function(response) {
        if (response.ok) {
          currentData = response.data;
          currentStats = response.statistics;
          
          // Update filters if first load
          if (response.filters) {
            populateFilterOptions(response.filters);
          }
          
          // Update statistics
          updateStatistics(response.statistics);
          
          // Update analysis tables
          updateAnalysisByMovil(response.statistics.analisis_por_movil);
          updateAnalysisByDeposito(response.statistics.analisis_por_deposito);
          updateAyudantesDistribution(response.statistics.distribucion_ayudantes);
          
          // Update main DataTable
          if (dataTable) {
            dataTable.clear().rows.add(response.data).draw();
          }
        } else {
          notify('error', 'Error', response.error || 'Error al cargar los datos');
        }
      })
      .fail(function(xhr) {
        let errorMsg = 'Error al cargar los datos';
        if (xhr.responseJSON && xhr.responseJSON.error) {
          errorMsg = xhr.responseJSON.error;
        } else if (xhr.statusText) {
          errorMsg = xhr.statusText;
        }
        notify('error', 'Error', errorMsg);
      })
      .always(function() {
        hideLoading();
      });
  }

  function exportPDF() {
    if (!currentData || currentData.length === 0) {
      notify('warning', 'Sin datos', 'No hay datos para exportar');
      return;
    }

    const params = new URLSearchParams({
      fecha_desde: $('#filtroFechaDesde').val() || '',
      fecha_hasta: $('#filtroFechaHasta').val() || '',
      estado: $('#filtroEstado').val() || '',
      movil_id: $('#filtroMovil').val() || '',
      deposito_id: $('#filtroDeposito').val() || ''
    });

    window.open(joinUrl(`api/reportes/embarque_tiempos_pdf.php?${params.toString()}`), '_blank');
  }

  function exportExcel() {
    if (!currentData || currentData.length === 0) {
      notify('warning', 'Sin datos', 'No hay datos para exportar');
      return;
    }

    // Simple CSV export
    let csvContent = "data:text/csv;charset=utf-8,";
    
    // Headers
    const headers = [
      'Código', 'Fecha', 'Estado', 'Móvil', 'Depósito', 'Destinos', 'Ayudantes',
      'Llegada', 'Inicio Carga', 'Fin Carga', 'Salida', 'T. Espera', 'T. Carga', 
      'T. Planta', 'T. Ruta', 'Eficiencia'
    ];
    csvContent += headers.join(',') + '\r\n';
    
    // Data rows
    currentData.forEach(row => {
      const values = [
        row.codigo || '',
        row.fecha_embarque || '',
        row.estado || '',
        row.movil || '',
        row.deposito_code || '',
        row.total_destinos || '',
        row.ayudantes_cant || '',
        formatTime(row.llegada_at),
        formatTime(row.carga_inicio_at),
        formatTime(row.carga_fin_at),
        formatTime(row.salida_at),
        row.tiempo_espera_min || '',
        row.tiempo_carga_min || '',
        row.tiempo_total_planta_min || '',
        row.tiempo_ruta_total_min || '',
        row.tiempo_carga_por_destino_min || ''
      ];
      csvContent += values.join(',') + '\r\n';
    });
    
    // Download
    const encodedUri = encodeURI(csvContent);
    const link = document.createElement('a');
    link.setAttribute('href', encodedUri);
    link.setAttribute('download', `embarques_tiempos_${new Date().toISOString().split('T')[0]}.csv`);
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  }

  // Initialize everything
  $(document).ready(function() {
    initializeFilters();
    initializeDataTable();
    loadData();

    // Bind export buttons
    $('#btnExportPDF').on('click', exportPDF);
    $('#btnExportExcel').on('click', exportExcel);
    $('#btnActualizar').on('click', loadData);
  });

})(window, jQuery);