/**
 * SOL · JavaScript · Reporte de Errores de Picking
 */

(function(window, $) {
  'use strict';

  if (!$ || typeof $.ajax !== 'function') {
    console.error('[Reporte Errores Picking] jQuery es requerido.');
    return;
  }

  const baseUrl = document.querySelector('meta[name="base-url"]')?.getAttribute('content') ||
                  (typeof window.BASE_URL !== 'undefined' ? window.BASE_URL : '/');

  function joinUrl(path) {
    return baseUrl.replace(/\/$/, '') + '/' + String(path || '').replace(/^\//, '');
  }

  const numberFormatter = new Intl.NumberFormat('es-AR');

  function formatNumber(value) {
    const numeric = Number(value);
    return numberFormatter.format(Number.isFinite(numeric) ? numeric : 0);
  }

  function escapeHtml(value) {
    return String(value)
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;')
      .replace(/'/g, '&#039;');
  }

  function formatDateTime(row) {
    const fecha = row && row.logged_fecha ? String(row.logged_fecha) : '';
    const hora = row && row.logged_hora ? String(row.logged_hora) : '';
    if (!fecha) {
      return '-';
    }
    return hora ? fecha + ' ' + hora : fecha;
  }

  function truncateText(text, maxLength) {
    if (!text) {
      return '';
    }
    const value = String(text);
    if (value.length <= maxLength) {
      return value;
    }
    return value.slice(0, maxLength - 1) + '…';
  }

  function notify(type, title, text) {
    if (typeof Swal !== 'undefined') {
      Swal.fire({ icon: type, title, text, timer: 3200, showConfirmButton: false });
      return;
    }
    const message = '[' + title + '] ' + text;
    if (type === 'error') {
      console.error(message);
    } else if (type === 'warning') {
      console.warn(message);
    } else {
      console.info(message);
    }
  }

  let dataTable = null;
  let currentData = [];
  let currentSummary = null;
  let currentAggregates = {};
  let currentLimit = 0;
  let currentTruncated = false;
  let combosLoaded = false;
  let defaultMotivoId = null;

  $(document).ready(function() {
    setDefaultDateRange();
    initDataTable();
    bindEvents();
    loadCombos();
  });

  function setDefaultDateRange() {
    const today = new Date();
    const start = new Date(today.getTime() - (13 * 24 * 60 * 60 * 1000));
    const hasta = today.toISOString().slice(0, 10);
    const desde = start.toISOString().slice(0, 10);
    $('#fecha_desde').val(desde);
    $('#fecha_hasta').val(hasta);
  }

  function applyDefaultMotivo(force) {
    if (!defaultMotivoId) {
      return;
    }
    const $motivo = $('#motivo_id');
    if (!$motivo.length) {
      return;
    }
    const current = $motivo.val();
    if (force || !current) {
      $motivo.val(String(defaultMotivoId));
    }
  }

  function resetFormToDefaults() {
    const form = document.getElementById('frmErroresPicking');
    if (form) {
      form.reset();
    }
    setDefaultDateRange();
    applyDefaultMotivo(true);
  }

  function bindEvents() {
    $('#frmErroresPicking').on('submit', function(evt) {
      evt.preventDefault();
      loadData();
    });

    $('#btnLimpiar').on('click', function() {
      resetFormToDefaults();
      loadData();
    });
  }

  function loadCombos() {
    $.ajax({
      url: joinUrl('api/reportes/errores_picking.php'),
      method: 'GET',
      data: { meta: 'filters' },
      dataType: 'json'
    })
      .done(function(response) {
        if (!response || !response.ok) {
          notify('error', 'Combos', 'No fue posible cargar los filtros disponibles.');
          return;
        }
        applyCombos(response);
        defaultMotivoId = response.default_motivo_id ? String(response.default_motivo_id) : null;
        applyDefaultMotivo(true);
        combosLoaded = true;
        loadData();
      })
      .fail(function(xhr) {
        console.error('[Reporte Errores Picking] Error al cargar combos', xhr);
        notify('error', 'Combos', 'No se pudieron cargar los filtros.');
      });
  }

  function applyCombos(data) {
    const motivos = Array.isArray(data.motivos) ? data.motivos : [];
    const clientes = Array.isArray(data.clientes) ? data.clientes : [];

    const $motivo = $('#motivo_id');
    $motivo.find('option:not(:first)').remove();
    motivos.forEach(function(motivo) {
      $('<option></option>').val(motivo.id).text(motivo.nombre).appendTo($motivo);
    });

    const $cliente = $('#cliente_id');
    $cliente.find('option:not(:first)').remove();
    clientes.forEach(function(cli) {
      $('<option></option>').val(cli.id).text(cli.nombre).appendTo($cliente);
    });
  }

  function initDataTable() {
    if (!$('#tblErrores').length) {
      console.error('[Reporte Errores Picking] Tabla no encontrada');
      return;
    }

    dataTable = $('#tblErrores').DataTable({
      processing: true,
      serverSide: false,
      responsive: true,
      language: {
        url: 'https://cdn.datatables.net/plug-ins/1.13.7/i18n/es-ES.json'
      },
      order: [[0, 'desc']],
      pageLength: 25,
      columns: [
        {
          data: null,
          title: 'Fecha',
          render: function(data, type, row) {
            if (type === 'display' || type === 'filter') {
              return escapeHtml(formatDateTime(row));
            }
            return row && row.logged_at ? row.logged_at : '';
          }
        },
        {
          data: 'pedido_codigo',
          title: 'Pedido',
          render: function(data, type, row) {
            if (!row) {
              return '-';
            }
            const codigo = row.pedido_codigo || '-';
            const pedidoId = row.so_id ? ' #' + row.so_id : '';
            const label = escapeHtml(codigo);
            if (type === 'display') {
              return '<span class="fw-semibold">' + label + '</span><div class="text-muted small">' + escapeHtml(pedidoId) + '</div>';
            }
            return codigo;
          }
        },
        {
          data: 'pre_codigo',
          title: 'PRE',
          render: function(value) {
            return value ? escapeHtml(value) : '—';
          }
        },
        {
          data: 'cliente_nombre',
          title: 'Cliente',
          render: function(value) {
            return value ? escapeHtml(value) : 'Sin cliente';
          }
        },
        {
          data: 'motivo_nombre',
          title: 'Motivo',
          render: function(value) {
            return value ? escapeHtml(value) : 'Sin motivo';
          }
        },
        {
          data: 'responsable_label',
          title: 'Responsable',
          render: function(value) {
            return value ? escapeHtml(value) : 'Sin responsable';
          }
        },
        {
          data: 'observacion',
          title: 'Observación',
          render: function(value) {
            if (!value) {
              return '<span class="text-muted">Sin observación</span>';
            }
            const full = String(value);
            const preview = truncateText(full, 140);
            return '<span title="' + escapeHtml(full) + '">' + escapeHtml(preview) + '</span>';
          }
        }
      ]
    });
  }

  function loadData() {
    if (!combosLoaded) {
      return;
    }

    const formData = $('#frmErroresPicking').serialize();

    $.ajax({
      url: joinUrl('api/reportes/errores_picking.php'),
      method: 'GET',
      data: formData,
      dataType: 'json'
    })
      .done(function(response) {
        if (!response || !response.ok) {
          const message = response && response.error ? response.error : 'No fue posible obtener los datos.';
          notify('error', 'Reporte', message);
          clearData();
          return;
        }

        currentData = Array.isArray(response.rows) ? response.rows : [];
        currentSummary = response.summary || null;
        currentAggregates = response.aggregates || {};
        currentLimit = response.limit || currentData.length || 0;
        currentTruncated = Boolean(response.truncated);

        refreshTable();
        updateSummaryCard();
        renderMotivos(currentAggregates.motivos || []);
        renderResponsables(currentAggregates.responsables || []);
        renderClientes(currentAggregates.clientes || []);
        updateLimitNotice();
      })
      .fail(function(xhr) {
        console.error('[Reporte Errores Picking] Error Ajax', xhr);
        notify('error', 'Reporte', 'No se pudo cargar el reporte.');
        clearData();
      });
  }

  function clearData() {
    currentData = [];
    currentSummary = null;
    currentAggregates = {};
    currentLimit = 0;
    currentTruncated = false;
    refreshTable();
    updateSummaryCard();
    renderMotivos([]);
    renderResponsables([]);
    renderClientes([]);
    updateLimitNotice();
  }

  function refreshTable() {
    if (!dataTable) {
      return;
    }
    dataTable.clear();
    if (currentData.length) {
      dataTable.rows.add(currentData);
    }
    dataTable.draw();
    $('#badgeTotalErrores').text(formatNumber(currentData.length));
  }

  function updateSummaryCard() {
    if (!currentSummary || !currentSummary.total_errores) {
      $('#summaryCard').hide();
      $('#summaryRange, #summaryMotivo').text('');
      $('#summaryMotivo').hide();
      $('#sumErrores, #sumErroresSub, #sumPedidos, #sumPedidosSub, #sumResponsables, #sumResponsablesSub, #sumClientes, #sumClientesSub').text('-');
      $('#summaryHighlight').hide();
      $('#summaryHighlightText').text('');
      return;
    }

    const totalErrores = Number(currentSummary.total_errores || 0);
    const totalPedidos = Number(currentSummary.total_pedidos || 0);
    const totalResponsables = Number(currentSummary.total_responsables || 0);
    const totalClientes = Number(currentSummary.total_clientes || 0);
    const promedioDiario = Number(currentSummary.promedio_diario || 0);

    const erroresPorResponsable = totalResponsables > 0 ? (totalErrores / totalResponsables) : 0;
    const erroresPorCliente = totalClientes > 0 ? (totalErrores / totalClientes) : 0;

    $('#sumErrores').text(formatNumber(totalErrores));
    $('#sumErroresSub').text('Promedio diario: ' + (promedioDiario ? promedioDiario.toFixed(2) : '0'));

    $('#sumPedidos').text(formatNumber(totalPedidos));
    $('#sumPedidosSub').text('Pedidos únicos en rango');

    $('#sumResponsables').text(formatNumber(totalResponsables));
    $('#sumResponsablesSub').text('Errores por responsable: ' + erroresPorResponsable.toFixed(2));

    $('#sumClientes').text(formatNumber(totalClientes));
    $('#sumClientesSub').text('Errores por cliente: ' + erroresPorCliente.toFixed(2));

    $('#summaryRange').text(currentSummary.range_label || '');

    if (currentSummary.motivo_filter_label) {
      $('#summaryMotivo').text('Motivo filtrado: ' + currentSummary.motivo_filter_label).show();
    } else {
      $('#summaryMotivo').hide().text('');
    }

    if (currentSummary.motivo_top_label) {
      $('#summaryHighlightText').text('Motivo más frecuente: ' + currentSummary.motivo_top_label + ' (' + formatNumber(currentSummary.motivo_top_total || 0) + ' incidentes)');
      $('#summaryHighlight').show();
    } else {
      $('#summaryHighlightText').text('');
      $('#summaryHighlight').hide();
    }

    $('#summaryCard').show();
  }

  function renderMotivos(list) {
    const $card = $('#cardMotivos');
    const $tbody = $('#tablaMotivos tbody');
    const $summary = $('#motivosSummary');

    if (!Array.isArray(list) || list.length === 0) {
      $tbody.html('<tr class="text-muted"><td colspan="4" class="text-center py-4">Sin datos disponibles</td></tr>');
      $summary.text('');
      $card.hide();
      return;
    }

    const totalErrores = list.reduce(function(acc, item) {
      const rows = Number(item.rows || 0);
      return acc + (Number.isFinite(rows) ? rows : 0);
    }, 0);

    const rowsHtml = list.map(function(item) {
      return (
        '<tr>' +
        '<td>' + escapeHtml(item.label || '-') + '</td>' +
        '<td class="text-end">' + formatNumber(item.rows || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.pedidos || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.clientes || 0) + '</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
    $summary.text('Top ' + list.length + ' · ' + formatNumber(totalErrores) + ' incidentes');
    $card.show();
  }

  function renderResponsables(list) {
    const $card = $('#cardResponsables');
    const $tbody = $('#tablaResponsables tbody');
    const $summary = $('#responsablesSummary');

    if (!Array.isArray(list) || list.length === 0) {
      $tbody.html('<tr class="text-muted"><td colspan="3" class="text-center py-4">Sin datos disponibles</td></tr>');
      $summary.text('');
      $card.hide();
      return;
    }

    const totalErrores = list.reduce(function(acc, item) {
      const rows = Number(item.rows || 0);
      return acc + (Number.isFinite(rows) ? rows : 0);
    }, 0);

    const rowsHtml = list.map(function(item) {
      return (
        '<tr>' +
        '<td>' + escapeHtml(item.label || '-') + '</td>' +
        '<td class="text-end">' + formatNumber(item.rows || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.pedidos || 0) + '</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
    $summary.text('Top ' + list.length + ' · ' + formatNumber(totalErrores) + ' incidentes');
    $card.show();
  }

  function renderClientes(list) {
    const $card = $('#cardClientes');
    const $tbody = $('#tablaClientes tbody');
    const $summary = $('#clientesSummary');

    if (!Array.isArray(list) || list.length === 0) {
      $tbody.html('<tr class="text-muted"><td colspan="3" class="text-center py-4">Sin datos disponibles</td></tr>');
      $summary.text('');
      $card.hide();
      return;
    }

    const totalErrores = list.reduce(function(acc, item) {
      const rows = Number(item.rows || 0);
      return acc + (Number.isFinite(rows) ? rows : 0);
    }, 0);

    const rowsHtml = list.map(function(item) {
      return (
        '<tr>' +
        '<td>' + escapeHtml(item.label || '-') + '</td>' +
        '<td class="text-end">' + formatNumber(item.rows || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.pedidos || 0) + '</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
    $summary.text('Top ' + list.length + ' · ' + formatNumber(totalErrores) + ' incidentes');
    $card.show();
  }

  function updateLimitNotice() {
    if (currentTruncated && currentLimit) {
      $('#limitValue').text(formatNumber(currentLimit));
      $('#limitNotice').show();
    } else {
      $('#limitNotice').hide();
    }
  }

})(window, jQuery);
