/**
 * SOL · JavaScript · Reporte Facturación vs Rendiciones
 */

(function(window, $) {
  'use strict';

  if (!$ || typeof $.ajax !== 'function') {
    console.error('[Reporte Facturación] jQuery es requerido.');
    return;
  }

  const baseUrl = document.querySelector('meta[name="base-url"]')?.getAttribute('content') ||
                  (typeof window.BASE_URL !== 'undefined' ? window.BASE_URL : '/');

  function joinUrl(path) {
    return baseUrl.replace(/\/$/, '') + '/' + String(path || '').replace(/^\//, '');
  }

  const numberFormatter = new Intl.NumberFormat('es-AR', { minimumFractionDigits: 0, maximumFractionDigits: 0 });
  const decimalFormatter = new Intl.NumberFormat('es-AR', { minimumFractionDigits: 2, maximumFractionDigits: 2 });
  const currencyFormatter = new Intl.NumberFormat('es-AR', { style: 'currency', currency: 'ARS', minimumFractionDigits: 2 });

  function escapeHtml(value) {
    return String(value)
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;')
      .replace(/'/g, '&#39;');
  }

  function formatInt(value) {
    const numeric = Number(value);
    return Number.isFinite(numeric) ? numberFormatter.format(numeric) : '0';
  }

  function formatDecimal(value, fallback) {
    const numeric = Number(value);
    if (!Number.isFinite(numeric)) {
      return fallback !== undefined ? fallback : '—';
    }
    return decimalFormatter.format(numeric);
  }

  function formatCurrency(value, fallback) {
    const numeric = Number(value);
    if (!Number.isFinite(numeric)) {
      return fallback !== undefined ? fallback : '—';
    }
    return currencyFormatter.format(numeric);
  }

  function formatDate(value) {
    if (!value) {
      return '—';
    }
    const str = String(value).slice(0, 10);
    const parts = str.split('-');
    if (parts.length !== 3) {
      return escapeHtml(str);
    }
    return parts[2] + '/' + parts[1] + '/' + parts[0];
  }

  function formatDateTime(value) {
    if (!value) {
      return '—';
    }
    const cleaned = String(value).replace('T', ' ');
    const datePart = cleaned.slice(0, 10);
    const timePart = cleaned.length >= 16 ? cleaned.slice(11, 19) : '';
    return formatDate(datePart) + (timePart ? ' ' + timePart : '');
  }

  function buildBadge(label, type) {
    const cls = type === 'success' ? 'bg-success' : type === 'warning' ? 'bg-warning text-dark' : 'bg-secondary';
    return '<span class="badge ' + cls + ' me-1">' + escapeHtml(label) + '</span>';
  }

  function notify(type, title, text) {
    if (typeof Swal !== 'undefined') {
      Swal.fire({ icon: type, title, text, timer: 3200, showConfirmButton: false });
      return;
    }
    const msg = '[' + title + '] ' + text;
    if (type === 'error') {
      console.error(msg);
    } else if (type === 'warning') {
      console.warn(msg);
    } else {
      console.info(msg);
    }
  }

  let dataTable = null;
  let combosReady = false;
  let currentRows = [];
  let currentSummary = null;
  let currentAggregates = {};
  let currentLimit = 0;
  let currentTruncated = false;

  $(document).ready(function() {
    setDefaultDateRange();
    initDataTable();
    bindEvents();
    loadCombos();
  });

  function setDefaultDateRange() {
    const today = new Date();
    const past = new Date(today.getTime() - (29 * 24 * 60 * 60 * 1000));
    document.getElementById('fecha_hasta').value = today.toISOString().slice(0, 10);
    document.getElementById('fecha_desde').value = past.toISOString().slice(0, 10);
  }

  function resetFilters() {
    const form = document.getElementById('frmFacturacion');
    if (form) {
      form.reset();
    }
    setDefaultDateRange();
  }

  function bindEvents() {
    $('#frmFacturacion').on('submit', function(evt) {
      evt.preventDefault();
      loadData();
    });

    $('#btnLimpiar').on('click', function() {
      resetFilters();
      loadData();
    });

    $('#solo_pendientes').on('change', function() {
      if (this.checked) {
        $('#solo_rendidas').prop('checked', false);
      }
    });

    $('#solo_rendidas').on('change', function() {
      if (this.checked) {
        $('#solo_pendientes').prop('checked', false);
      }
    });
  }

  function loadCombos() {
    console.debug('[Reporte Facturación] baseUrl ->', baseUrl, 'request ->', joinUrl('api/reportes/facturacion.php?meta=filters'));
    $.ajax({
      url: joinUrl('api/reportes/facturacion.php'),
      method: 'GET',
      data: { meta: 'filters' },
      dataType: 'json'
    })
      .done(function(response) {
        if (!response || !response.ok) {
          console.error('[Reporte Facturación] combos response inválido', response);
          notify('error', 'Filtros', 'No fue posible obtener los combos del reporte.');
          return;
        }
        applyCombos(response);
        combosReady = true;
        loadData();
      })
      .fail(function(xhr) {
        const status = xhr?.status || 0;
        const text = xhr?.responseText ? xhr.responseText.slice(0, 500) : '(sin cuerpo)';
        console.error('[Reporte Facturación] Error combos', { status, text, xhr });
        const message = status === 401 ? 'Sesión expirada o no autorizada. Refresque la página e intente nuevamente.' : 'Se produjo un error al cargar los combos.';
        notify('error', 'Filtros', message);
      });
  }

  function applyCombos(data) {
    const depositos = Array.isArray(data.depositos) ? data.depositos : [];
    const choferes = Array.isArray(data.choferes) ? data.choferes : [];
    const clientes = Array.isArray(data.clientes) ? data.clientes : [];
    const condiciones = Array.isArray(data.condiciones) ? data.condiciones : [];

    const $dep = $('#deposito_id');
    $dep.find('option:not(:first)').remove();
    depositos.forEach(function(item) {
      $('<option></option>').val(item.id).text(item.label).appendTo($dep);
    });

    const $cho = $('#chofer_id');
    $cho.find('option:not(:first)').remove();
    choferes.forEach(function(item) {
      $('<option></option>').val(item.id).text(item.label).appendTo($cho);
    });

    const $cli = $('#cliente_id');
    $cli.find('option:not(:first)').remove();
    clientes.forEach(function(item) {
      $('<option></option>').val(item.id).text(item.label).appendTo($cli);
    });

    const $cond = $('#condicion');
    $cond.find('option:not(:first)').remove();
    condiciones.forEach(function(label) {
      $('<option></option>').val(label).text(label).appendTo($cond);
    });
  }

  function initDataTable() {
    if (!$('#tblFacturacion').length) {
      console.error('[Reporte Facturación] Tabla principal no encontrada.');
      return;
    }

    dataTable = $('#tblFacturacion').DataTable({
      processing: true,
      serverSide: false,
      responsive: true,
      language: {
        url: 'https://cdn.datatables.net/plug-ins/1.13.7/i18n/es-ES.json'
      },
      order: [[1, 'desc']],
      pageLength: 25,
      columns: [
        {
          data: null,
          title: 'Factura',
          render: function(row) {
            if (!row) {
              return '—';
            }
            const numero = row.factura ? '<span class="fw-semibold">' + escapeHtml(row.factura) + '</span>' : '<span class="text-muted">Sin número</span>';
            const estadoBadge = row.estado === 'RENDIDA'
              ? buildBadge('Rendida', 'success')
              : buildBadge('Pendiente', 'warning');
            const docs = Array.isArray(row.doc_fuentes) && row.doc_fuentes.length
              ? '<div class="small text-muted">Docs: ' + row.doc_fuentes.map(escapeHtml).join(', ') + '</div>'
              : '';
            return estadoBadge + numero + docs;
          }
        },
        {
          data: null,
          title: 'Fechas',
          render: function(row) {
            if (!row) { return '—'; }
            const partes = [];
            if (row.fecha_factura) {
              partes.push('<div><span class="text-muted">Factura:</span> ' + formatDate(row.fecha_factura) + '</div>');
            } else if (row.fecha_documento) {
              partes.push('<div><span class="text-muted">Documento:</span> ' + formatDate(row.fecha_documento) + '</div>');
            }
            if (row.fecha_rendicion) {
              partes.push('<div><span class="text-muted">Rendición:</span> ' + formatDateTime(row.fecha_rendicion) + '</div>');
            }
            if (row.salida_at) {
              partes.push('<div><span class="text-muted">Salida:</span> ' + formatDateTime(row.salida_at) + '</div>');
            }
            if (!partes.length) {
              partes.push('<span class="text-muted">Sin fechas registradas</span>');
            }
            return partes.join('');
          }
        },
        {
          data: null,
          title: 'Importes',
          className: 'text-end',
          render: function(row) {
            if (!row) { return '—'; }
            const monto = row.monto !== null ? formatCurrency(row.monto) : '—';
            const rendido = formatCurrency(row.importe_rendido || 0, '$ 0,00');
            const pendiente = row.importe_pendiente !== null ? formatCurrency(row.importe_pendiente, '$ 0,00') : '—';
            return '<div><span class="text-muted">Registrado:</span> ' + monto + '</div>' +
                   '<div><span class="text-muted">Rendido:</span> ' + rendido + '</div>' +
                   '<div><span class="text-muted">Pendiente:</span> ' + pendiente + '</div>';
          }
        },
        {
          data: null,
          title: 'Clientes / Destinatarios',
          render: function(row) {
            if (!row) { return '—'; }
            const clientes = row.clientes_label ? escapeHtml(row.clientes_label) : 'Sin cliente';
            const dests = row.destinatarios_label ? escapeHtml(row.destinatarios_label) : 'Sin destinatario';
            return '<div><span class="text-muted">Clientes:</span> ' + clientes + '</div>' +
                   '<div><span class="text-muted">Destinatarios:</span> ' + dests + '</div>';
          }
        },
        {
          data: null,
          title: 'Embarque / Depósito',
          render: function(row) {
            if (!row) { return '—'; }
            const emb = row.embarque_codigo ? escapeHtml(row.embarque_codigo) : 'Sin embarque';
            const dep = row.deposito_label ? escapeHtml(row.deposito_label) : 'Sin depósito';
            return '<div><span class="text-muted">Embarque:</span> ' + emb + '</div>' +
                   '<div><span class="text-muted">Depósito:</span> ' + dep + '</div>';
          }
        },
        {
          data: null,
          title: 'Chofer / Móvil',
          render: function(row) {
            if (!row) { return '—'; }
            const cho = row.chofer_nombre ? escapeHtml(row.chofer_nombre) : 'Sin chofer';
            const mov = row.movil_label ? escapeHtml(row.movil_label) : 'Sin móvil';
            return '<div><span class="text-muted">Chofer:</span> ' + cho + '</div>' +
                   '<div><span class="text-muted">Móvil:</span> ' + mov + '</div>';
          }
        },
        {
          data: null,
          title: 'Condición',
          render: function(row) {
            if (!row) { return '—'; }
            return row.condicion ? escapeHtml(row.condicion) : '<span class="text-muted">Sin condición</span>';
          }
        },
        {
          data: null,
          title: 'Días',
          className: 'text-end',
          render: function(row) {
            if (!row) { return '—'; }
            const rend = row.dias_a_rendir !== null ? formatDecimal(row.dias_a_rendir) + ' d' : '—';
            const pend = row.dias_pendientes !== null ? formatDecimal(row.dias_pendientes) + ' d' : '—';
            return '<div><span class="text-muted">Rendir:</span> ' + rend + '</div>' +
                   '<div><span class="text-muted">Pendiente:</span> ' + pend + '</div>';
          }
        }
      ]
    });
  }

  function loadData() {
    if (!combosReady) {
      return;
    }

    const query = $('#frmFacturacion').serialize();

    $.ajax({
      url: joinUrl('api/reportes/facturacion.php'),
      method: 'GET',
      data: query,
      dataType: 'json'
    })
      .done(function(response) {
        if (!response || !response.ok) {
          const msg = response && response.error ? response.error : 'No se pudieron obtener los datos del reporte.';
          notify('error', 'Reporte de facturación', msg);
          clearData();
          return;
        }
        currentRows = Array.isArray(response.rows) ? response.rows : [];
        currentSummary = response.summary || null;
        currentAggregates = response.aggregates || {};
        currentLimit = response.limit || currentRows.length || 0;
        currentTruncated = Boolean(response.truncated);

        refreshTable();
        updateSummaryCard();
        renderClientes(currentAggregates.clientes || []);
        renderDepositos(currentAggregates.depositos || []);
        renderChoferes(currentAggregates.choferes || []);
        renderCondiciones(currentAggregates.condiciones || []);
        updateLimitNotice();
      })
      .fail(function(xhr) {
        console.error('[Reporte Facturación] Error datos', xhr);
        notify('error', 'Reporte de facturación', 'Se produjo un error al cargar los datos.');
        clearData();
      });
  }

  function clearData() {
    currentRows = [];
    currentSummary = null;
    currentAggregates = {};
    currentLimit = 0;
    currentTruncated = false;
    refreshTable();
    updateSummaryCard();
    renderClientes([]);
    renderDepositos([]);
    renderChoferes([]);
    renderCondiciones([]);
    updateLimitNotice();
  }

  function refreshTable() {
    if (!dataTable) {
      return;
    }
    dataTable.clear();
    if (currentRows.length) {
      dataTable.rows.add(currentRows);
    }
    dataTable.draw();
    $('#badgeTotalFacturas').text(formatInt(currentRows.length));
  }

  function updateSummaryCard() {
    if (!currentSummary || !currentSummary.total_facturas) {
      $('#summaryCard').hide();
      $('#summaryRange').text('');
      $('#sumTotalFacturas, #sumTotalRendidas, #sumTotalPendientes, #sumImporteTotal, #sumImporteRendido, #sumImportePendiente, #sumPromedioDias, #sumMaxPendiente, #sumPendientes7, #sumPendientes15').text('-');
      $('#summaryHighlight').hide();
      $('#summaryHighlightText').text('');
      return;
    }

    $('#sumTotalFacturas').text(formatInt(currentSummary.total_facturas));
    $('#sumTotalRendidas').text(formatInt(currentSummary.total_rendidas));
    $('#sumTotalPendientes').text(formatInt(currentSummary.total_pendientes));
    $('#sumImporteTotal').text(formatCurrency(currentSummary.importe_total || 0, '$ 0,00'));
    $('#sumImporteRendido').text('Rendido: ' + formatCurrency(currentSummary.importe_rendido || 0, '$ 0,00'));
    $('#sumImportePendiente').text(formatCurrency(currentSummary.importe_pendiente || 0, '$ 0,00'));

    if (currentSummary.avg_dias_rendicion !== null && currentSummary.avg_dias_rendicion !== undefined) {
      $('#sumPromedioDias').text(formatDecimal(currentSummary.avg_dias_rendicion) + ' días');
    } else {
      $('#sumPromedioDias').text('—');
    }

    if (currentSummary.max_dias_pendiente !== null && currentSummary.max_dias_pendiente !== undefined) {
      $('#sumMaxPendiente').text('Máximo pendiente: ' + formatDecimal(currentSummary.max_dias_pendiente) + ' días');
    } else {
      $('#sumMaxPendiente').text('');
    }

    $('#sumPendientes7').text(formatInt(currentSummary.pendientes_mayor_7 || 0));
    $('#sumPendientes15').text(formatInt(currentSummary.pendientes_mayor_15 || 0));
    $('#summaryRange').text(currentSummary.range_label || '');

    const highlights = [];
    if ((currentSummary.total_pendientes || 0) > 0) {
      highlights.push('Pendientes: ' + formatInt(currentSummary.total_pendientes));
    }
    if ((currentSummary.importe_pendiente || 0) > 0) {
      highlights.push('Monto pendiente: ' + formatCurrency(currentSummary.importe_pendiente));
    }
    if (highlights.length) {
      $('#summaryHighlightText').text(highlights.join(' · '));
      $('#summaryHighlight').show();
    } else {
      $('#summaryHighlightText').text('');
      $('#summaryHighlight').hide();
    }

    $('#summaryCard').show();
  }

  function renderClientes(list) {
    const $card = $('#cardClientes');
    const $tbody = $('#tablaClientes tbody');
    const $summary = $('#clientesSummary');

    if (!Array.isArray(list) || list.length === 0) {
      $tbody.html('<tr class="text-muted"><td colspan="4" class="text-center py-4">Sin datos disponibles</td></tr>');
      $summary.text('');
      $card.hide();
      return;
    }

    const totalFacturas = list.reduce(function(acc, item) { return acc + Number(item.facturas || 0); }, 0);
    const rowsHtml = list.map(function(item) {
      return (
        '<tr>' +
          '<td>' + escapeHtml(item.label || 'Sin cliente') + '</td>' +
          '<td class="text-end">' + formatInt(item.facturas || 0) + '</td>' +
          '<td class="text-end">' + formatInt(item.pendientes || 0) + '</td>' +
          '<td class="text-end">' + formatCurrency(item.monto_total || 0, '$ 0,00') + '</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
    $summary.text('Top ' + list.length + ' · ' + formatInt(totalFacturas) + ' facturas');
    $card.show();
  }

  function renderDepositos(list) {
    const $card = $('#cardDepositos');
    const $tbody = $('#tablaDepositos tbody');
    const $summary = $('#depositosSummary');

    if (!Array.isArray(list) || list.length === 0) {
      $tbody.html('<tr class="text-muted"><td colspan="4" class="text-center py-4">Sin datos disponibles</td></tr>');
      $summary.text('');
      $card.hide();
      return;
    }

    const total = list.reduce(function(acc, item) { return acc + Number(item.facturas || 0); }, 0);
    const rowsHtml = list.map(function(item) {
      return (
        '<tr>' +
          '<td>' + escapeHtml(item.label || 'Sin depósito') + '</td>' +
          '<td class="text-end">' + formatInt(item.facturas || 0) + '</td>' +
          '<td class="text-end">' + formatInt(item.pendientes || 0) + '</td>' +
          '<td class="text-end">' + formatCurrency(item.monto_total || 0, '$ 0,00') + '</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
    $summary.text(formatInt(total) + ' facturas');
    $card.show();
  }

  function renderChoferes(list) {
    const $card = $('#cardChoferes');
    const $tbody = $('#tablaChoferes tbody');
    const $summary = $('#choferesSummary');

    if (!Array.isArray(list) || list.length === 0) {
      $tbody.html('<tr class="text-muted"><td colspan="3" class="text-center py-4">Sin datos disponibles</td></tr>');
      $summary.text('');
      $card.hide();
      return;
    }

    const total = list.reduce(function(acc, item) { return acc + Number(item.facturas || 0); }, 0);
    const rowsHtml = list.map(function(item) {
      return (
        '<tr>' +
          '<td>' + escapeHtml(item.label || 'Sin chofer') + '</td>' +
          '<td class="text-end">' + formatInt(item.facturas || 0) + '</td>' +
          '<td class="text-end">' + formatInt(item.pendientes || 0) + '</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
    $summary.text(formatInt(total) + ' facturas');
    $card.show();
  }

  function renderCondiciones(list) {
    const $card = $('#cardCondiciones');
    const $tbody = $('#tablaCondiciones tbody');
    const $summary = $('#condicionesSummary');

    if (!Array.isArray(list) || list.length === 0) {
      $tbody.html('<tr class="text-muted"><td colspan="3" class="text-center py-4">Sin datos disponibles</td></tr>');
      $summary.text('');
      $card.hide();
      return;
    }

    const total = list.reduce(function(acc, item) { return acc + Number(item.facturas || 0); }, 0);
    const rowsHtml = list.map(function(item) {
      return (
        '<tr>' +
          '<td>' + escapeHtml(item.condicion || 'Sin condición') + '</td>' +
          '<td class="text-end">' + formatInt(item.facturas || 0) + '</td>' +
          '<td class="text-end">' + formatInt(item.pendientes || 0) + '</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
    $summary.text(formatInt(total) + ' facturas');
    $card.show();
  }

  function updateLimitNotice() {
    if (currentTruncated && currentLimit) {
      $('#limitValue').text(formatInt(currentLimit));
      $('#limitNotice').show();
    } else {
      $('#limitNotice').hide();
    }
  }

})(window, jQuery);
