/**
 * SOL · JavaScript · Reporte de Ingresos
 */

(function(window, $) {
  'use strict';

  if (!$ || typeof $.ajax !== 'function') {
    console.error('[Reporte Ingresos] jQuery es requerido.');
    return;
  }

  const baseUrl = document.querySelector('meta[name="base-url"]')?.getAttribute('content') ||
                  (typeof window.BASE_URL !== 'undefined' ? window.BASE_URL : '/');

  function joinUrl(path) {
    return baseUrl.replace(/\/$/, '') + '/' + String(path || '').replace(/^\//, '');
  }

  const numberFormatter = new Intl.NumberFormat('es-AR');
  const dateFormatter = new Intl.DateTimeFormat('es-AR');

  function formatNumber(value) {
    const numeric = Number(value);
    return numberFormatter.format(Number.isFinite(numeric) ? numeric : 0);
  }

  function toDate(value) {
    if (!value) return null;
    const iso = typeof value === 'string' ? value.replace(' ', 'T') : value;
    const date = new Date(iso);
    return Number.isNaN(date.getTime()) ? null : date;
  }

  function formatDate(value) {
    const date = toDate(value);
    return date ? dateFormatter.format(date) : '-';
  }

  function notify(type, title, text) {
    if (typeof Swal !== 'undefined') {
      Swal.fire({ icon: type, title, text, timer: 3200, showConfirmButton: false });
      return;
    }
    const message = '[' + title + '] ' + text;
    if (type === 'error') {
      console.error(message);
    } else if (type === 'warning') {
      console.warn(message);
    } else {
      console.info(message);
    }
  }

  let dataTable = null;
  let combosLoaded = false;
  let currentData = [];
  let currentSummary = null;
  let currentAggregates = {};
  let currentLimit = 0;
  let currentTruncated = false;

  $(document).ready(function() {
    initFormDefaults();
    initDataTable();
    bindEvents();
    loadCombos();
  });

  function initFormDefaults() {
    const today = new Date();
    const start = new Date(today.getTime() - (29 * 24 * 60 * 60 * 1000));
    const inputHasta = document.getElementById('fecha_hasta');
    const inputDesde = document.getElementById('fecha_desde');
    if (inputHasta) inputHasta.value = today.toISOString().slice(0, 10);
    if (inputDesde) inputDesde.value = start.toISOString().slice(0, 10);
  }

  function bindEvents() {
    $('#frmIngresos').on('submit', function(evt) {
      evt.preventDefault();
      loadData();
    });

    $('#btnLimpiar').on('click', function() {
      const form = document.getElementById('frmIngresos');
      if (form) {
        form.reset();
      }
      initFormDefaults();
      loadData();
    });

    $('#btnExportPdf').on('click', function() {
      exportPdf();
    });
  }

  function loadCombos() {
    $.ajax({
      url: joinUrl('api/reportes/ingresos.php'),
      method: 'GET',
      data: { meta: 'filters' },
      dataType: 'json'
    })
      .done(function(response) {
        if (!response || !response.ok) {
          notify('error', 'Combos', 'No fue posible cargar los filtros');
          return;
        }
        applyCombos(response);
        combosLoaded = true;
        loadData();
      })
      .fail(function(xhr) {
        console.error('[Reporte Ingresos] Error combos', xhr);
        notify('error', 'Combos', 'No se pudieron cargar los filtros');
      });
  }

  function applyCombos(data) {
    const clientes = Array.isArray(data.clientes) ? data.clientes : [];
    const depositos = Array.isArray(data.depositos) ? data.depositos : [];
    const estados = Array.isArray(data.estados) ? data.estados : [];
    const moviles = Array.isArray(data.moviles) ? data.moviles : [];

    const $cliente = $('#cliente_id');
    $cliente.find('option:not(:first)').remove();
    clientes.forEach(function(cli) {
      $('<option></option>').val(cli.id).text(cli.nombre).appendTo($cliente);
    });

    const $deposito = $('#deposito_id');
    $deposito.find('option:not(:first)').remove();
    depositos.forEach(function(dep) {
      $('<option></option>').val(dep.id).text(dep.nombre).appendTo($deposito);
    });

    const $estado = $('#estado');
    $estado.find('option:not(:first)').remove();
    estados.forEach(function(est) {
      $('<option></option>').val(est.code).text(est.nombre + ' [' + est.code + ']').appendTo($estado);
    });

    const $movil = $('#movil_id');
    $movil.find('option:not(:first)').remove();
    moviles.forEach(function(mov) {
      $('<option></option>').val(mov.id).text(mov.label).appendTo($movil);
    });
  }

  function initDataTable() {
    if (!$('#tblIngresos').length) {
      console.error('[Reporte Ingresos] Tabla no encontrada');
      return;
    }

    dataTable = $('#tblIngresos').DataTable({
      processing: true,
      serverSide: false,
      responsive: true,
      language: {
        url: 'https://cdn.datatables.net/plug-ins/1.13.7/i18n/es-ES.json'
      },
      order: [[0, 'desc']],
      pageLength: 25,
      columns: [
        {
          data: 'fecha_ingreso',
          title: 'Fecha',
          render: function(data, type, row) {
            if (type === 'display' || type === 'filter') {
              const fecha = formatDate(data);
              const hora = row && row.hora_llegada ? row.hora_llegada : '';
              return hora ? fecha + ' ' + hora : fecha;
            }
            return data || '';
          }
        },
        { data: 'packinglist_codigo', title: 'Packing List', defaultContent: '-' },
        { data: 'cliente_nombre', title: 'Cliente', defaultContent: '-' },
        { data: 'deposito_nombre', title: 'Depósito', defaultContent: '-' },
        {
          data: 'estado_nombre',
          title: 'Estado',
          render: function(data, type, row) {
            if (type === 'display' && row && row.estado_code) {
              return '<span>' + data + '</span><div class="text-muted small">' + row.estado_code + '</div>';
            }
            return data || '-';
          }
        },
        { data: 'doc_referencia', title: 'Documento', defaultContent: '-' },
        { data: 'movil_chapa', title: 'Camión', defaultContent: '-' },
        { data: 'chofer_nombre', title: 'Chofer', defaultContent: '-' },
        {
          data: 'pallets',
          title: 'Pallets',
          className: 'text-end',
          render: function(value) { return formatNumber(value); }
        },
        {
          data: 'uv_cajas',
          title: 'UV',
          className: 'text-end',
          render: function(value) { return formatNumber(value); }
        },
        {
          data: 'uc_unidades',
          title: 'UC',
          className: 'text-end',
          render: function(value) { return formatNumber(value); }
        },
        {
          data: 'items',
          title: 'Ítems',
          className: 'text-end',
          render: function(value) { return formatNumber(value); }
        },
        {
          data: 'operarios_cant',
          title: 'Operarios',
          className: 'text-end',
          render: function(value) { return formatNumber(value); }
        }
      ]
    });
  }

  function loadData() {
    if (!combosLoaded) {
      return;
    }

    const formData = $('#frmIngresos').serialize();

    $.ajax({
      url: joinUrl('api/reportes/ingresos.php'),
      method: 'GET',
      data: formData,
      dataType: 'json',
      beforeSend: function() {
        $('#btnExportPdf').prop('disabled', true);
      }
    })
      .done(function(response) {
        if (!response || !response.ok) {
          const errorMsg = response && response.error ? response.error : 'No fue posible obtener los datos.';
          notify('error', 'Reporte', errorMsg);
          clearData();
          return;
        }

        currentData = Array.isArray(response.rows) ? response.rows : [];
        currentSummary = response.summary || null;
        currentAggregates = response.aggregates || {};
        currentLimit = response.limit || currentData.length || 0;
        currentTruncated = Boolean(response.truncated);

        refreshTable();
        updateSummaryCard();
        renderClientes(currentAggregates.clientes || []);
        renderDepositos(currentAggregates.depositos || []);
        renderEstados(currentAggregates.estados || []);
        updateLimitNotice();

        $('#btnExportPdf').prop('disabled', currentData.length === 0);
      })
      .fail(function(xhr) {
        console.error('[Reporte Ingresos] Error Ajax', xhr);
        notify('error', 'Reporte', 'No se pudo cargar el reporte.');
        clearData();
      })
      .always(function() {
        $('#btnExportPdf').prop('disabled', currentData.length === 0);
      });
  }

  function clearData() {
    currentData = [];
    currentSummary = null;
    currentAggregates = {};
    currentLimit = 0;
    currentTruncated = false;
    refreshTable();
    updateSummaryCard();
    renderClientes([]);
    renderDepositos([]);
    renderEstados([]);
    updateLimitNotice();
  }

  function refreshTable() {
    if (!dataTable) return;
    dataTable.clear();
    if (currentData.length) {
      dataTable.rows.add(currentData);
    }
    dataTable.draw();
    $('#badgeTotalIngresos').text(formatNumber(currentData.length));
  }

  function updateSummaryCard() {
    if (!currentSummary || !currentSummary.total_ingresos) {
      $('#summaryCard').hide();
      $('#sumIngresos, #sumIngresosSub, #sumClientes, #sumClientesSub, #sumPallets, #sumPalletsSub, #sumUc, #sumUcSub').text('-');
      $('#summaryRange').text('');
      return;
    }

    const totalIngresos = currentSummary.total_ingresos || 0;
    const totalUc = currentSummary.total_uc || 0;
    const totalPallets = currentSummary.total_pallets || 0;
    const totalUv = currentSummary.total_uv || 0;
    const avgUc = totalIngresos > 0 ? (totalUc / totalIngresos) : 0;

    $('#sumIngresos').text(formatNumber(totalIngresos));
    $('#sumIngresosSub').text('Depósitos: ' + formatNumber(currentSummary.total_depositos || 0));

    $('#sumClientes').text(formatNumber(currentSummary.total_clientes || 0));
    $('#sumClientesSub').text('Móviles: ' + formatNumber(currentSummary.total_moviles || 0));

    $('#sumPallets').text(formatNumber(totalPallets));
    $('#sumPalletsSub').text('UV totales: ' + formatNumber(totalUv));

    $('#sumUc').text(formatNumber(totalUc));
    $('#sumUcSub').text('Prom. UC/ingreso: ' + formatNumber(Math.round(avgUc)));

    $('#summaryRange').text(currentSummary.range_label || '');
    $('#summaryCard').show();
  }

  function renderClientes(list) {
    const $card = $('#cardClientes');
    const $tbody = $('#tablaClientes tbody');
    const $summary = $('#clientesSummary');

    if (!Array.isArray(list) || list.length === 0) {
      $tbody.html('<tr class="text-muted"><td colspan="5" class="text-center py-4">Sin datos disponibles</td></tr>');
      $summary.text('');
      $card.hide();
      return;
    }

    const totals = list.reduce(function(acc, item) {
      const rows = Number(item.rows || 0);
      const pallets = Number(item.pallets || 0);
      const uv = Number(item.uv || 0);
      const uc = Number(item.uc || 0);
      return {
        rows: acc.rows + (Number.isFinite(rows) ? rows : 0),
        pallets: acc.pallets + (Number.isFinite(pallets) ? pallets : 0),
        uv: acc.uv + (Number.isFinite(uv) ? uv : 0),
        uc: acc.uc + (Number.isFinite(uc) ? uc : 0)
      };
    }, { rows: 0, pallets: 0, uv: 0, uc: 0 });

    const rowsHtml = list.map(function(item) {
      return (
        '<tr>' +
        '<td>' + (item.label || '-') + '</td>' +
        '<td class="text-end">' + formatNumber(item.rows || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.pallets || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.uv || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.uc || 0) + '</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
    $summary.text('Top ' + list.length + ' · ' + formatNumber(totals.rows) + ' ingresos');
    $card.show();
  }

  function renderDepositos(list) {
    const $card = $('#cardDepositos');
    const $tbody = $('#tablaDepositos tbody');
    const $summary = $('#depositosSummary');

    if (!Array.isArray(list) || list.length === 0) {
      $tbody.html('<tr class="text-muted"><td colspan="5" class="text-center py-4">Sin datos disponibles</td></tr>');
      $summary.text('');
      $card.hide();
      return;
    }

    const totals = list.reduce(function(acc, item) {
      const rows = Number(item.rows || 0);
      const pallets = Number(item.pallets || 0);
      const uv = Number(item.uv || 0);
      const uc = Number(item.uc || 0);
      return {
        rows: acc.rows + (Number.isFinite(rows) ? rows : 0),
        pallets: acc.pallets + (Number.isFinite(pallets) ? pallets : 0),
        uv: acc.uv + (Number.isFinite(uv) ? uv : 0),
        uc: acc.uc + (Number.isFinite(uc) ? uc : 0)
      };
    }, { rows: 0, pallets: 0, uv: 0, uc: 0 });

    const rowsHtml = list.map(function(item) {
      return (
        '<tr>' +
        '<td>' + (item.label || '-') + '</td>' +
        '<td class="text-end">' + formatNumber(item.rows || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.pallets || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.uv || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.uc || 0) + '</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
    $summary.text('Top ' + list.length + ' · ' + formatNumber(totals.rows) + ' ingresos');
    $card.show();
  }

  function renderEstados(list) {
    const $card = $('#cardEstados');
    const $tbody = $('#tablaEstados tbody');
    const $summary = $('#estadosSummary');

    if (!Array.isArray(list) || list.length === 0) {
      $tbody.html('<tr class="text-muted"><td colspan="5" class="text-center py-4">Sin datos disponibles</td></tr>');
      $summary.text('');
      $card.hide();
      return;
    }

    const totals = list.reduce(function(acc, item) {
      const rows = Number(item.rows || 0);
      const pallets = Number(item.pallets || 0);
      const uv = Number(item.uv || 0);
      const uc = Number(item.uc || 0);
      return {
        rows: acc.rows + (Number.isFinite(rows) ? rows : 0),
        pallets: acc.pallets + (Number.isFinite(pallets) ? pallets : 0),
        uv: acc.uv + (Number.isFinite(uv) ? uv : 0),
        uc: acc.uc + (Number.isFinite(uc) ? uc : 0)
      };
    }, { rows: 0, pallets: 0, uv: 0, uc: 0 });

    const rowsHtml = list.map(function(item) {
      const code = item.code ? '<div class="text-muted small">' + item.code + '</div>' : '';
      return (
        '<tr>' +
        '<td>' + (item.label || '-') + code + '</td>' +
        '<td class="text-end">' + formatNumber(item.rows || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.pallets || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.uv || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.uc || 0) + '</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
    $summary.text('Total ingresos: ' + formatNumber(totals.rows));
    $card.show();
  }

  function updateLimitNotice() {
    if (currentTruncated && currentLimit) {
      $('#limitValue').text(formatNumber(currentLimit));
      $('#limitNotice').show();
    } else {
      $('#limitNotice').hide();
    }
  }

  function exportPdf() {
    if (!Array.isArray(currentData) || currentData.length === 0) {
      notify('warning', 'Exportar', 'No hay datos para exportar.');
      return;
    }
    const query = $('#frmIngresos').serialize();
    const url = joinUrl('reportes/ingresos/pdf?' + query);
    window.open(url, '_blank');
  }

})(window, jQuery);
