/**
 * SOL · JavaScript · Reporte de Móviles Disponibles
 */

(function(window, $) {
  'use strict';

  if (!$ || typeof $.ajax !== 'function') {
    console.error('[Reporte Móviles] jQuery es requerido.');
    return;
  }

  const baseUrl = document.querySelector('meta[name="base-url"]')?.getAttribute('content') ||
                  (typeof window.BASE_URL !== 'undefined' ? window.BASE_URL : '/');

  function joinUrl(path) {
    return baseUrl.replace(/\/$/, '') + '/' + String(path || '').replace(/^\//, '');
  }

  const numberFormatter = new Intl.NumberFormat('es-AR');

  function formatNumber(value) {
    const numeric = Number(value);
    return numberFormatter.format(Number.isFinite(numeric) ? numeric : 0);
  }

  function notify(type, title, text) {
    if (typeof Swal !== 'undefined') {
      Swal.fire({ icon: type, title, text, timer: 3200, showConfirmButton: false });
      return;
    }
    const message = '[' + title + '] ' + text;
    if (type === 'error') {
      console.error(message);
    } else if (type === 'warning') {
      console.warn(message);
    } else {
      console.info(message);
    }
  }

  let dataTable = null;
  let combosLoaded = false;
  let currentData = [];
  let currentSummary = null;
  let currentAggregates = {};
  let currentLimit = 0;
  let currentTruncated = false;

  $(document).ready(function() {
    initFormDefaults();
    initDataTable();
    bindEvents();
    loadCombos();
  });

  function initFormDefaults() {
    const today = new Date();
    const start = new Date(today.getTime() - (6 * 24 * 60 * 60 * 1000));
    const inputHasta = document.getElementById('fecha_hasta');
    const inputDesde = document.getElementById('fecha_desde');
    if (inputHasta) inputHasta.value = today.toISOString().slice(0, 10);
    if (inputDesde) inputDesde.value = start.toISOString().slice(0, 10);
  }

  function bindEvents() {
    $('#frmMoviles').on('submit', function(evt) {
      evt.preventDefault();
      loadData();
    });

    $('#btnLimpiar').on('click', function() {
      const form = document.getElementById('frmMoviles');
      if (form) {
        form.reset();
      }
      initFormDefaults();
      loadData();
    });

    $('#btnExportPdf').on('click', function() {
      exportPdf();
    });
  }

  function loadCombos() {
    $.ajax({
      url: joinUrl('api/reportes/moviles_disponibles.php'),
      method: 'GET',
      data: { meta: 'filters' },
      dataType: 'json'
    })
      .done(function(response) {
        if (!response || !response.ok) {
          notify('error', 'Combos', 'No fue posible cargar los filtros');
          return;
        }
        applyCombos(response);
        combosLoaded = true;
        loadData();
      })
      .fail(function(xhr) {
        console.error('[Reporte Móviles] Error combos', xhr);
        notify('error', 'Combos', 'No se pudieron cargar los filtros');
      });
  }

  function applyCombos(data) {
    const moviles = Array.isArray(data.moviles) ? data.moviles : [];
    const transportadoras = Array.isArray(data.transportadoras) ? data.transportadoras : [];

    const $movil = $('#movil_id');
    $movil.find('option:not(:first)').remove();
    moviles.forEach(function(item) {
      $('<option></option>').val(item.id).text(item.label).appendTo($movil);
    });

    const $transportadora = $('#transportadora_id');
    $transportadora.find('option:not(:first)').remove();
    transportadoras.forEach(function(item) {
      $('<option></option>').val(item.id).text(item.nombre).appendTo($transportadora);
    });
  }

  function initDataTable() {
    if (!$('#tblMoviles').length) {
      console.error('[Reporte Móviles] Tabla no encontrada');
      return;
    }

    dataTable = $('#tblMoviles').DataTable({
      processing: true,
      serverSide: false,
      responsive: true,
      language: {
        url: 'https://cdn.datatables.net/plug-ins/1.13.7/i18n/es-ES.json'
      },
      order: [[0, 'desc']],
      pageLength: 25,
      columns: [
        { data: 'fecha', title: 'Fecha', defaultContent: '-' },
        { data: 'movil_label', title: 'Móvil', defaultContent: '-' },
        { data: 'transportadora_nombre', title: 'Transportadora', defaultContent: '-' },
        {
          data: 'disponible',
          title: 'Disponible',
          render: function(value) { return value ? 'Sí' : 'No'; }
        },
        {
          data: null,
          title: 'Ventana',
          render: function(_, type, row) {
            const ini = row && row.ventana_ini ? row.ventana_ini : '-';
            const fin = row && row.ventana_fin ? row.ventana_fin : '-';
            return ini + ' · ' + fin;
          }
        },
        { data: 'llegada_time', title: 'Llegada', defaultContent: '-', render: function(v) { return v || '-'; } },
        {
          data: 'no_asistio',
          title: 'No asistió',
          render: function(value) { return value ? 'Sí' : 'No'; }
        },
        {
          data: 'no_utilizado',
          title: 'No utilizado',
          render: function(value) { return value ? 'Sí' : 'No'; }
        },
        {
          data: 'notas',
          title: 'Notas',
          render: function(value) { return value ? String(value) : '-'; }
        }
      ]
    });
  }

  function loadData() {
    if (!combosLoaded) {
      return;
    }

    const formData = $('#frmMoviles').serialize();

    $.ajax({
      url: joinUrl('api/reportes/moviles_disponibles.php'),
      method: 'GET',
      data: formData,
      dataType: 'json',
      beforeSend: function() {
        $('#btnExportPdf').prop('disabled', true);
      }
    })
      .done(function(response) {
        if (!response || !response.ok) {
          const errorMsg = response && response.error ? response.error : 'No fue posible obtener los datos.';
          notify('error', 'Reporte', errorMsg);
          clearData();
          return;
        }

        currentData = Array.isArray(response.rows) ? response.rows : [];
        currentSummary = response.summary || null;
        currentAggregates = response.aggregates || {};
        currentLimit = response.limit || currentData.length || 0;
        currentTruncated = Boolean(response.truncated);

        refreshTable();
        updateSummaryCard();
        renderTransportadoras(currentAggregates.transportadoras || []);
        renderMoviles(currentAggregates.moviles || []);
        renderFechas(currentAggregates.fechas || []);
        updateLimitNotice();

        $('#btnExportPdf').prop('disabled', currentData.length === 0);
      })
      .fail(function(xhr) {
        console.error('[Reporte Móviles] Error Ajax', xhr);
        notify('error', 'Reporte', 'No se pudo cargar el reporte.');
        clearData();
      })
      .always(function() {
        $('#btnExportPdf').prop('disabled', currentData.length === 0);
      });
  }

  function clearData() {
    currentData = [];
    currentSummary = null;
    currentAggregates = {};
    currentLimit = 0;
    currentTruncated = false;
    refreshTable();
    updateSummaryCard();
    renderTransportadoras([]);
    renderMoviles([]);
    renderFechas([]);
    updateLimitNotice();
  }

  function refreshTable() {
    if (!dataTable) return;
    dataTable.clear();
    if (currentData.length) {
      dataTable.rows.add(currentData);
    }
    dataTable.draw();
    $('#badgeTotal').text(formatNumber(currentData.length));
  }

  function updateSummaryCard() {
    if (!currentSummary || !currentSummary.total_registros) {
      $('#summaryCard').hide();
      $('#sumRegistros, #sumRegistrosSub, #sumDisponibles, #sumDisponiblesSub, #sumNoAsistio, #sumNoAsistioSub, #sumNoUtilizado, #sumNoUtilizadoSub').text('-');
      $('#summaryRange').text('');
      return;
    }

    const total = currentSummary.total_registros || 0;
    const disponibles = currentSummary.total_disponibles || 0;
    const noDisponibles = currentSummary.total_no_disponibles || 0;
    const noAsistio = currentSummary.total_no_asistio || 0;
    const noUtilizado = currentSummary.total_no_utilizado || 0;

    $('#sumRegistros').text(formatNumber(total));
    $('#sumRegistrosSub').text('Móviles: ' + formatNumber(currentSummary.total_moviles || 0) + ' · Transportadoras: ' + formatNumber(currentSummary.total_transportadoras || 0));

    $('#sumDisponibles').text(formatNumber(disponibles));
    $('#sumDisponiblesSub').text('No disponibles: ' + formatNumber(noDisponibles));

    $('#sumNoAsistio').text(formatNumber(noAsistio));
    $('#sumNoAsistioSub').text('Prom. por día: ' + formatAverage(noAsistio, total));

    $('#sumNoUtilizado').text(formatNumber(noUtilizado));
    $('#sumNoUtilizadoSub').text('Prom. por día: ' + formatAverage(noUtilizado, total));

    $('#summaryRange').text(currentSummary.range_label || '');
    $('#summaryCard').show();
  }

  function formatAverage(value, total) {
    if (!total) return '0';
    return numberFormatter.format(Math.round((value / total) * 10) / 10);
  }

  function renderTransportadoras(list) {
    const $card = $('#cardTransportadoras');
    const $tbody = $('#tablaTransportadoras tbody');
    const $summary = $('#transportadorasSummary');

    if (!Array.isArray(list) || list.length === 0) {
      $tbody.html('<tr class="text-muted"><td colspan="4" class="text-center py-4">Sin datos disponibles</td></tr>');
      $summary.text('');
      $card.hide();
      return;
    }

    const totals = list.reduce(function(acc, item) {
      const rows = Number(item.rows || 0);
      const disponibles = Number(item.disponibles || 0);
      const noAsistio = Number(item.no_asistio || 0);
      return {
        rows: acc.rows + (Number.isFinite(rows) ? rows : 0),
        disponibles: acc.disponibles + (Number.isFinite(disponibles) ? disponibles : 0),
        noAsistio: acc.noAsistio + (Number.isFinite(noAsistio) ? noAsistio : 0)
      };
    }, { rows: 0, disponibles: 0, noAsistio: 0 });

    const rowsHtml = list.map(function(item) {
      return (
        '<tr>' +
        '<td>' + (item.label || '-') + '</td>' +
        '<td class="text-end">' + formatNumber(item.rows || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.disponibles || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.no_asistio || 0) + '</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
    $summary.text('Top ' + list.length + ' · ' + formatNumber(totals.rows) + ' registros');
    $card.show();
  }

  function renderMoviles(list) {
    const $card = $('#cardMoviles');
    const $tbody = $('#tablaMoviles tbody');
    const $summary = $('#movilesSummary');

    if (!Array.isArray(list) || list.length === 0) {
      $tbody.html('<tr class="text-muted"><td colspan="4" class="text-center py-4">Sin datos disponibles</td></tr>');
      $summary.text('');
      $card.hide();
      return;
    }

    const totals = list.reduce(function(acc, item) {
      const rows = Number(item.rows || 0);
      const disponibles = Number(item.disponibles || 0);
      const noUtilizado = Number(item.no_utilizado || 0);
      return {
        rows: acc.rows + (Number.isFinite(rows) ? rows : 0),
        disponibles: acc.disponibles + (Number.isFinite(disponibles) ? disponibles : 0),
        noUtilizado: acc.noUtilizado + (Number.isFinite(noUtilizado) ? noUtilizado : 0)
      };
    }, { rows: 0, disponibles: 0, noUtilizado: 0 });

    const rowsHtml = list.map(function(item) {
      return (
        '<tr>' +
        '<td>' + (item.label || '-') + '</td>' +
        '<td class="text-end">' + formatNumber(item.rows || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.disponibles || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.no_utilizado || 0) + '</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
    $summary.text('Top ' + list.length + ' · ' + formatNumber(totals.rows) + ' registros');
    $card.show();
  }

  function renderFechas(list) {
    const $card = $('#cardFechas');
    const $tbody = $('#tablaFechas tbody');
    const $summary = $('#fechasSummary');

    if (!Array.isArray(list) || list.length === 0) {
      $tbody.html('<tr class="text-muted"><td colspan="4" class="text-center py-4">Sin datos disponibles</td></tr>');
      $summary.text('');
      $card.hide();
      return;
    }

    const totals = list.reduce(function(acc, item) {
      const rows = Number(item.rows || 0);
      const disponibles = Number(item.disponibles || 0);
      const noDisponibles = Number(item.no_disponibles || 0);
      return {
        rows: acc.rows + (Number.isFinite(rows) ? rows : 0),
        disponibles: acc.disponibles + (Number.isFinite(disponibles) ? disponibles : 0),
        noDisponibles: acc.noDisponibles + (Number.isFinite(noDisponibles) ? noDisponibles : 0)
      };
    }, { rows: 0, disponibles: 0, noDisponibles: 0 });

    const rowsHtml = list.map(function(item) {
      return (
        '<tr>' +
        '<td>' + (item.fecha || '-') + '</td>' +
        '<td class="text-end">' + formatNumber(item.rows || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.disponibles || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.no_disponibles || 0) + '</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
    $summary.text(formatNumber(totals.rows) + ' registros');
    $card.show();
  }

  function updateLimitNotice() {
    if (!currentTruncated || !currentLimit) {
      $('#limitNotice').hide();
      return;
    }
    $('#limitValue').text(formatNumber(currentLimit));
    $('#limitNotice').show();
  }

  function exportPdf() {
    if (!currentData.length) {
      notify('warning', 'Exportación', 'No hay datos para exportar.');
      return;
    }

    const params = $('#frmMoviles').serializeArray();
    params.push({ name: 'limit', value: currentLimit || 1500 });
    const query = $.param(params);
    const url = joinUrl('reportes/moviles_disponibles/pdf') + '?' + query;
    window.open(url, '_blank');
  }

})(window, window.jQuery);
