/**
 * SOL · JavaScript · Reporte de Pedidos
 */

(function(window, $) {
  'use strict';

  if (!$ || typeof $.ajax !== 'function') {
    console.error('[Reporte Pedidos] jQuery es requerido.');
    return;
  }

  const baseUrl = document.querySelector('meta[name="base-url"]')?.getAttribute('content') ||
                  (typeof window.BASE_URL !== 'undefined' ? window.BASE_URL : '/');

  function joinUrl(path) {
    return baseUrl.replace(/\/$/, '') + '/' + String(path || '').replace(/^\//, '');
  }

  const numberFormatter = new Intl.NumberFormat('es-AR');
  const percentFormatter = new Intl.NumberFormat('es-AR', { minimumFractionDigits: 1, maximumFractionDigits: 1 });
  const dateFormatter = new Intl.DateTimeFormat('es-AR');

  function formatNumber(value) {
    const numeric = Number(value);
    return numberFormatter.format(Number.isFinite(numeric) ? numeric : 0);
  }

  function formatPercent(value) {
    if (value === null || value === undefined) {
      return '-';
    }
    const numeric = Number(value);
    if (!Number.isFinite(numeric)) {
      return '-';
    }
    return percentFormatter.format(numeric) + '%';
  }

  function toDate(value) {
    if (!value) return null;
    const iso = typeof value === 'string' ? value.replace(' ', 'T') : value;
    const date = new Date(iso);
    return Number.isNaN(date.getTime()) ? null : date;
  }

  function formatDate(value) {
    const date = toDate(value);
    return date ? dateFormatter.format(date) : '-';
  }

  function notify(type, title, text) {
    if (typeof Swal !== 'undefined') {
      Swal.fire({ icon: type, title, text, timer: 3200, showConfirmButton: false });
      return;
    }
    const message = '[' + title + '] ' + text;
    if (type === 'error') {
      console.error(message);
    } else if (type === 'warning') {
      console.warn(message);
    } else {
      console.info(message);
    }
  }

  let dataTable = null;
  let currentData = [];
  let currentSummary = null;
  let currentAggregates = {};
  let currentLimit = 0;
  let currentTruncated = false;
  let combosLoaded = false;

  $(document).ready(function() {
    initFormDefaults();
    initDataTable();
    bindEvents();
    loadCombos();
  });

  function initFormDefaults() {
    const today = new Date();
    const start = new Date(today.getTime() - (29 * 24 * 60 * 60 * 1000));
    document.getElementById('fecha_hasta').value = today.toISOString().slice(0, 10);
    document.getElementById('fecha_desde').value = start.toISOString().slice(0, 10);
  }

  function bindEvents() {
    $('#frmPedidos').on('submit', function(evt) {
      evt.preventDefault();
      loadData();
    });

    $('#btnLimpiar').on('click', function() {
      const form = document.getElementById('frmPedidos');
      if (form) {
        form.reset();
      }
      initFormDefaults();
      loadData();
    });

    $('#btnExportPdf').on('click', function() {
      exportPdf();
    });
  }

  function loadCombos() {
    $.ajax({
      url: joinUrl('api/reportes/pedidos.php'),
      method: 'GET',
      data: { meta: 'filters' },
      dataType: 'json'
    })
      .done(function(response) {
        if (!response || !response.ok) {
          notify('error', 'Combos', 'No fue posible cargar los combos de filtros.');
          return;
        }
        applyCombos(response);
        combosLoaded = true;
        loadData();
      })
      .fail(function(xhr) {
        console.error('[Reporte Pedidos] Error al cargar combos', xhr);
        notify('error', 'Combos', 'No se pudieron cargar los filtros');
      });
  }

  function applyCombos(data) {
    const clientes = Array.isArray(data.clientes) ? data.clientes : [];
    const estados = Array.isArray(data.estados) ? data.estados : [];

    const $cliente = $('#cliente_id');
    $cliente.find('option:not(:first)').remove();
    clientes.forEach(function(cli) {
      $('<option></option>').val(cli.id).text(cli.nombre).appendTo($cliente);
    });

    const $estado = $('#estado_id');
    $estado.find('option:not(:first)').remove();
    estados.forEach(function(est) {
      const label = est.code ? est.nombre + ' [' + est.code + ']' : est.nombre;
      $('<option></option>').val(est.id).text(label).appendTo($estado);
    });
  }

  function initDataTable() {
    if (!$('#tblPedidos').length) {
      console.error('[Reporte Pedidos] Tabla no encontrada');
      return;
    }

    dataTable = $('#tblPedidos').DataTable({
      processing: true,
      serverSide: false,
      responsive: true,
      language: {
        url: 'https://cdn.datatables.net/plug-ins/1.13.7/i18n/es-ES.json'
      },
      order: [[0, 'desc']],
      pageLength: 25,
      columns: [
        {
          data: 'fecha_pedido',
          title: 'Fecha',
          render: function(data, type, row) {
            if (type === 'display' || type === 'filter') {
              const fecha = formatDate(data);
              const hora = row && row.hora_pedido ? row.hora_pedido : '';
              return hora ? fecha + ' ' + hora : fecha;
            }
            return data || '';
          }
        },
        { data: 'pedido_codigo', title: 'Código', defaultContent: '-' },
        { data: 'cliente_nombre', title: 'Cliente', defaultContent: '-' },
        {
          data: 'estado_nombre',
          title: 'Estado',
          render: function(data, type, row) {
            if (type === 'display' && row && row.estado_code) {
              return '<span>' + data + '</span><div class="text-muted small">' + row.estado_code + '</div>';
            }
            return data || '-';
          }
        },
        {
          data: 'destinos',
          title: 'Destinos',
          className: 'text-end',
          render: function(value) { return formatNumber(value); }
        },
        {
          data: 'destinatarios',
          title: 'Destinatarios',
          className: 'text-end',
          render: function(value) { return formatNumber(value); }
        },
        {
          data: 'items',
          title: 'Ítems',
          className: 'text-end',
          render: function(value) { return formatNumber(value); }
        },
        {
          data: 'expected_uv',
          title: 'UV (Plan)',
          className: 'text-end',
          render: function(value) { return formatNumber(value); }
        },
        {
          data: 'shipped_uv',
          title: 'UV (Desp)',
          className: 'text-end',
          render: function(value) { return formatNumber(value); }
        },
        {
          data: 'expected_uc',
          title: 'UC (Plan)',
          className: 'text-end',
          render: function(value) { return formatNumber(value); }
        },
        {
          data: 'shipped_uc',
          title: 'UC (Desp)',
          className: 'text-end',
          render: function(value) { return formatNumber(value); }
        },
        {
          data: 'avance_uc_pct',
          title: 'Avance UC',
          className: 'text-end',
          render: function(value) { return formatPercent(value); }
        }
      ]
    });
  }

  function loadData() {
    if (!combosLoaded) {
      return;
    }

    const formData = $('#frmPedidos').serialize();

    $.ajax({
      url: joinUrl('api/reportes/pedidos.php'),
      method: 'GET',
      data: formData,
      dataType: 'json',
      beforeSend: function() {
        $('#btnExportPdf').prop('disabled', true);
      }
    })
      .done(function(response) {
        if (!response || !response.ok) {
          const errorMsg = response && response.error ? response.error : 'No fue posible obtener los datos.';
          notify('error', 'Reporte', errorMsg);
          clearData();
          return;
        }

        currentData = Array.isArray(response.rows) ? response.rows : [];
        currentSummary = response.summary || null;
        currentAggregates = response.aggregates || {};
        currentLimit = response.limit || currentData.length || 0;
        currentTruncated = Boolean(response.truncated);

        refreshTable();
        updateSummaryCard();
        renderClientes(currentAggregates.clientes || []);
        renderEstados(currentAggregates.estados || []);
        updateLimitNotice();

        $('#btnExportPdf').prop('disabled', currentData.length === 0);
      })
      .fail(function(xhr) {
        console.error('[Reporte Pedidos] Error Ajax', xhr);
        notify('error', 'Reporte', 'No se pudo cargar el reporte.');
        clearData();
      })
      .always(function() {
        $('#btnExportPdf').prop('disabled', currentData.length === 0);
      });
  }

  function clearData() {
    currentData = [];
    currentSummary = null;
    currentAggregates = {};
    currentLimit = 0;
    currentTruncated = false;
    refreshTable();
    updateSummaryCard();
    renderClientes([]);
    renderEstados([]);
    updateLimitNotice();
  }

  function refreshTable() {
    if (!dataTable) return;
    dataTable.clear();
    if (currentData.length) {
      dataTable.rows.add(currentData);
    }
    dataTable.draw();
    $('#badgeTotalPedidos').text(formatNumber(currentData.length));
  }

  function updateSummaryCard() {
    if (!currentSummary || !currentSummary.total_pedidos) {
      $('#summaryCard').hide();
      $('#sumPedidos, #sumPedidosSub, #sumClientes, #sumClientesSub, #sumUc, #sumUcSub, #sumUv, #sumUvSub').text('-');
      $('#summaryRange').text('');
      return;
    }

    $('#sumPedidos').text(formatNumber(currentSummary.total_pedidos || 0));
    $('#sumPedidosSub').text('Destinos: ' + formatNumber(currentSummary.total_destinos || 0));

    $('#sumClientes').text(formatNumber(currentSummary.total_clientes || 0));
    $('#sumClientesSub').text('Destinatarios: ' + formatNumber(currentSummary.total_destinatarios || 0));

    $('#sumUc').text(formatNumber(currentSummary.total_expected_uc || 0));
    $('#sumUcSub').text('Despachadas: ' + formatNumber(currentSummary.total_shipped_uc || 0));

    $('#sumUv').text(formatNumber(currentSummary.total_expected_uv || 0));
    $('#sumUvSub').text('Despachadas: ' + formatNumber(currentSummary.total_shipped_uv || 0));

    $('#summaryRange').text(currentSummary.range_label || '');
    $('#summaryCard').show();
  }

  function renderClientes(list) {
    const $card = $('#cardClientes');
    const $tbody = $('#tablaClientes tbody');
    const $summary = $('#clientesSummary');

    if (!Array.isArray(list) || list.length === 0) {
      $tbody.html('<tr class="text-muted"><td colspan="5" class="text-center py-4">Sin datos disponibles</td></tr>');
      $summary.text('');
      $card.hide();
      return;
    }

    const totals = list.reduce(function(acc, item) {
      const rows = Number(item.rows || 0);
      const destinatarios = Number(item.destinatarios || 0);
      const expectedUc = Number(item.expected_uc || 0);
      const shippedUc = Number(item.shipped_uc || 0);
      return {
        rows: acc.rows + (Number.isFinite(rows) ? rows : 0),
        destinatarios: acc.destinatarios + (Number.isFinite(destinatarios) ? destinatarios : 0),
        expectedUc: acc.expectedUc + (Number.isFinite(expectedUc) ? expectedUc : 0),
        shippedUc: acc.shippedUc + (Number.isFinite(shippedUc) ? shippedUc : 0)
      };
    }, { rows: 0, destinatarios: 0, expectedUc: 0, shippedUc: 0 });

    const rowsHtml = list.map(function(item) {
      return (
        '<tr>' +
        '<td>' + (item.label || '-') + '</td>' +
        '<td class="text-end">' + formatNumber(item.rows || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.destinatarios || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.expected_uc || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.shipped_uc || 0) + '</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
    $summary.text('Top ' + list.length + ' · ' + formatNumber(totals.rows) + ' pedidos');
    $card.show();
  }

  function renderEstados(list) {
    const $card = $('#cardEstados');
    const $tbody = $('#tablaEstados tbody');
    const $summary = $('#estadosSummary');

    if (!Array.isArray(list) || list.length === 0) {
      $tbody.html('<tr class="text-muted"><td colspan="4" class="text-center py-4">Sin datos disponibles</td></tr>');
      $summary.text('');
      $card.hide();
      return;
    }

    const totals = list.reduce(function(acc, item) {
      const rows = Number(item.rows || 0);
      const expectedUc = Number(item.expected_uc || 0);
      const shippedUc = Number(item.shipped_uc || 0);
      return {
        rows: acc.rows + (Number.isFinite(rows) ? rows : 0),
        expectedUc: acc.expectedUc + (Number.isFinite(expectedUc) ? expectedUc : 0),
        shippedUc: acc.shippedUc + (Number.isFinite(shippedUc) ? shippedUc : 0)
      };
    }, { rows: 0, expectedUc: 0, shippedUc: 0 });

    const rowsHtml = list.map(function(item) {
      const code = item.code ? '<div class="text-muted small">' + item.code + '</div>' : '';
      return (
        '<tr>' +
        '<td>' + (item.label || '-') + code + '</td>' +
        '<td class="text-end">' + formatNumber(item.rows || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.expected_uc || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.shipped_uc || 0) + '</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
    $summary.text('Total pedidos: ' + formatNumber(totals.rows));
    $card.show();
  }

  function updateLimitNotice() {
    if (currentTruncated && currentLimit) {
      $('#limitValue').text(formatNumber(currentLimit));
      $('#limitNotice').show();
    } else {
      $('#limitNotice').hide();
    }
  }

  function exportPdf() {
    if (!Array.isArray(currentData) || currentData.length === 0) {
      notify('warning', 'Exportar', 'No hay datos para exportar.');
      return;
    }
    const query = $('#frmPedidos').serialize();
    const url = joinUrl('reportes/pedidos/pdf?' + query);
    window.open(url, '_blank');
  }

})(window, jQuery);
