/**
 * SOL · JavaScript · Reporte de Pedidos por Hora
 */

(function(window, $) {
  'use strict';

  if (!$ || typeof $.ajax !== 'function') {
    console.error('[Reporte Pedidos/Hora] jQuery es requerido.');
    return;
  }

  const baseUrl = document.querySelector('meta[name="base-url"]')?.getAttribute('content') ||
                  (typeof window.BASE_URL !== 'undefined' ? window.BASE_URL : '/');

  const numberFormatter = new Intl.NumberFormat('es-AR');
  const decimalFormatter = new Intl.NumberFormat('es-AR', { maximumFractionDigits: 2, minimumFractionDigits: 0 });

  let chartInstance = null;
  let currentSummary = null;
  let currentHours = [];
  let currentDaily = [];
  let currentAggregates = { clientes: [], estados: [] };
  let currentLimitDays = 0;

  $(document).ready(function() {
    initDefaultDates();
    bindEvents();
    loadCombos();
  });

  function initDefaultDates() {
    const today = new Date();
    const start = new Date(today.getTime() - (6 * 24 * 60 * 60 * 1000));

    const inputHasta = document.getElementById('fecha_hasta');
    const inputDesde = document.getElementById('fecha_desde');
    if (inputHasta) inputHasta.value = today.toISOString().slice(0, 10);
    if (inputDesde) inputDesde.value = start.toISOString().slice(0, 10);
  }

  function bindEvents() {
    $('#frmPedidosHora').on('submit', function(evt) {
      evt.preventDefault();
      loadData();
    });

    $('#btnLimpiar').on('click', function() {
      const form = document.getElementById('frmPedidosHora');
      if (form) {
        form.reset();
      }
      initDefaultDates();
      loadData();
    });

    $('#btnExportPdf').on('click', function() {
      exportPdf();
    });
  }

  function joinUrl(path) {
    return baseUrl.replace(/\/$/, '') + '/' + String(path || '').replace(/^\//, '');
  }

  function notify(type, title, text) {
    if (typeof Swal !== 'undefined') {
      Swal.fire({ icon: type, title, text, timer: 3200, showConfirmButton: false });
      return;
    }
    const message = '[' + title + '] ' + text;
    if (type === 'error') {
      console.error(message);
    } else if (type === 'warning') {
      console.warn(message);
    } else {
      console.info(message);
    }
  }

  function loadCombos() {
    $.ajax({
      url: joinUrl('api/reportes/pedidos_hora.php'),
      method: 'GET',
      data: { meta: 'filters' },
      dataType: 'json'
    })
      .done(function(response) {
        if (!response || !response.ok) {
          notify('error', 'Filtros', 'No se pudieron cargar los filtros iniciales.');
          return;
        }
        applyCombos(response);
        loadData();
      })
      .fail(function(xhr) {
        console.error('[Reporte Pedidos/Hora] Error combos', xhr);
        notify('error', 'Filtros', 'Error al contactar el servidor.');
      });
  }

  function applyCombos(data) {
    const clientes = Array.isArray(data.clientes) ? data.clientes : [];
    const estados = Array.isArray(data.estados) ? data.estados : [];

    const $cliente = $('#cliente_id');
    $cliente.find('option:not(:first)').remove();
    clientes.forEach(function(item) {
      $('<option></option>').val(item.id).text(item.nombre).appendTo($cliente);
    });

    const $estado = $('#estado_id');
    $estado.find('option:not(:first)').remove();
    estados.forEach(function(item) {
      $('<option></option>').val(item.id).text(item.nombre).appendTo($estado);
    });
  }

  function loadData() {
    const formData = $('#frmPedidosHora').serialize();

    $.ajax({
      url: joinUrl('api/reportes/pedidos_hora.php'),
      method: 'GET',
      data: formData,
      dataType: 'json',
      beforeSend: function() {
        $('#btnExportPdf').prop('disabled', true);
      }
    })
      .done(function(response) {
        if (!response || !response.ok) {
          const errorMsg = response && response.error ? response.error : 'No se obtuvo respuesta válida del servidor.';
          notify('error', 'Reporte', errorMsg);
          clearData();
          return;
        }
        currentSummary = response.summary || null;
        currentHours = Array.isArray(response.hours) ? response.hours : [];
        currentDaily = Array.isArray(response.daily) ? response.daily : [];
        currentAggregates = response.aggregates || { clientes: [], estados: [] };
        currentLimitDays = response.limit_days || 0;

        updateSummary();
        renderChart(response.chart || null);
        renderHoursTable();
        renderDailyTable();
        renderAggregates();
        updateLimitNotice();

        $('#btnExportPdf').prop('disabled', (currentSummary && currentSummary.total_pedidos > 0) ? false : true);
      })
      .fail(function(xhr) {
        console.error('[Reporte Pedidos/Hora] Error datos', xhr);
        notify('error', 'Reporte', 'No se pudo generar el reporte.');
        clearData();
      });
  }

  function clearData() {
    currentSummary = null;
    currentHours = [];
    currentDaily = [];
    currentAggregates = { clientes: [], estados: [] };
    currentLimitDays = 0;

    updateSummary();
    renderChart(null);
    renderHoursTable();
    renderDailyTable();
    renderAggregates();
    updateLimitNotice();
  }

  function updateSummary() {
    if (!currentSummary) {
      $('#sumTotalPedidos').text('-');
      $('#sumClientes').text('-');
      $('#sumDias').text('-');
      $('#sumPromedioDia').text('-');
      $('#sumPromedioHora').text('-');
      $('#sumHoraPico').text('-');
      $('#sumHoraPicoPedidos').text('-');
      $('#sumHorasActivas').text('-');
      $('#sumRange').text('-');
      $('#sumPrimerRegistro').text('-');
      $('#sumUltimoRegistro').text('-');
      $('#rangeLabel').text('');
      $('#badgeTotalHoras').text('0');
      return;
    }

    $('#sumTotalPedidos').text(numberFormatter.format(currentSummary.total_pedidos || 0));
    $('#sumClientes').text(numberFormatter.format(currentSummary.total_clientes || 0));
    $('#sumDias').text(numberFormatter.format(currentSummary.total_dias || 0));

    const promedioDia = currentSummary.promedio_diario || 0;
    const promedioHora = currentSummary.promedio_por_hora || 0;
    $('#sumPromedioDia').text(decimalFormatter.format(promedioDia));
    $('#sumPromedioHora').text(decimalFormatter.format(promedioHora));

    $('#sumHoraPico').text(currentSummary.hora_pico || '-');
    $('#sumHoraPicoPedidos').text(numberFormatter.format(currentSummary.hora_pico_pedidos || 0));
    $('#sumHorasActivas').text(numberFormatter.format(currentSummary.horas_activas || 0));

    $('#sumRange').text(currentSummary.range_label || '-');
    $('#sumPrimerRegistro').text(formatDateTime(currentSummary.primer_registro));
    $('#sumUltimoRegistro').text(formatDateTime(currentSummary.ultimo_registro));

    $('#rangeLabel').text(currentSummary.range_label || '');
    $('#badgeTotalHoras').text(numberFormatter.format(currentSummary.total_pedidos || 0));
  }

  function formatDateTime(value) {
    if (!value) {
      return '-';
    }
    const str = String(value);
    const match = str.match(/^(\d{4})-(\d{2})-(\d{2})(?:[ T](\d{2}):(\d{2}))?/);
    if (!match) {
      return str;
    }
    const day = match[3] + '/' + match[2] + '/' + match[1];
    if (match[4] && match[5]) {
      return day + ' ' + match[4] + ':' + match[5];
    }
    return day;
  }

  function renderChart(chartData) {
    const canvas = document.getElementById('chartPedidosHora');
    if (!canvas) {
      return;
    }

    if (chartInstance) {
      chartInstance.destroy();
      chartInstance = null;
    }

    if (!chartData || !Array.isArray(chartData.labels)) {
      return;
    }

    if (typeof window.Chart === 'undefined') {
      console.warn('[Reporte Pedidos/Hora] Chart.js no está disponible.');
      return;
    }

    const dataValues = Array.isArray(chartData.values) ? chartData.values : [];

    chartInstance = new window.Chart(canvas.getContext('2d'), {
      type: 'bar',
      data: {
        labels: chartData.labels,
        datasets: [{
          label: 'Pedidos',
          data: dataValues,
          backgroundColor: 'rgba(13, 110, 253, 0.65)',
          borderColor: '#0d6efd',
          borderWidth: 1.5,
          hoverBackgroundColor: 'rgba(13, 110, 253, 0.85)'
        }]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        scales: {
          y: {
            beginAtZero: true,
            ticks: {
              precision: 0
            }
          }
        },
        plugins: {
          legend: { display: false },
          tooltip: {
            callbacks: {
              label: function(context) {
                return 'Pedidos: ' + numberFormatter.format(context.parsed.y || 0);
              }
            }
          }
        }
      }
    });
  }

  function renderHoursTable() {
    const $tbody = $('#tablaHoras tbody');
    if (!Array.isArray(currentHours) || currentHours.length === 0) {
      $tbody.html('<tr><td colspan="5" class="text-center text-muted py-4">Sin datos disponibles</td></tr>');
      return;
    }

    const rowsHtml = currentHours.map(function(item) {
      const pedidos = numberFormatter.format(item.pedidos || 0);
      const promDia = decimalFormatter.format(item.promedio_por_dia || 0);
      const clientes = numberFormatter.format(item.clientes || 0);
      const porcentaje = decimalFormatter.format(item.porcentaje || 0);
      return (
        '<tr>' +
          '<td>' + item.label + '</td>' +
          '<td class="text-end fw-semibold">' + pedidos + '</td>' +
          '<td class="text-end">' + promDia + '</td>' +
          '<td class="text-end">' + clientes + '</td>' +
          '<td class="text-end">' + porcentaje + '%</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
  }

  function renderDailyTable() {
    const $tbody = $('#tablaDiaria tbody');
    $('#limitDias').text(numberFormatter.format(currentLimitDays || 0));

    if (!Array.isArray(currentDaily) || currentDaily.length === 0) {
      $tbody.html('<tr><td colspan="5" class="text-center text-muted py-4">Sin datos disponibles</td></tr>');
      return;
    }

    const rowsHtml = currentDaily.map(function(item) {
      const fecha = item.fecha ? formatDateLabel(item.fecha) : '-';
      const pedidos = numberFormatter.format(item.pedidos || 0);
      const clientes = numberFormatter.format(item.clientes || 0);
      const primer = item.primer_pedido ? item.primer_pedido : '-';
      const ultimo = item.ultimo_pedido ? item.ultimo_pedido : '-';
      return (
        '<tr>' +
          '<td>' + fecha + '</td>' +
          '<td class="text-end fw-semibold">' + pedidos + '</td>' +
          '<td class="text-end">' + clientes + '</td>' +
          '<td class="text-end">' + primer + '</td>' +
          '<td class="text-end">' + ultimo + '</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
  }

  function renderAggregates() {
    renderAggregateList('#listaTopClientes', currentAggregates.clientes, 'pedidos');
    renderAggregateList('#listaTopEstados', currentAggregates.estados, 'pedidos', true);

    const totalClientes = Array.isArray(currentAggregates.clientes)
      ? currentAggregates.clientes.reduce(function(acc, item) { return acc + (item.pedidos || 0); }, 0)
      : 0;
    const totalEstados = Array.isArray(currentAggregates.estados)
      ? currentAggregates.estados.reduce(function(acc, item) { return acc + (item.pedidos || 0); }, 0)
      : 0;

    $('#clientesSummary').text(totalClientes ? numberFormatter.format(totalClientes) + ' pedidos' : '');
    $('#estadosSummary').text(totalEstados ? numberFormatter.format(totalEstados) + ' pedidos' : '');
  }

  function renderAggregateList(selector, list, metric, includeCode) {
    const $list = $(selector);
    if (!Array.isArray(list) || list.length === 0) {
      $list.html('<li class="list-group-item text-muted text-center">Sin información disponible</li>');
      return;
    }

    const itemsHtml = list.map(function(item) {
      const label = includeCode && item.code ? item.code + ' · ' + item.nombre : item.nombre;
      const value = numberFormatter.format(item[metric] || 0);
      return (
        '<li class="list-group-item d-flex justify-content-between align-items-center">' +
          '<span>' + label + '</span>' +
          '<span class="badge-soft">' + value + '</span>' +
        '</li>'
      );
    }).join('');

    $list.html(itemsHtml);
  }

  function formatDateLabel(value) {
    if (!value) {
      return '-';
    }
    const parts = String(value).split('-');
    if (parts.length !== 3) {
      return value;
    }
    return parts[2] + '/' + parts[1] + '/' + parts[0];
  }

  function updateLimitNotice() {
    if (!currentDaily || !currentDaily.length) {
      $('#limitNotice').hide();
      return;
    }
    if (currentLimitDays && currentDaily.length >= currentLimitDays) {
      $('#limitNotice').show();
    } else {
      $('#limitNotice').hide();
    }
  }

  function exportPdf() {
    if (!currentSummary || !currentSummary.total_pedidos) {
      notify('warning', 'Exportación', 'No hay datos para exportar.');
      return;
    }
    const params = $('#frmPedidosHora').serializeArray();
    params.push({ name: 'limit', value: currentLimitDays || 90 });
    const query = $.param(params);
    const url = joinUrl('reportes/pedidos_hora/pdf') + '?' + query;
    window.open(url, '_blank');
  }

})(window, window.jQuery);
