/**
 * SOL · JavaScript · Reporte de Salidas
 */

(function(window, $) {
  'use strict';

  if (!$ || typeof $.ajax !== 'function') {
    console.error('[Reporte Salidas] jQuery es requerido.');
    return;
  }

  const baseUrl = document.querySelector('meta[name="base-url"]')?.getAttribute('content') ||
                  (typeof window.BASE_URL !== 'undefined' ? window.BASE_URL : '/');

  function joinUrl(path) {
    return baseUrl.replace(/\/$/, '') + '/' + String(path || '').replace(/^\//, '');
  }

  const numberFormatter = new Intl.NumberFormat('es-AR');
  function formatNumber(value) {
    const numeric = Number(value);
    return numberFormatter.format(Number.isFinite(numeric) ? numeric : 0);
  }

  function formatDecimal(value, decimals) {
    const numeric = Number(value);
    if (!Number.isFinite(numeric)) {
      return numberFormatter.format(0);
    }
    return new Intl.NumberFormat('es-AR', {
      minimumFractionDigits: decimals,
      maximumFractionDigits: decimals
    }).format(numeric);
  }

  function notify(type, title, text) {
    if (typeof Swal !== 'undefined') {
      Swal.fire({ icon: type, title, text, timer: 3200, showConfirmButton: false });
      return;
    }
    const message = '[' + title + '] ' + text;
    if (type === 'error') {
      console.error(message);
    } else if (type === 'warning') {
      console.warn(message);
    } else {
      console.info(message);
    }
  }

  let dataTable = null;
  let combosLoaded = false;
  let currentData = [];
  let currentSummary = null;
  let currentAggregates = {};
  let currentLimit = 0;
  let currentTruncated = false;

  $(document).ready(function() {
    initFormDefaults();
    initDataTable();
    bindEvents();
    loadCombos();
  });

  function initFormDefaults() {
    const today = new Date();
    const start = new Date(today.getTime() - (29 * 24 * 60 * 60 * 1000));
    const inputHasta = document.getElementById('fecha_hasta');
    const inputDesde = document.getElementById('fecha_desde');
    if (inputHasta) inputHasta.value = today.toISOString().slice(0, 10);
    if (inputDesde) inputDesde.value = start.toISOString().slice(0, 10);
  }

  function bindEvents() {
    $('#frmSalidas').on('submit', function(evt) {
      evt.preventDefault();
      loadData();
    });

    $('#btnLimpiar').on('click', function() {
      const form = document.getElementById('frmSalidas');
      if (form) {
        form.reset();
      }
      initFormDefaults();
      loadData();
    });

    $('#btnExportPdf').on('click', function() {
      exportPdf();
    });
  }

  function loadCombos() {
    $.ajax({
      url: joinUrl('api/reportes/salidas.php'),
      method: 'GET',
      data: { meta: 'filters' },
      dataType: 'json'
    })
      .done(function(response) {
        if (!response || !response.ok) {
          notify('error', 'Combos', 'No fue posible cargar los filtros');
          return;
        }
        applyCombos(response);
        combosLoaded = true;
        loadData();
      })
      .fail(function(xhr) {
        console.error('[Reporte Salidas] Error combos', xhr);
        notify('error', 'Combos', 'No se pudieron cargar los filtros');
      });
  }

  function applyCombos(data) {
    const depositos = Array.isArray(data.depositos) ? data.depositos : [];
    const estados = Array.isArray(data.estados) ? data.estados : [];
    const moviles = Array.isArray(data.moviles) ? data.moviles : [];
    const choferes = Array.isArray(data.choferes) ? data.choferes : [];

    const $deposito = $('#deposito_id');
    $deposito.find('option:not(:first)').remove();
    depositos.forEach(function(dep) {
      $('<option></option>').val(dep.id).text(dep.nombre).appendTo($deposito);
    });

    const $estado = $('#estado');
    $estado.find('option:not(:first)').remove();
    estados.forEach(function(est) {
      $('<option></option>').val(est.code).text(est.nombre + ' [' + est.code + ']').appendTo($estado);
    });

    const $movil = $('#movil_id');
    $movil.find('option:not(:first)').remove();
    moviles.forEach(function(mov) {
      $('<option></option>').val(mov.id).text(mov.label).appendTo($movil);
    });

    const $chofer = $('#chofer_id');
    $chofer.find('option:not(:first)').remove();
    choferes.forEach(function(cho) {
      $('<option></option>').val(cho.id).text(cho.nombre).appendTo($chofer);
    });
  }

  function initDataTable() {
    if (!$('#tblSalidas').length) {
      console.error('[Reporte Salidas] Tabla no encontrada');
      return;
    }

    dataTable = $('#tblSalidas').DataTable({
      processing: true,
      serverSide: false,
      responsive: true,
      language: {
        url: 'https://cdn.datatables.net/plug-ins/1.13.7/i18n/es-ES.json'
      },
      order: [[0, 'desc']],
      pageLength: 25,
      columns: [
        {
          data: 'fecha_creado',
          title: 'Fecha',
          render: function(data, type, row) {
            if (type === 'display' || type === 'filter') {
              const fecha = data || '-';
              const hora = row && row.hora_creado ? row.hora_creado : '';
              return hora ? fecha + ' ' + hora : fecha;
            }
            return data || '';
          }
        },
        { data: 'codigo', title: 'Código', defaultContent: '-' },
        { data: 'deposito_nombre', title: 'Depósito', defaultContent: '-' },
        {
          data: 'estado_nombre',
          title: 'Estado',
          render: function(data, type, row) {
            if (type === 'display' && row && row.estado_code) {
              return '<span>' + (data || '-') + '</span><div class="text-muted small">' + row.estado_code + '</div>';
            }
            return data || '-';
          }
        },
        { data: 'movil_chapa', title: 'Móvil', defaultContent: '-' },
        { data: 'chofer_nombre', title: 'Chofer', defaultContent: '-' },
        {
          data: 'destinos',
          title: 'Destinos',
          className: 'text-end',
          render: function(value) { return formatNumber(value); }
        },
        {
          data: 'preembarques',
          title: 'Preembarques',
          className: 'text-end',
          render: function(value) { return formatNumber(value); }
        },
        {
          data: 'tiempo_espera_min',
          title: 'Espera (min)',
          className: 'text-end',
          render: function(value) { return value === null ? '-' : formatNumber(value); }
        },
        {
          data: 'tiempo_carga_min',
          title: 'Carga (min)',
          className: 'text-end',
          render: function(value) { return value === null ? '-' : formatNumber(value); }
        },
        {
          data: 'tiempo_planta_min',
          title: 'Planta (min)',
          className: 'text-end',
          render: function(value) { return value === null ? '-' : formatNumber(value); }
        },
        {
          data: 'ayudantes_cant',
          title: 'Ayudantes',
          className: 'text-end',
          render: function(value) { return value === null ? '-' : formatNumber(value); }
        },
        {
          data: 'temp_salida_c',
          title: 'Temp. salida',
          className: 'text-end',
          render: function(value) { return value === null ? '-' : formatDecimal(value, 1); }
        },
        {
          data: 'km_inicial',
          title: 'Km inicial',
          className: 'text-end',
          render: function(value) { return value === null ? '-' : formatDecimal(value, 1); }
        },
        {
          data: 'observacion',
          title: 'Observaciones',
          render: function(value) { return value ? String(value) : '-'; }
        }
      ]
    });
  }

  function loadData() {
    if (!combosLoaded) {
      return;
    }

    const formData = $('#frmSalidas').serialize();

    $.ajax({
      url: joinUrl('api/reportes/salidas.php'),
      method: 'GET',
      data: formData,
      dataType: 'json',
      beforeSend: function() {
        $('#btnExportPdf').prop('disabled', true);
      }
    })
      .done(function(response) {
        if (!response || !response.ok) {
          const errorMsg = response && response.error ? response.error : 'No fue posible obtener los datos.';
          notify('error', 'Reporte', errorMsg);
          clearData();
          return;
        }

        currentData = Array.isArray(response.rows) ? response.rows : [];
        currentSummary = response.summary || null;
        currentAggregates = response.aggregates || {};
        currentLimit = response.limit || currentData.length || 0;
        currentTruncated = Boolean(response.truncated);

        refreshTable();
        updateSummaryCard();
        renderDepositos(currentAggregates.depositos || []);
        renderMoviles(currentAggregates.moviles || []);
        renderEstados(currentAggregates.estados || []);
        updateLimitNotice();

        $('#btnExportPdf').prop('disabled', currentData.length === 0);
      })
      .fail(function(xhr) {
        console.error('[Reporte Salidas] Error Ajax', xhr);
        notify('error', 'Reporte', 'No se pudo cargar el reporte.');
        clearData();
      })
      .always(function() {
        $('#btnExportPdf').prop('disabled', currentData.length === 0);
      });
  }

  function clearData() {
    currentData = [];
    currentSummary = null;
    currentAggregates = {};
    currentLimit = 0;
    currentTruncated = false;
    refreshTable();
    updateSummaryCard();
    renderDepositos([]);
    renderMoviles([]);
    renderEstados([]);
    updateLimitNotice();
  }

  function refreshTable() {
    if (!dataTable) return;
    dataTable.clear();
    if (currentData.length) {
      dataTable.rows.add(currentData);
    }
    dataTable.draw();
    $('#badgeTotalSalidas').text(formatNumber(currentData.length));
  }

  function updateSummaryCard() {
    if (!currentSummary || !currentSummary.total_salidas) {
      $('#summaryCard').hide();
      $('#sumSalidas, #sumSalidasSub, #sumDestinos, #sumDestinosSub, #sumPreembarques, #sumPreembarquesSub, #sumTiempos, #sumTiemposSub').text('-');
      $('#summaryRange').text('');
      return;
    }

    const totalSalidas = currentSummary.total_salidas || 0;
    const totalDestinos = currentSummary.total_destinos || 0;
    const totalPre = currentSummary.total_preembarques || 0;
    const avgDestinos = totalSalidas > 0 ? totalDestinos / totalSalidas : 0;
    const avgPre = totalSalidas > 0 ? totalPre / totalSalidas : 0;

    $('#sumSalidas').text(formatNumber(totalSalidas));
    $('#sumSalidasSub').text('Depósitos: ' + formatNumber(currentSummary.total_depositos || 0) + ' · Choferes: ' + formatNumber(currentSummary.total_choferes || 0));

    $('#sumDestinos').text(formatNumber(totalDestinos));
    $('#sumDestinosSub').text('Prom. por salida: ' + formatDecimal(avgDestinos, 1));

    $('#sumPreembarques').text(formatNumber(totalPre));
    $('#sumPreembarquesSub').text('Móviles: ' + formatNumber(currentSummary.total_moviles || 0) + ' · Prom.: ' + formatDecimal(avgPre, 1));

    const avgPlanta = currentSummary.avg_planta_min || 0;
    const avgEspera = currentSummary.avg_espera_min || 0;
    const avgCarga = currentSummary.avg_carga_min || 0;
    $('#sumTiempos').text(formatDecimal(avgPlanta, 1));
    $('#sumTiemposSub').text('Espera: ' + formatDecimal(avgEspera, 1) + ' · Carga: ' + formatDecimal(avgCarga, 1));

    $('#summaryRange').text(currentSummary.range_label || '');
    $('#summaryCard').show();
  }

  function renderDepositos(list) {
    const $card = $('#cardDepositos');
    const $tbody = $('#tablaDepositos tbody');
    const $summary = $('#depositosSummary');

    if (!Array.isArray(list) || list.length === 0) {
      $tbody.html('<tr class="text-muted"><td colspan="4" class="text-center py-4">Sin datos disponibles</td></tr>');
      $summary.text('');
      $card.hide();
      return;
    }

    const totals = list.reduce(function(acc, item) {
      const rows = Number(item.rows || 0);
      const destinos = Number(item.destinos || 0);
      const pre = Number(item.preembarques || 0);
      return {
        rows: acc.rows + (Number.isFinite(rows) ? rows : 0),
        destinos: acc.destinos + (Number.isFinite(destinos) ? destinos : 0),
        pre: acc.pre + (Number.isFinite(pre) ? pre : 0)
      };
    }, { rows: 0, destinos: 0, pre: 0 });

    const rowsHtml = list.map(function(item) {
      return (
        '<tr>' +
        '<td>' + (item.label || '-') + '</td>' +
        '<td class="text-end">' + formatNumber(item.rows || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.destinos || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.preembarques || 0) + '</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
    $summary.text('Top ' + list.length + ' · ' + formatNumber(totals.rows) + ' salidas');
    $card.show();
  }

  function renderMoviles(list) {
    const $card = $('#cardMoviles');
    const $tbody = $('#tablaMoviles tbody');
    const $summary = $('#movilesSummary');

    if (!Array.isArray(list) || list.length === 0) {
      $tbody.html('<tr class="text-muted"><td colspan="4" class="text-center py-4">Sin datos disponibles</td></tr>');
      $summary.text('');
      $card.hide();
      return;
    }

    const totals = list.reduce(function(acc, item) {
      const rows = Number(item.rows || 0);
      const destinos = Number(item.destinos || 0);
      const planta = Number(item.planta_min || 0);
      return {
        rows: acc.rows + (Number.isFinite(rows) ? rows : 0),
        destinos: acc.destinos + (Number.isFinite(destinos) ? destinos : 0),
        planta: acc.planta + (Number.isFinite(planta) ? planta : 0)
      };
    }, { rows: 0, destinos: 0, planta: 0 });

    const rowsHtml = list.map(function(item) {
      const rows = Number(item.rows || 0);
      const planta = Number(item.planta_min || 0);
      const promedioPlanta = rows > 0 ? planta / rows : 0;
      return (
        '<tr>' +
        '<td>' + (item.label || '-') + '</td>' +
        '<td class="text-end">' + formatNumber(rows) + '</td>' +
        '<td class="text-end">' + formatNumber(item.destinos || 0) + '</td>' +
        '<td class="text-end">' + formatDecimal(promedioPlanta, 1) + '</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
    const promedio = totals.rows > 0 ? totals.planta / totals.rows : 0;
    $summary.text('Top ' + list.length + ' · Prom. planta: ' + formatDecimal(promedio, 1) + ' min');
    $card.show();
  }

  function renderEstados(list) {
    const $card = $('#cardEstados');
    const $tbody = $('#tablaEstados tbody');
    const $summary = $('#estadosSummary');

    if (!Array.isArray(list) || list.length === 0) {
      $tbody.html('<tr class="text-muted"><td colspan="4" class="text-center py-4">Sin datos disponibles</td></tr>');
      $summary.text('');
      $card.hide();
      return;
    }

    const totals = list.reduce(function(acc, item) {
      const rows = Number(item.rows || 0);
      const destinos = Number(item.destinos || 0);
      const pre = Number(item.preembarques || 0);
      return {
        rows: acc.rows + (Number.isFinite(rows) ? rows : 0),
        destinos: acc.destinos + (Number.isFinite(destinos) ? destinos : 0),
        pre: acc.pre + (Number.isFinite(pre) ? pre : 0)
      };
    }, { rows: 0, destinos: 0, pre: 0 });

    const rowsHtml = list.map(function(item) {
      const code = item.code ? ' [' + item.code + ']' : '';
      return (
        '<tr>' +
        '<td>' + (item.label || '-') + code + '</td>' +
        '<td class="text-end">' + formatNumber(item.rows || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.destinos || 0) + '</td>' +
        '<td class="text-end">' + formatNumber(item.preembarques || 0) + '</td>' +
        '</tr>'
      );
    }).join('');

    $tbody.html(rowsHtml);
    $summary.text(formatNumber(totals.rows) + ' salidas · ' + formatNumber(totals.destinos) + ' destinos');
    $card.show();
  }

  function updateLimitNotice() {
    if (!currentTruncated || !currentLimit) {
      $('#limitNotice').hide();
      return;
    }
    $('#limitValue').text(formatNumber(currentLimit));
    $('#limitNotice').show();
  }

  function exportPdf() {
    if (!currentData.length) {
      notify('warning', 'Exportación', 'No hay datos para exportar.');
      return;
    }

    const params = $('#frmSalidas').serializeArray();
    params.push({ name: 'limit', value: currentLimit || 1500 });
    const query = $.param(params);
    const url = joinUrl('reportes/salidas/pdf') + '?' + query;
    window.open(url, '_blank');
  }

})(window, window.jQuery);
