/**
 * SOL · Reporte de Servicio al Cliente
 */
(function (window, $) {
  'use strict';

  const baseUrl =
    document.querySelector('meta[name="base-url"]')?.getAttribute('content') ||
    (typeof window.BASE_URL !== 'undefined' ? window.BASE_URL : '/');
  const joinUrl = (path) => `${baseUrl.replace(/\/$/, '')}/${String(path || '').replace(/^\//, '')}`;
  const apiUrl = joinUrl('api/reportes/servicio_cliente.php');

  const $form = $('#frmServicioCliente');
  const $periodo = $('#servicioPeriodo');
  const $summary = $('#servicioResumen');

  $(function () {
    bindEvents();
    loadFilters().always(loadData);
  });

  function bindEvents() {
    $form.on('submit', function (event) {
      event.preventDefault();
      loadData();
    });

    $('#btnServicioLimpiar').on('click', function () {
      const formEl = $form.get(0);
      if (formEl) {
        formEl.reset();
      }
      $('#deposito_id').val('');
      $('#cliente_id').val('');
      $('#movil_id').val('');
      loadData();
    });
  }

  function loadFilters() {
    return $.getJSON(apiUrl, { meta: 'filters' })
      .done(function (response) {
        if (!response || response.ok === false) {
          notifier('warning', response?.error || 'No se pudieron cargar los filtros');
          return;
        }
        fillSelect('#deposito_id', response.depositos, 'Todos');
        fillSelect('#cliente_id', response.clientes, 'Todos');
        fillSelect('#movil_id', response.moviles, 'Todos');
      })
      .fail(function () {
        notifier('error', 'Error de conexión al cargar filtros');
      });
  }

  function fillSelect(selector, items, placeholder) {
    const $select = $(selector);
    if (!$select.length) return;
    const current = $select.val();
    $select.find('option').not(':first').remove();
    if (Array.isArray(items)) {
      items.forEach(function (item) {
        const id = item.id ?? '';
        const label = item.label ?? item.nombre ?? item.descripcion ?? item.codigo ?? `ID ${id}`;
        $select.append(new Option(label, id));
      });
    }
    if (current && $select.find(`option[value="${current}"]`).length) {
      $select.val(current);
    } else {
      $select.val('');
    }
  }

  function loadData() {
    const formEl = $form.get(0);
    if (!formEl) {
      return;
    }
    const params = Object.fromEntries(new FormData(formEl).entries());
    setLoading(true);

    $.ajax({
      url: apiUrl,
      method: 'GET',
      data: params,
      dataType: 'json',
    })
      .done(function (response) {
        if (!response || response.ok === false) {
          notifier('error', response?.error || 'No se pudo generar el reporte');
          resetView();
          return;
        }
        updateFormFilters(response.filters || {});
        updateMetrics(response.metrics || []);
        updateSummary(response.summary || {});
        updatePeriodo(response.periodo || '');
      })
      .fail(function () {
        notifier('error', 'Error de conexión con el servidor');
        resetView();
      })
      .always(function () {
        setLoading(false);
      });
  }

  function updateFormFilters(filters) {
    if (!filters) return;
    $('#fecha_desde').val(filters.fecha_desde ?? '');
    $('#fecha_hasta').val(filters.fecha_hasta ?? '');
    $('#tolerancia_min').val(filters.tolerancia_min ?? 60);
    $('#meta_otif').val(filters.meta_otif ?? 98);
    $('#meta_reclamaciones').val(filters.meta_reclamaciones ?? 1);
    $('#meta_satisfaccion').val(filters.meta_satisfaccion ?? 95);
    if (filters.cliente_id) {
      $('#cliente_id').val(String(filters.cliente_id));
    }
    if (filters.deposito_id) {
      $('#deposito_id').val(String(filters.deposito_id));
    }
    if (filters.movil_id) {
      $('#movil_id').val(String(filters.movil_id));
    }
  }

  function updateMetrics(metrics) {
    const map = {};
    if (Array.isArray(metrics)) {
      metrics.forEach(function (metric) {
        if (metric && metric.clave) {
          map[metric.clave] = metric;
        }
      });
    }

    $('[data-metric]').each(function () {
      const $row = $(this);
      const key = $row.data('metric');
      const metric = map[key] || null;
      const estado = metric?.estado || 'SIN_DATOS';
      const valorTexto = metric?.valor !== null && metric?.valor !== undefined
        ? formatValue(metric.valor, metric.unidad)
        : '-';
      const metaTexto = metric?.meta !== null && metric?.meta !== undefined
        ? formatValue(metric.meta, metric.unidad)
        : '-';

      $row.find('[data-field="indicador"]').text(metric?.indicador || $row.find('[data-field="indicador"]').data('default'));
      $row.find('[data-field="formula"]').text(metric?.formula || '-');
      $row.find('[data-field="meta"]').text(metaTexto);
      $row.find('[data-field="resultado"]').text(valorTexto);
      $row.find('[data-field="observaciones"]').text(metric?.observaciones || 'Sin datos');

      const $badge = $row.find('[data-field="semaforo"]');
      $badge.attr('class', `badge ${mapEstadoToBadge(estado)}`).text(mapEstadoToLabel(estado));
    });
  }

  function updateSummary(summary) {
    const pedidos = summary?.pedidos_totales ?? 0;
    const otif = summary?.otif_pct ?? null;
    const reclamos = summary?.reclamos ?? 0;
    const reclamosPct = summary?.reclamos_pct ?? null;
    const encuestas = summary?.encuestas ?? 0;
    const satisfaccion = summary?.satisfaccion_pct ?? null;

    $summary.find('[data-field="pedidos_totales"]').text(formatNumber(pedidos, 0));
    $summary.find('[data-field="otif_pct"]').text(formatPercent(otif));
    $summary.find('[data-field="reclamos"]').text(`${formatNumber(reclamos, 0)} (${formatPercent(reclamosPct)})`);
    $summary.find('[data-field="encuestas"]').text(`${formatNumber(encuestas, 0)} (${formatPercent(satisfaccion)})`);
  }

  function updatePeriodo(periodo) {
    if (periodo) {
      $periodo.text(`Periodo: ${periodo}`);
    } else {
      $periodo.text('Seleccione filtros para ver resultados');
    }
  }

  function resetView() {
    updateMetrics([]);
    $summary.find('[data-field]').text('-');
    updatePeriodo('');
  }

  function setLoading(isLoading) {
    const $target = $('#tablaServicioCliente');
    if (isLoading) {
      $target.addClass('opacity-50 position-relative');
      $target.find('.loading-overlay').remove();
      $target.append('<div class="loading-overlay position-absolute top-50 start-50 translate-middle text-muted"><div class="spinner-border spinner-border-sm me-2" role="status"></div>Procesando...</div>');
    } else {
      $target.removeClass('opacity-50 position-relative');
      $target.find('.loading-overlay').remove();
    }
  }

  function formatValue(value, unidad) {
    if (value === null || value === undefined) {
      return '-';
    }
    const number = Number(value);
    if (Number.isNaN(number)) {
      return '-';
    }
    const decimals = unidad === '%' ? 2 : unidad === 'pts' ? 2 : 2;
    const formatted = formatNumber(number, decimals);
    if (!unidad) {
      return formatted;
    }
    if (unidad === '%') {
      return `${formatted}%`;
    }
    return `${formatted} ${unidad}`.trim();
  }

  function formatNumber(value, decimals) {
    const num = Number(value);
    if (!Number.isFinite(num)) {
      return '-';
    }
    try {
      return new Intl.NumberFormat('es-AR', {
        minimumFractionDigits: decimals,
        maximumFractionDigits: decimals,
      }).format(num);
    } catch (err) {
      return num.toFixed(decimals);
    }
  }

  function formatPercent(value) {
    if (value === null || value === undefined || Number.isNaN(Number(value))) {
      return '-';
    }
    return `${formatNumber(Number(value), 2)}%`;
  }

  function mapEstadoToBadge(estado) {
    switch (estado) {
      case 'OK':
        return 'bg-success';
      case 'ALERTA':
        return 'bg-warning text-dark';
      case 'CRITICO':
        return 'bg-danger';
      default:
        return 'bg-secondary';
    }
  }

  function mapEstadoToLabel(estado) {
    switch (estado) {
      case 'OK':
        return 'Verde';
      case 'ALERTA':
        return 'Amarillo';
      case 'CRITICO':
        return 'Rojo';
      default:
        return 'Sin datos';
    }
  }

  function notifier(type, message) {
    if (typeof Swal !== 'undefined' && typeof Swal.fire === 'function') {
      Swal.fire({ icon: type, title: message, timer: 2600, showConfirmButton: false });
      return;
    }
    if (type === 'error') {
      console.error('[Servicio al Cliente]', message);
    } else if (type === 'warning') {
      console.warn('[Servicio al Cliente]', message);
    } else {
      console.info('[Servicio al Cliente]', message);
    }
  }
})(window, window.jQuery);
