/**
 * SOL · JavaScript · Reporte de Quiebre de Stock
 */
(function (window, $) {
  'use strict';

  const baseUrl =
    document.querySelector('meta[name="base-url"]')?.getAttribute('content') ||
    (typeof window.BASE_URL !== 'undefined' ? window.BASE_URL : '/');

  const joinUrl = (path) => `${baseUrl.replace(/\/$/, '')}/${String(path || '').replace(/^\//, '')}`;

  const formatNumber = (value) => {
    const num = Number(value ?? 0);
    if (!Number.isFinite(num)) return '0';
    try {
      return new Intl.NumberFormat('es-AR').format(num);
    } catch (err) {
      console.warn('[StockQuiebre] Intl format fallback', err);
      return num.toString();
    }
  };

  const formatPercent = (value) => {
    if (value === null || value === undefined || Number.isNaN(Number(value))) {
      return '-';
    }
    const num = Number(value);
    const formatter = new Intl.NumberFormat('es-AR', {
      minimumFractionDigits: 1,
      maximumFractionDigits: 1,
    });
    return `${formatter.format(num)}%`;
  };

  const notify = (type, message) => {
    if (typeof Swal !== 'undefined' && typeof Swal.fire === 'function') {
      Swal.fire({ icon: type, title: message, timer: 2800, showConfirmButton: false });
      return;
    }
    const prefix = `[StockQuiebre] ${message}`;
    if (type === 'error') {
      console.error(prefix);
    } else if (type === 'warning') {
      console.warn(prefix);
    } else {
      console.info(prefix);
    }
  };

  let dataTable = null;
  let currentRows = [];
  let latestLimit = 0;

  $(document).ready(function () {
    initTable();
    bindEvents();
    loadFilters();
    loadData();
  });

  function initTable() {
    dataTable = $('#tblQuiebre').DataTable({
      data: [],
      columns: [
        { data: 'cliente', title: 'Cliente', defaultContent: '-' },
        { data: 'operativa', title: 'Operativa', defaultContent: '-' },
        { data: 'sku', title: 'SKU', defaultContent: '-' },
        { data: 'denominacion', title: 'Producto', defaultContent: '-' },
        {
          data: 'minimo',
          title: 'Mínimo',
          className: 'text-end',
          render: (value) => formatNumber(value),
        },
        {
          data: 'stock',
          title: 'Stock',
          className: 'text-end',
          render: (value) => formatNumber(value),
        },
        {
          data: 'reservados',
          title: 'Reservados',
          className: 'text-end',
          render: (value) => formatNumber(value),
        },
        {
          data: 'disponibles',
          title: 'Disponibles',
          className: 'text-end',
          render: (value) => formatNumber(value),
        },
        {
          data: 'faltante_min',
          title: 'Faltante',
          className: 'text-end text-danger',
          render: (value) => formatNumber(value),
        },
        {
          data: 'cobertura_pct',
          title: 'Cobertura',
          className: 'text-end',
          render: (value) => formatPercent(value),
        },
      ],
      order: [[8, 'desc']],
      searching: false,
      lengthMenu: [25, 50, 100, 250],
      pageLength: 25,
      language: {
        url: 'https://cdn.datatables.net/plug-ins/1.13.7/i18n/es-ES.json',
      },
    });
  }

  function bindEvents() {
    $('#frmStockQuiebre').on('submit', function (event) {
      event.preventDefault();
      loadData();
    });

    $('#btnLimpiar').on('click', function () {
      const form = document.getElementById('frmStockQuiebre');
      if (form) {
        form.reset();
      }
      $('#cliente_id').val('');
      $('#operativa_id').val('');
      $('#criterio').val('disponibles');
      $('#solo_con_min').prop('checked', true);
      loadData();
    });

    $('#btnExportPdf').on('click', function () {
      exportPdf();
    });
  }

  function loadFilters() {
    $.getJSON(joinUrl('api/reportes/stock_quiebre.php'), { meta: 'filters' })
      .done(function (response) {
        if (!response || response.ok === false) {
          notify('warning', response?.error || 'No se pudieron cargar los filtros');
          return;
        }

        if (Array.isArray(response.clientes)) {
          const $cliente = $('#cliente_id');
          $cliente.find('option:not(:first)').remove();
          response.clientes.forEach(function (item) {
            const label = item.razon_social ? `${item.razon_social}` : `Cliente ${item.id}`;
            $cliente.append(new Option(label, item.id));
          });
        }

        if (Array.isArray(response.operativas)) {
          const $operativa = $('#operativa_id');
          $operativa.find('option:not(:first)').remove();
          response.operativas.forEach(function (item) {
            const label = item.nombre ? `${item.nombre}` : `Operativa ${item.id}`;
            $operativa.append(new Option(label, item.id));
          });
        }
      })
      .fail(function () {
        notify('error', 'Error de conexión al cargar filtros');
      });
  }

  function loadData() {
    const $tbody = $('#tblQuiebre tbody');
    $tbody.html(`
      <tr class="text-muted">
        <td colspan="10" class="text-center py-4">
          <div class="spinner-border spinner-border-sm me-2" role="status"></div>
          Calculando quiebres...
        </td>
      </tr>
    `);

    const params = Object.fromEntries(new FormData(document.getElementById('frmStockQuiebre')).entries());

    $.ajax({
      url: joinUrl('api/reportes/stock_quiebre.php'),
      method: 'GET',
      data: params,
      dataType: 'json',
    })
      .done(function (response) {
        if (!response || response.ok === false) {
          notify('error', response?.error || 'No se pudo obtener el reporte');
          resetView();
          return;
        }

        currentRows = Array.isArray(response.rows) ? response.rows : [];
        latestLimit = Number(response.limit ?? 0);

        if (dataTable) {
          dataTable.clear();
          dataTable.rows.add(currentRows);
          dataTable.draw();
        }

        updateSummary(response.totals || {});
        updateLimitNotice(Boolean(response.truncated));
        $('#btnExportPdf').prop('disabled', currentRows.length === 0);
      })
      .fail(function () {
        notify('error', 'Error de conexión con el servidor');
        resetView();
      });
  }

  function resetView() {
    currentRows = [];
    if (dataTable) {
      dataTable.clear().draw();
    }
    updateSummary({});
    updateLimitNotice(false);
    $('#btnExportPdf').prop('disabled', true);
  }

  function updateSummary(totals) {
    const productos = Number(totals.productos ?? currentRows.length ?? 0);
    const faltante = totals.faltante_total ?? 0;
    const disponibles = totals.disponibles_total ?? 0;
    const cobertura = totals.cobertura_promedio;

    $('#metricProductos').text(formatNumber(productos));
    $('#metricFaltante').text(formatNumber(faltante));
    $('#metricDisponibles').text(formatNumber(disponibles));
    $('#metricCobertura').text(formatPercent(cobertura));
    $('#badgeTotalProductos').text(formatNumber(productos));
  }

  function updateLimitNotice(truncated) {
    if (truncated && latestLimit > 0) {
      $('#limitValue').text(formatNumber(latestLimit));
      $('#limitNotice').show();
    } else {
      $('#limitNotice').hide();
    }
  }

  function exportPdf() {
    if (!currentRows.length) {
      notify('warning', 'No hay datos para exportar');
      return;
    }

    const params = new URLSearchParams(new FormData(document.getElementById('frmStockQuiebre')));
    const url = joinUrl('reportes/stock_quiebre/pdf');
    const targetUrl = params.toString() ? `${url}?${params.toString()}` : url;
    window.open(targetUrl, '_blank', 'noopener');
  }
})(window, jQuery);
