/**
 * SOL · JavaScript · Reporte Trazabilidad de Productos
 */

(function(window, $) {
  'use strict';

  if (!$ || typeof $.ajax !== 'function') {
    console.error('[Reporte Trazabilidad] jQuery es requerido.');
    return;
  }

  const baseUrl = document.querySelector('meta[name="base-url"]')?.getAttribute('content') ||
                  (typeof window.BASE_URL !== 'undefined' ? window.BASE_URL : '/');

  function joinUrl(path) {
    return baseUrl.replace(/\/$/, '') + '/' + String(path || '').replace(/^\//, '');
  }

  const intFormatter = new Intl.NumberFormat('es-AR', { minimumFractionDigits: 0, maximumFractionDigits: 0 });
  const decimalFormatter = new Intl.NumberFormat('es-AR', { minimumFractionDigits: 2, maximumFractionDigits: 2 });

  function formatInt(value, fallback) {
    const numeric = Number(value);
    if (Number.isFinite(numeric)) {
      return intFormatter.format(numeric);
    }
    return fallback !== undefined ? fallback : '0';
  }

  function formatDecimal(value, fallback) {
    const numeric = Number(value);
    if (Number.isFinite(numeric)) {
      return decimalFormatter.format(numeric);
    }
    return fallback !== undefined ? fallback : '0,00';
  }

  function formatDate(value) {
    if (!value) { return '—'; }
    const str = String(value).slice(0, 10);
    const parts = str.split('-');
    if (parts.length !== 3) { return str; }
    return parts[2] + '/' + parts[1] + '/' + parts[0];
  }

  function escapeHtml(value) {
    return String(value)
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;')
      .replace(/'/g, '&#39;');
  }

  function buildBadge(label, tone) {
    let cls = 'bg-secondary';
    if (tone === 'match') { cls = 'bg-primary'; }
    if (tone === 'warn') { cls = 'bg-warning text-dark'; }
    return '<span class="badge ' + cls + ' me-1">' + escapeHtml(label) + '</span>';
  }

  function formatDateTime(value) {
    if (!value) { return '—'; }
    const normalized = String(value).replace('T', ' ');
    const date = normalized.slice(0, 10);
    const time = normalized.length >= 16 ? normalized.slice(11, 19) : '';
    return formatDate(date) + (time ? ' ' + time : '');
  }

  function formatNullable(value) {
    if (!value) { return '<span class="text-muted">—</span>'; }
    return escapeHtml(value);
  }

  function debounce(fn, delay) {
    let timer = null;
    return function() {
      const args = arguments;
      const ctx = this;
      clearTimeout(timer);
      timer = setTimeout(function() { fn.apply(ctx, args); }, delay);
    };
  }

  let tblPallets = null;
  let tblMovimientos = null;
  let currentData = {
    pallets: [],
    movements: [],
    summary: null,
    filters: null,
  };

  $(document).ready(function() {
    setDefaultDateRange();
    initTables();
    bindEvents();
  });

  function setDefaultDateRange() {
    const today = new Date();
    const past = new Date(today.getTime() - (90 * 24 * 60 * 60 * 1000));
    $('#fecha_hasta').val(today.toISOString().slice(0, 10));
    $('#fecha_desde').val(past.toISOString().slice(0, 10));
  }

  function bindEvents() {
    $('#frmTrazabilidad').on('submit', function(evt) {
      evt.preventDefault();
      fetchData();
    });

    $('#btnLimpiar').on('click', function() {
      const form = document.getElementById('frmTrazabilidad');
      if (form) {
        form.reset();
      }
      setDefaultDateRange();
      clearResults();
    });

    $('#sku').on('input', debounce(function() {
      const query = String(this.value || '').trim();
      if (query.length < 2) {
        updateSkuSuggestions([]);
        return;
      }
      fetchSkuSuggestions(query);
    }, 280));
  }

  function initTables() {
    tblPallets = $('#tblTrazabilidadPallets').DataTable({
      language: { url: 'https://cdn.datatables.net/plug-ins/1.13.7/i18n/es-ES.json' },
      responsive: true,
      pageLength: 10,
      columns: [
        {
          data: null,
          title: 'Pallet',
          render: function(row) {
            if (!row) { return '—'; }
            const estado = row.estado ? buildBadge(row.estado, row.estado === 'LIBRE' ? 'match' : 'warn') : '';
            const created = row.created_at ? '<div class="small text-muted">Creado: ' + formatDateTime(row.created_at) + '</div>' : '';
            return (
              '<div class="fw-semibold">' + escapeHtml(row.codigo || 'Sin código') + '</div>' +
              estado + created
            );
          }
        },
        {
          data: null,
          title: 'Productos / Lotes',
          render: function(row) {
            if (!row || !Array.isArray(row.items_summary) || row.items_summary.length === 0) {
              return '<span class="text-muted">Sin items asociados</span>';
            }
            return '<ul class="ps-3 mb-0">' + row.items_summary.map(function(text) {
              return '<li>' + escapeHtml(text) + '</li>';
            }).join('') + '</ul>';
          }
        },
        {
          data: null,
          title: 'Stock',
          className: 'text-end',
          render: function(row) {
            if (!row) { return '—'; }
            const uv = Number(row.uv_total || 0);
            const uc = Number(row.uc_total || 0);
            const uvLabel = uv ? formatInt(uv) + ' cajas' : null;
            const ucLabel = uc ? formatInt(uc) + ' UC' : null;
            const parts = [];
            if (uvLabel) { parts.push(uvLabel); }
            if (ucLabel) { parts.push(ucLabel); }
            return parts.length ? parts.join('<br>') : '<span class="text-muted">Sin stock</span>';
          }
        },
        {
          data: null,
          title: 'Ubicación actual',
          render: function(row) {
            if (!row) { return '—'; }
            const depo = row.deposito_label ? '<div>' + escapeHtml(row.deposito_label) + '</div>' : '';
            const pos = row.position_label ? '<div>' + escapeHtml(row.position_label) + '</div>' : '<div class="text-muted">Sin posición</div>';
            const occ = row.ocupando_desde ? '<div class="small text-muted">Desde: ' + formatDateTime(row.ocupando_desde) + '</div>' : '';
            return depo + pos + occ;
          }
        },
        {
          data: null,
          title: 'Coincidencias',
          render: function(row) {
            if (!row) { return '—'; }
            if (!row.match_summary) {
              return '<span class="text-muted">Sin coincidencias directas</span>';
            }
            return row.match_summary.split(',').map(function(label) {
              return buildBadge(label.trim(), 'match');
            }).join('');
          }
        }
      ]
    });

    tblMovimientos = $('#tblTrazabilidadMovimientos').DataTable({
      language: { url: 'https://cdn.datatables.net/plug-ins/1.13.7/i18n/es-ES.json' },
      responsive: true,
      pageLength: 20,
      order: [[0, 'desc']],
      columns: [
        {
          data: null,
          title: 'Fecha',
          render: function(row) {
            return row && row.fecha_ref_label ? escapeHtml(row.fecha_ref_label) : '<span class="text-muted">—</span>';
          }
        },
        {
          data: null,
          title: 'Pallet',
          render: function(row) {
            if (!row) { return '—'; }
            return '<div class="fw-semibold">' + escapeHtml(row.pallet_codigo || '') + '</div>';
          }
        },
        {
          data: null,
          title: 'Movimiento',
          render: function(row) {
            if (!row) { return '—'; }
            const tipo = row.tipo ? '<div class="fw-semibold">' + escapeHtml(row.tipo) + '</div>' : '';
            const motivo = row.motivo ? '<div class="small text-muted">' + escapeHtml(row.motivo) + '</div>' : '';
            return tipo + motivo;
          }
        },
        {
          data: null,
          title: 'Origen',
          render: function(row) {
            if (!row) { return '—'; }
            return formatNullable(row.desde_label);
          }
        },
        {
          data: null,
          title: 'Destino',
          render: function(row) {
            if (!row) { return '—'; }
            return formatNullable(row.hasta_label);
          }
        },
        {
          data: null,
          title: 'Detalle',
          render: function(row) {
            if (!row) { return '—'; }
            const items = row.detalle_items ? '<div>' + escapeHtml(row.detalle_items) + '</div>' : '';
            const qty = row.cantidad_label ? '<div class="small text-muted">' + escapeHtml(row.cantidad_label) + '</div>' : '';
            const dur = row.duracion_label ? '<div class="small text-muted">Duración: ' + escapeHtml(row.duracion_label) + '</div>' : '';
            return items || qty || dur ? items + qty + dur : '<span class="text-muted">Sin detalle</span>';
          }
        },
        {
          data: null,
          title: 'Referencia',
          render: function(row) {
            if (!row) { return '—'; }
            return row.ref_label ? escapeHtml(row.ref_label) : '<span class="text-muted">—</span>';
          }
        },
        {
          data: null,
          title: 'Responsables',
          render: function(row) {
            if (!row) { return '—'; }
            const sol = row.solicitado_label ? '<div><span class="text-muted">Solicitado:</span> ' + escapeHtml(row.solicitado_label) + '</div>' : '';
            const asg = row.asignado_label ? '<div><span class="text-muted">Asignado:</span> ' + escapeHtml(row.asignado_label) + '</div>' : '';
            const eje = row.ejecutado_label ? '<div><span class="text-muted">Ejecutado:</span> ' + escapeHtml(row.ejecutado_label) + '</div>' : '';
            return (sol + asg + eje) || '<span class="text-muted">—</span>';
          }
        }
      ]
    });
  }

  function fetchData() {
    const params = $('#frmTrazabilidad').serialize();
     const $btn = $('#btnBuscar');
     const original = $btn.data('original-label') || $btn.html();
     $btn.data('original-label', original);
     $btn.prop('disabled', true).html('<span class="spinner-border spinner-border-sm me-2" role="status" aria-hidden="true"></span>Buscando');

    $.ajax({
      url: joinUrl('api/reportes/trazabilidad_productos.php'),
      method: 'GET',
      data: params,
      dataType: 'json'
    })
      .done(function(response) {
        if (!response || !response.ok) {
          const msg = response && response.error ? response.error : 'No se pudo obtener la trazabilidad.';
          notify('error', 'Reporte de trazabilidad', msg);
          clearResults();
          return;
        }
        currentData = {
          pallets: Array.isArray(response.pallets) ? response.pallets : [],
          movements: Array.isArray(response.movements) ? response.movements : [],
          summary: response.summary || null,
          filters: response.filters || null,
        };
        renderResults();
      })
      .fail(function(xhr) {
        const status = xhr?.status || 0;
        let message = 'Se produjo un error al obtener la trazabilidad.';
        if (status === 422 && xhr.responseJSON?.error) {
          message = xhr.responseJSON.error;
        } else if (status === 401) {
          message = 'Sesión expirada. Refresque la página.';
        }
        notify('error', 'Reporte de trazabilidad', message);
        clearResults();
      })
      .always(function() {
        $btn.prop('disabled', false).html($btn.data('original-label') || 'Buscar');
      });
  }

  function notify(type, title, text) {
    if (typeof Swal !== 'undefined') {
      Swal.fire({ icon: type, title, text, timer: 3200, showConfirmButton: false });
      return;
    }
    const msg = '[' + title + '] ' + text;
    if (type === 'error') {
      console.error(msg);
    } else if (type === 'warning') {
      console.warn(msg);
    } else {
      console.info(msg);
    }
  }

  function clearResults() {
    currentData = { pallets: [], movements: [], summary: null, filters: null };
    renderResults();
  }

  function renderResults() {
    renderSummary();
    renderPallets();
    renderMovements();
    renderChips();
    updateEmptyState();
  }

  function renderSummary() {
    const summary = currentData.summary;
    if (!summary) {
      $('#cardResumen').hide();
      $('#summaryRange').text('');
      $('#sumPallets, #sumItems, #sumMovimientos, #sumStockUc, #sumStockUv, #sumFirstMove, #sumLastMove').text('-');
      return;
    }

    $('#sumPallets').text(formatInt(summary.total_pallets || 0, '0'));
    $('#sumItems').text(formatInt(summary.total_items || 0, '0'));
    $('#sumMovimientos').text(formatInt(summary.total_movimientos || 0, '0'));
    $('#sumStockUc').text(formatInt(summary.uc_total || 0) + ' UC');
    $('#sumStockUv').text('≈ ' + formatInt(summary.uv_total || 0) + ' cajas');
    $('#sumFirstMove').text(summary.primer_movimiento || '—');
    $('#sumLastMove').text(summary.ultimo_movimiento || '—');
    $('#summaryRange').text(summary.range_label || '');

    $('#cardResumen').show();
  }

  function renderPallets() {
    if (!tblPallets) { return; }
    tblPallets.clear();
    if (currentData.pallets.length) {
      tblPallets.rows.add(currentData.pallets);
    }
    tblPallets.draw();
    $('#badgePallets').text(formatInt(currentData.pallets.length));
  }

  function renderMovements() {
    if (!tblMovimientos) { return; }
    tblMovimientos.clear();
    if (currentData.movements.length) {
      tblMovimientos.rows.add(currentData.movements);
    }
    tblMovimientos.draw();
    $('#badgeMovimientos').text(formatInt(currentData.movements.length));
  }

  function renderChips() {
    const summary = currentData.summary;
    const $wrapper = $('#matchChips');
    const $container = $('#chipsContainer');
    if (!summary) {
      $wrapper.hide();
      $container.empty();
      return;
    }

    const chips = [];
    (summary.match_productos || []).forEach(function(prod) {
      if (!prod) { return; }
      const parts = [];
      if (prod.sku) { parts.push(prod.sku); }
      if (prod.nombre) { parts.push(prod.nombre); }
      const label = parts.join(' · ');
      chips.push('<span class="badge bg-info text-dark">' + escapeHtml(label) + '</span>');
    });
    (summary.match_lotes || []).forEach(function(lote) {
      chips.push('<span class="badge bg-secondary">Lote ' + escapeHtml(lote) + '</span>');
    });
    (summary.match_pallets || []).forEach(function(pallet) {
      chips.push('<span class="badge bg-primary">Pallet ' + escapeHtml(pallet) + '</span>');
    });

    if (!chips.length) {
      $wrapper.hide();
      $container.empty();
      return;
    }

    $container.html(chips.join(''));
    $wrapper.show();
  }

  function updateEmptyState() {
    const hasData = currentData.pallets.length > 0 || currentData.movements.length > 0;
    const executed = currentData.filters !== null;
    $('#alertSinResultados').toggle(executed && !hasData);
  }

  function fetchSkuSuggestions(query) {
    $.ajax({
      url: joinUrl('api/reportes/trazabilidad_productos.php'),
      method: 'GET',
      data: { meta: 'productos', q: query },
      dataType: 'json'
    })
      .done(function(response) {
        if (!response || !response.ok) {
          updateSkuSuggestions([]);
          return;
        }
        const list = Array.isArray(response.productos) ? response.productos : [];
        updateSkuSuggestions(list.map(function(item) {
          const sku = item.sku || '';
          const name = item.nombre ? ' · ' + item.nombre : '';
          return sku + name;
        }));
      })
      .fail(function() {
        updateSkuSuggestions([]);
      });
  }

  function updateSkuSuggestions(options) {
    const datalist = document.getElementById('skuSuggestions');
    if (!datalist) { return; }
    datalist.innerHTML = '';
    options.forEach(function(opt) {
      const option = document.createElement('option');
      option.value = opt;
      datalist.appendChild(option);
    });
  }

})(window, jQuery);
