// public/page-scripts/warehouse/layout-core.js
(function (window, $) {
  "use strict";

  const WL = (window.WarehouseLayout = window.WarehouseLayout || {});

  const BASE = (typeof window !== "undefined" && window.BASE_URL) ? window.BASE_URL : "/";
  function joinUrl(p) {
    return (BASE.endsWith("/") ? BASE : BASE + "/") + String(p || "").replace(/^\/+/, "");
  }

  const storage = {
    get(k, def = null) {
      try { const v = localStorage.getItem(k); return v === null ? def : v; } catch { return def; }
    },
    set(k, v) { try { localStorage.setItem(k, v); } catch {} },
  };

  function natCmp(a, b) {
    return String(a).localeCompare(String(b), undefined, { numeric: true, sensitivity: "base" });
  }

  function defNotify(icon, title, text) {
    if (typeof window.Swal === "undefined") {
      console[icon === "error" ? "error" : "log"](`${title}: ${text}`);
      alert(`${title}\n\n${text}`);
      return;
    }
    window.Swal.fire({ icon, title, text, confirmButtonText: "OK" });
  }

  WL.utils = { joinUrl, storage, natCmp };

  const RENDERERS = Object.create(null);
  WL.registerRenderer = function (name, fn) {
    if (!name || typeof fn !== "function") return;
    RENDERERS[String(name)] = fn;
  };

  // ---------------------------------------------------------------------------
  // API factory
  // ---------------------------------------------------------------------------
  function makeApi(apiUrl) {
    function qsAppend(base, paramsObj) {
      const parts = [];
      for (const k in paramsObj || {}) {
        if (paramsObj[k] === undefined || paramsObj[k] === null || paramsObj[k] === "") continue;
        parts.push(encodeURIComponent(k) + "=" + encodeURIComponent(paramsObj[k]));
      }
      return base + (parts.length ? (base.includes("?") ? "&" : "?") + parts.join("&") : "");
    }
    return {
      loadDepositos() {
        return $.getJSON(joinUrl(`${apiUrl}?meta=depositos`));
      },
      loadRacks(depId) {
        return $.getJSON(joinUrl(`${apiUrl}?meta=racks&deposito_id=${encodeURIComponent(depId)}`));
      },
      // ⬇️ ahora acepta extraParams para incluir 'fecha' u otros
      fetchPositions(depId, rackId, onlyActive = 0, extraParams = null) {
        const base = `${apiUrl}?deposito_id=${encodeURIComponent(depId)}&rack=${encodeURIComponent(rackId)}&only_active=${onlyActive ? 1 : 0}`;
        return $.getJSON(joinUrl(qsAppend(base, extraParams || {})));
      },
    };
  }

  // ---------------------------------------------------------------------------
  // Instancia
  // ---------------------------------------------------------------------------
  function Instance($root, options) {
    this.$root = $root;
    this.$canvas = $root.find(".warehouse-layout__canvas");
    this.$toolbar = $root.find(".d-flex.align-items-center.justify-content-between.mb-3").first();
    this.$toolbarRight = this.$toolbar.find(".d-flex.align-items-center.gap-2").last();

    this.opts = $.extend(true, {
      apiUrl: "api/control/conteo_layout.php",
      defaultMode: "microtiles",
      enableLegend: true,     // inyección de leyenda interna (si injectControls=true)
      injectControls: true,   // inyección de toolbar interna
      // Controles externos (si injectControls=false):
      modeSelector: null,     // ej: "#flt-view"
      depositoSelector: null, // ej: "#flt-deposito"
      rackSelector: null,     // ej: "#flt-rack"
      formSelector: null,     // ej: "#frmConteo" (submit => _reload)
      queryParams: null,      // ⬅️ parámetros extra (e.g., {fecha: '2025-10-03'})
      storagePrefix: "wl@" + ((window.location && window.location.pathname) || "/"),
      notify: defNotify,
    }, options || {});

    this.API = makeApi(this.opts.apiUrl);
    this.keys = {
      MODE: `${this.opts.storagePrefix}:mode`,
      DEP:  `${this.opts.storagePrefix}:deposito`,
      RACK: `${this.opts.storagePrefix}:rack`,
    };

    // Selector de modo (interno o externo)
    this.$modeSel = this.opts.modeSelector ? $(this.opts.modeSelector) : $root.find("select[data-role='layout-mode']");

    // Estado
    this.currentFilter  = null;
    this.lastMeta       = null;
    this.lastPositions  = [];

    // Canvas garantizado
    this._ensureCanvas();

    // Controles: internos o externos
    if (this.opts.injectControls) {
      this._injectControls();
      if (this.opts.enableLegend) this._injectLegend();
    } else {
      // usar controles externos
      this.$selDep    = $(this.opts.depositoSelector || []);
      this.$selRack   = $(this.opts.rackSelector || []);
      this.$btnReload = $(); // no obligatorio si hay form externo
      this.$spin      = $(); // no spinner interno
      if (!this.$selDep.length || !this.$selRack.length) {
        this.opts.notify("error", "Controles", "Faltan selectores externos de Depósito/Rack en layout-core.js");
      }
    }

    this._restoreState();
    this._bindEvents();
    this._firstLoad();
  }

  Instance.prototype._ensureCanvas = function () {
    if (this.$canvas.length === 0) {
      this.$canvas = $('<div class="warehouse-layout__canvas"></div>');
      this.$root.append(this.$canvas);
    }
  };

  Instance.prototype._controlsHtml = function () {
    return `
      <div class="d-flex align-items-center gap-2" data-role="wl-filters">
        <select class="form-select form-select-sm" name="deposito_id" style="min-width:160px">
          <option value="">Depósito...</option>
        </select>
        <select class="form-select form-select-sm" name="rack" style="min-width:140px" disabled>
          <option value="">Rack...</option>
        </select>
        <button type="button" class="btn btn-sm btn-primary" name="btnReload" disabled>Cargar</button>
        <div class="spinner-border spinner-border-sm text-secondary ms-1 d-none" role="status" aria-hidden="true"></div>
      </div>
    `;
  };

  Instance.prototype._legendHtml = function () {
    return `
      <div class="d-flex align-items-center flex-wrap gap-2 small mb-2" data-role="wl-legend">
        <span class="text-muted me-1">Leyenda:</span>
        <button type="button" class="btn btn-outline-success btn-sm" data-filter="state:ACTIVE">Activo</button>
        <button type="button" class="btn btn-outline-secondary btn-sm" data-filter="state:INACTIVE">Inactivo</button>
        <button type="button" class="btn btn-outline-danger btn-sm" data-filter="state:BLOCKED">Bloqueado</button>
        <button type="button" class="btn btn-outline-primary btn-sm" data-filter="pick:1">Pick-face</button>
        <button type="button" class="btn btn-light btn-sm" data-filter="clear">Limpiar</button>
      </div>
    `;
  };

  Instance.prototype._injectControls = function () {
    const $ctl = $(this._controlsHtml());
    if (this.$toolbarRight.length) this.$toolbarRight.append($ctl);
    else this.$toolbar.append($ctl);

    this.$ctl     = $ctl;
    this.$selDep  = $ctl.find('select[name="deposito_id"]');
    this.$selRack = $ctl.find('select[name="rack"]');
    this.$btnReload = $ctl.find('button[name="btnReload"]');
    this.$spin    = $ctl.find(".spinner-border");
  };

  Instance.prototype._injectLegend = function () {
    const $legend = $(this._legendHtml());
    this.$toolbar.after($legend);
    this.$legend = $legend;
    if (window.bootstrap && bootstrap.Tooltip) {
      $legend.find("[data-bs-toggle='tooltip']").each(function () {
        new bootstrap.Tooltip(this);
      });
    }
  };

  Instance.prototype._restoreState = function () {
    const savedMode = storage.get(this.keys.MODE, null);
    if (this.$modeSel && this.$modeSel.length) {
      if (savedMode) {
        this.$modeSel.val(savedMode);
        this.$root.attr("data-mode", savedMode);
      } else if (this.$modeSel.val() === null) {
        this.$modeSel.val(this.opts.defaultMode);
        this.$root.attr("data-mode", this.opts.defaultMode);
      }
    }
    this._savedDep  = storage.get(this.keys.DEP, "");
    this._savedRack = storage.get(this.keys.RACK, "");
  };

  Instance.prototype._bindEvents = function () {
    const self = this;

    // Selects (internos o externos)
    this.$selDep.on("change", function () {
      const depId = $(this).val();
      storage.set(self.keys.DEP, String(depId || ""));
      self._loadRacks(depId, null);
    });

    this.$selRack.on("change", function () {
      const rackId = $(this).val();
      storage.set(self.keys.RACK, String(rackId || ""));
      if (self.$btnReload && self.$btnReload.length) {
        self.$btnReload.prop("disabled", self.$selDep.val() === "" || rackId === "");
      }
    });

    // Botón interno (si existe)
    if (this.$btnReload && this.$btnReload.length) {
      this.$btnReload.on("click", function () { self._reload(); });
    }

    // Submit del form externo (si hay)
    if (this.opts.formSelector) {
      $(this.opts.formSelector).on("submit", function (ev) {
        ev.preventDefault();
        self._reload();
      });
    }

    if (this.$modeSel && this.$modeSel.length) {
      this.$modeSel.on("change", function () {
        const mode = $(this).val() || self.opts.defaultMode;
        storage.set(self.keys.MODE, mode);
        if (self.lastPositions && self.lastPositions.length) {
          self._render(mode, self.lastMeta, self.lastPositions);
        }
      });
    }

    if (this.$legend && this.$legend.length) {
      this.$legend.on("click", "button[data-filter]", function () {
        const filter = $(this).data("filter");
        self.$legend.find("button[data-filter]").removeClass("active");
        if (filter === "clear") {
          self.applyLegendFilter(null);
        } else {
          $(this).addClass("active");
          self.applyLegendFilter(String(filter));
        }
      });
    }
  };

  Instance.prototype._firstLoad = function () {
    const self = this;
    self._loadDepositos(self._savedDep)
      .then(() => self._loadRacks(self.$selDep.val(), self._savedRack))
      .then(() => {
        if (self.$selDep.val() && self.$selRack.val()) {
          if (self.$btnReload && self.$btnReload.length) self.$btnReload.prop("disabled", false);
          self._reload();
        }
      });
  };

  Instance.prototype._busy = function (on) {
    this.$spin && this.$spin.toggleClass && this.$spin.toggleClass("d-none", !on);
  };

  Instance.prototype._loadDepositos = function (selectedId) {
    const self = this;
    self._busy(true);
    return this.API.loadDepositos()
      .done((res) => {
        const list = (res && res.depositos) || [];
        self.$selDep.empty().append('<option value="">Depósito...</option>');
        list.forEach((d) => {
          const id = Number(d.id);
          const txt = d.code || "DEP" + id;
          self.$selDep.append(`<option value="${id}">${txt}</option>`);
        });
        if (selectedId) self.$selDep.val(String(selectedId));
        if (self.$btnReload && self.$btnReload.length) {
          self.$btnReload.prop("disabled", self.$selDep.val() === "" || self.$selRack.val() === "");
        }
      })
      .fail(() => this.opts.notify("error", "Depósitos", "No se pudieron cargar los depósitos."))
      .always(() => self._busy(false));
  };

  Instance.prototype._loadRacks = function (depId, selectedRack) {
    const self = this;
    if (!depId) {
      self.$selRack.prop("disabled", true).empty().append('<option value="">Rack...</option>');
      if (self.$btnReload && self.$btnReload.length) self.$btnReload.prop("disabled", true);
      return $.Deferred().resolve().promise();
    }
    self._busy(true);
    return this.API.loadRacks(depId)
      .done((res) => {
        const list = (res && res.racks) || [];
        self.$selRack.empty().append('<option value="">Rack...</option>');
        list.forEach((r) => {
          const val = String(r.id);
          const txt = r.code || "R" + String(r.id).padStart(2, "0");
          self.$selRack.append(`<option value="${val}">${txt}</option>`);
        });
        if (selectedRack) self.$selRack.val(String(selectedRack));
        self.$selRack.prop("disabled", self.$selDep.val() === "");
        if (self.$btnReload && self.$btnReload.length) {
          self.$btnReload.prop("disabled", self.$selDep.val() === "" || self.$selRack.val() === "");
        }
      })
      .fail(() => this.opts.notify("error", "Racks", "No se pudieron cargar los racks."))
      .always(() => self._busy(false));
  };

  // ---------------------------------------------------------------------------
  // Recarga de posiciones (ahora con soporte de 'fecha' / extraParams)
  // ---------------------------------------------------------------------------
  Instance.prototype._reload = function () {
    const depId = this.$selDep.val();
    const rackId = this.$selRack.val();
    if (!depId || !rackId) return;

    // ⬇️ construir extraParams: 1) options.queryParams; 2) formSelector → name="fecha"
    const extra = $.extend({}, this.opts.queryParams || {});
    if (this.opts.formSelector) {
      const $form = $(this.opts.formSelector);
      const fechaVal = $form.find("[name='fecha']").val();
      if (fechaVal) extra.fecha = fechaVal;
    }

    const self = this;
    self._busy(true);
    this.API.fetchPositions(depId, rackId, 0, extra)
      .done((res) => {
        if (!res || res.ok !== true) {
          this.opts.notify("error", "Carga", (res && (res.error || "Respuesta inválida")) || "Error de datos");
          return;
        }
        self.lastMeta = { deposito: res.deposito, rack: res.rack, date: res.date || null };
        self.lastPositions = res.positions || [];
        const mode = (self.$modeSel && self.$modeSel.length ? self.$modeSel.val() : self.opts.defaultMode) || self.opts.defaultMode;
        self._render(mode, self.lastMeta, self.lastPositions);
      })
      .fail((jq) => {
        const j = jq.responseJSON || {};
        this.opts.notify("error", "Carga", j.error || "No se pudo cargar el layout.");
      })
      .always(() => self._busy(false));
  };

  // ---------------------------------------------------------------------------
  // Modelo → Renderer
  // ---------------------------------------------------------------------------
  Instance.prototype._buildModel = function (positions) {
    const colsSet = {}, nivSet = {}, byKey = {};
    (positions || []).forEach((p) => {
      const col = String(p.col_code || ""), niv = String(p.niv_code || ""), f = Number(p.depth_index || 0);
      if (!col || !niv || f <= 0) return;
      colsSet[col] = true;
      nivSet[niv] = true;
      const k = col + "#" + niv;
      if (!byKey[k]) byKey[k] = { col_code: col, niv_code: niv, fondos: [] };
      byKey[k].fondos.push({ f, pos: p });
    });
    const cols = Object.keys(colsSet).sort(natCmp);
    const nivs = Object.keys(nivSet).sort(natCmp).reverse();
    Object.values(byKey).forEach((cell) => cell.fondos.sort((a, b) => a.f - b.f));
    return { cols, nivs, byKey };
  };

  Instance.prototype._render = function (mode, meta, positions) {
    const $canvas = this.$canvas;
    const model = this._buildModel(positions);
    const renderer = RENDERERS[mode];
    if (typeof renderer === "function") {
      const helpers = {
        reapplyFilter: () => this.applyLegendFilter(this.currentFilter),
        joinUrl,
        natCmp,
        notify: this.opts.notify,
      };
      renderer($canvas, model, meta, helpers);
      if (this.currentFilter) this.applyLegendFilter(this.currentFilter);
    } else {
      $canvas.empty().append(`<div class="alert alert-warning mb-0">El modo <b>${mode}</b> no está implementado.</div>`);
    }
  };

  Instance.prototype.applyLegendFilter = function (filter) {
    this.currentFilter = filter || null;
    const $tiles = this.$canvas.find(".pos-tile");
    if ($tiles.length === 0) return;
    if (!this.currentFilter) {
      $tiles.removeClass("opacity-25 border-2 border-primary shadow");
      return;
    }
    const [type, valRaw] = String(this.currentFilter).split(":");
    const val = String(valRaw || "");
    $tiles.each(function () {
      const $t = $(this);
      let match = true;
      if (type === "state") {
        const want = val.toUpperCase();
        match =
          (want === "ACTIVE"   && $t.hasClass("pos-active"))   ||
          (want === "INACTIVE" && $t.hasClass("pos-inactive")) ||
          (want === "BLOCKED"  && $t.hasClass("pos-blocked"));
      } else if (type === "pick") {
        match = String($t.attr("data-pick") || "0") === val;
      }
      if (match) {
        $t.removeClass("opacity-25").addClass("border-2 border-primary shadow");
      } else {
        $t.removeClass("border-2 border-primary shadow").addClass("opacity-25");
      }
    });
  };

  // ---------------------------------------------------------------------------
  // Inicialización
  // ---------------------------------------------------------------------------
  WL.init = function init($roots, options) {
    if (!$roots || typeof $roots === "string") {
      $roots = $($roots || ".warehouse-layout");
    }
    const instances = [];
    $roots.each(function () {
      const $root = $(this);

      // 🔒 evitar doble init
      if ($root.data("wlInit") === 1) return;
      $root.data("wlInit", 1);

      instances.push(new Instance($root, options));
    });
    return instances.length === 1 ? instances[0] : instances;
  };
})(window, jQuery);
