// public/page-scripts/warehouse/render-column-cards.js
// ----------------------------------------------------------------------------
// Renderer: column_cards
// Disposición: una tarjeta por COLUMNA. Dentro, filas por NIVEL (de mayor a menor)
// y en cada fila una “tira” de fondos (F01, F02, ...).
//
// Reglas SCSS esperadas (según tu estructura):
//   .rack-column-cards   .column-card   .column-card-level   .column-card-cell
// Estados/clases para integrarse con el core/leyenda:
//   .pos-tile + (.pos-active | .pos-inactive | .pos-blocked) y data-pick="1"
// ----------------------------------------------------------------------------
(function (window, $) {
  "use strict";

  if (!window.WarehouseLayout) return;

  function cssState(status) {
    const s = String(status || "ACTIVE").toUpperCase();
    if (s === "BLOCKED")  return "pos-blocked";
    if (s === "INACTIVE") return "pos-inactive";
    return "pos-active";
  }

  function tileInfo(pos) {
    const lines = [
      `Ubicación: ${pos.code_full || "-"}`,
      `Estado: ${pos.status || "-"}`,
      `Pick face: ${Number(pos.is_pick_face) === 1 ? "Sí" : "No"}`,
      `ID fondo: ${pos.fondo_id || pos.id || "-"}`,
      `Col: ${pos.col_code || "-"}`,
      `Nivel: ${pos.niv_code || "-"}`,
      `Fondo: ${pos.depth_index || "-"}`,
    ];
    return lines.join("\n");
  }

  function renderHeader(meta) {
    const depCode  = (meta && meta.deposito && meta.deposito.code) || "";
    const rackCode = (meta && meta.rack && meta.rack.code) || "";
    return $(`
      <div class="d-flex justify-content-between align-items-center mb-2">
        <div class="small text-muted">
          <span class="me-2">Depósito: <span class="badge text-bg-light">${depCode || "-"}</span></span>
          <span>Rack: <span class="badge text-bg-secondary">${rackCode || "-"}</span></span>
        </div>
        <div class="small text-muted">Column cards</div>
      </div>
    `);
  }

  function buildColumnCard(colCode, nivs, byKey) {
    // Tarjeta de columna
    const $card = $(`
      <div class="column-card card shadow-sm h-100">
        <div class="card-header py-2 d-flex justify-content-between align-items-center">
          <div class="fw-semibold">${colCode}</div>
          <div class="small text-muted" data-role="counts"></div>
        </div>
        <div class="card-body p-2">
          <div class="d-flex flex-column gap-2" data-role="levels"></div>
        </div>
      </div>
    `);

    const $levels = $card.find('[data-role="levels"]');
    let totalFondos = 0;

    // Para cada nivel (de mayor a menor), dibujar la fila con sus fondos
    nivs.forEach(nv => {
      const k = colCode + "#" + nv;
      const cell = byKey[k] || { fondos: [] };

      const $row = $(`
        <div class="column-card-level">
          <div class="d-flex justify-content-between align-items-center mb-1">
            <div class="small text-muted fw-semibold">${nv}</div>
            <div class="small text-muted" data-role="lvl-count"></div>
          </div>
          <div class="d-flex flex-row flex-wrap gap-1 justify-content-start align-items-center" data-role="lvl-strip"></div>
        </div>
      `);

      const $strip = $row.find('[data-role="lvl-strip"]');

      if (!cell.fondos.length) {
        $strip.append('<div class="text-muted small">—</div>');
      } else {
        cell.fondos.forEach(({ f, pos }) => {
          const clsState = cssState(pos.status);
          const pick  = Number(pos.is_pick_face || 0) === 1;
          const title = tileInfo(pos);
          const label = "F" + String(f).padStart(2, "0");

          // Usamos clase común "pos-tile" para que el core aplique filtros
          const $tile = $(`<div class="pos-tile column-card-cell ${clsState}" title="${title}">${label}</div>`);
          if (pick) $tile.attr("data-pick", "1");
          $tile.data("pos", pos);
          $strip.append($tile);
        });
      }

      // Contador de fondos por nivel
      const lvlFondos = cell.fondos.length;
      totalFondos += lvlFondos;
      $row.find('[data-role="lvl-count"]').text(lvlFondos ? `${lvlFondos} fondo(s)` : `0`);

      $levels.append($row);
    });

    // Totales por columna (en header)
    $card.find('[data-role="counts"]').text(`${totalFondos} fondo(s)`);

    return $card;
  }

  window.WarehouseLayout.registerRenderer("column_cards", function ($canvas, model, meta, helpers) {
    const { cols, nivs, byKey } = model;

    // limpiar canvas
    $canvas.empty();

    // Envoltura para activar las reglas SCSS específicas
    const $wrap = $('<div class="rack-column-cards"></div>');

    if (!cols.length || !nivs.length) {
      $wrap.append(`
        <div class="alert alert-info mb-0">
          No hay posiciones para mostrar. Seleccioná otro rack o depósito.
        </div>
      `);
      $canvas.append($wrap);
      return;
    }

    // Header general
    $wrap.append(renderHeader(meta));

    // Grid de columnas (Bootstrap grid responsive)
    const $grid = $('<div class="row g-2"></div>');

    cols.forEach(colCode => {
      const $col = $('<div class="col-12 col-sm-6 col-md-4 col-lg-3"></div>');
      const $card = buildColumnCard(colCode, nivs, byKey);
      $col.append($card);
      $grid.append($col);
    });

    $wrap.append($grid);
    $canvas.append($wrap);

    // Click en tiles → detalle
    $canvas.off("click", ".pos-tile").on("click", ".pos-tile", function () {
      const pos = $(this).data("pos");
      if (!pos) return;
      if (helpers && typeof helpers.notify === "function") {
        helpers.notify("info", "Posición", tileInfo(pos));
      } else {
        alert(tileInfo(pos));
      }
    });

    // Re-aplicar filtro si existía
    if (helpers && typeof helpers.reapplyFilter === "function") {
      helpers.reapplyFilter();
    }
  });

})(window, jQuery);
