// public/page-scripts/warehouse/render-microtiles.js
// ----------------------------------------------------------------------------
// Renderer: microtiles
// Requisitos SCSS (ya en tu proyecto):
//   .rack-microtiles   .microtile   .microtile-cell
// Estados esperados por el core/leyenda:
//   .pos-active | .pos-inactive | .pos-blocked  y data-pick="1" para pick-face
// Marcado de conteo (nuevo):
//   .pos-counted  y data-counted="1"
// Atributos de integración con conteo.js (nuevo):
//   data-fondo-id, data-code-full
// ----------------------------------------------------------------------------
(function (window, $) {
  "use strict";

  if (!window.WarehouseLayout) return;

  function cssState(status) {
    const s = String(status || "ACTIVE").toUpperCase();
    if (s === "BLOCKED")  return "pos-blocked";
    if (s === "INACTIVE") return "pos-inactive";
    return "pos-active"; // ACTIVE
  }

  function tileInfo(pos) {
    const lines = [
      `Ubicación: ${pos.code_full || "-"}`,
      `Estado: ${pos.status || "-"}`,
      `Pick face: ${Number(pos.is_pick_face) === 1 ? "Sí" : "No"}`,
      `ID fondo: ${pos.fondo_id || pos.id || "-"}`,
      `Col: ${pos.col_code || "-"}`,
      `Nivel: ${pos.niv_code || "-"}`,
      `Fondo: ${pos.depth_index || "-"}`,
    ];
    return lines.join("\n");
  }

  // Registrar renderer
  window.WarehouseLayout.registerRenderer("microtiles", function ($canvas, model, meta, helpers) {
    const { cols, nivs, byKey } = model;

    // Limpia canvas
    $canvas.empty();

    // Envoltura para activar las reglas SCSS de microtiles
    const $wrap = $('<div class="rack-microtiles"></div>');

    if (!cols.length || !nivs.length) {
      $wrap.append(`
        <div class="alert alert-info mb-0">
          No hay posiciones para mostrar. Seleccioná otro rack o depósito.
        </div>
      `);
      $canvas.append($wrap);
      return;
    }

    const depCode  = (meta && meta.deposito && meta.deposito.code) || "";
    const rackCode = (meta && meta.rack && meta.rack.code) || "";

    const $info = $(`
      <div class="d-flex justify-content-between align-items-center mb-2">
        <div class="small text-muted">
          <span class="me-2">Depósito: <span class="badge text-bg-light">${depCode || "-"}</span></span>
          <span>Rack: <span class="badge text-bg-secondary">${rackCode || "-"}</span></span>
        </div>
        <div class="small text-muted">Microtiles</div>
      </div>
    `);

    const $table = $(`
      <table class="table table-sm table-bordered align-middle mb-0">
        <thead>
          <tr>
            <th style="width:120px">Nivel \\ Col</th>
          </tr>
        </thead>
        <tbody></tbody>
      </table>
    `);

    // Encabezado de columnas
    cols.forEach(c => $table.find("thead tr").append(`<th class="text-center">${c}</th>`));

    // Filas por nivel (de mayor a menor)
    nivs.forEach(nv => {
      const $tr = $(`<tr><th class="text-center">${nv}</th></tr>`);

      cols.forEach(c => {
        const k = c + "#" + nv;
        const cell = byKey[k] || { fondos: [] };

        const $td = $('<td class="p-0"></td>');
        const $micro = $('<div class="microtile"></div>');

        if (cell.fondos.length === 0) {
          $micro.append('<div class="text-muted small p-2">—</div>');
        } else {
          cell.fondos.forEach(({ f, pos }) => {
            const clsState = cssState(pos.status);
            const pick = Number(pos.is_pick_face || 0) === 1;
            const title = tileInfo(pos); // tooltip con info formateada
            const label = String(f).padStart(2, "0");

            // elemento base del tile
            const $tile = $(
              `<div class="pos-tile microtile-cell ${clsState}" title="${title}">${label}</div>`
            );

            // atributos para integración con conteo.js
            const fondoId = pos.fondo_id || pos.id || null;
            if (fondoId != null) $tile.attr("data-fondo-id", String(fondoId));
            if (pos.code_full)   $tile.attr("data-code-full", String(pos.code_full));
            if (pick)            $tile.attr("data-pick", "1");

            // marcar si ya fue contado en la fecha seleccionada
            if (pos.counted === true) {
              $tile.addClass("pos-counted").attr("data-counted", "1");
            }

            // guardar pos por si algún helper lo requiere
            $tile.data("pos", pos);

            $micro.append($tile);
          });
        }

        $td.append($micro);
        $tr.append($td);
      });

      $table.find("tbody").append($tr);
    });

    $wrap.append($info, $table);
    $canvas.append($wrap);

    // IMPORTANTE:
    // No enlazamos click interno (notify/alert) para no interferir con el flujo de conteo.
    // El click es interceptado por conteo.js (fase de captura) para abrir el modal.
    // Si quisieras reactivar un notify en modo "mapeo", podrías hacerlo acá,
    // chequeando helpers/mode y agregando un onClick condicional.

    // Reaplicar filtro si el core lo expuso
    if (helpers && typeof helpers.reapplyFilter === "function") {
      helpers.reapplyFilter();
    }
  });

})(window, jQuery);
