// public/page-scripts/warehouse/render-shelf-strips.js
// ----------------------------------------------------------------------------
// Renderer: shelf_strips
// Estructura visual: por cada celda (Col x Nivel) una "tira" horizontal de fondos
// Reglas SCSS esperadas (ya en tu proyecto):
//   .rack-shelf-strips   .shelf-strip   .shelf-strip-cell
// Estados/clases que usa el core para filtros:
//   .pos-tile + (.pos-active | .pos-inactive | .pos-blocked) y data-pick="1"
// ----------------------------------------------------------------------------
(function (window, $) {
  "use strict";

  if (!window.WarehouseLayout) return;

  function cssState(status) {
    const s = String(status || "ACTIVE").toUpperCase();
    if (s === "BLOCKED")  return "pos-blocked";
    if (s === "INACTIVE") return "pos-inactive";
    return "pos-active";
  }

  function tileInfo(pos) {
    const lines = [
      `Ubicación: ${pos.code_full || "-"}`,
      `Estado: ${pos.status || "-"}`,
      `Pick face: ${Number(pos.is_pick_face) === 1 ? "Sí" : "No"}`,
      `ID fondo: ${pos.fondo_id || pos.id || "-"}`,
      `Col: ${pos.col_code || "-"}`,
      `Nivel: ${pos.niv_code || "-"}`,
      `Fondo: ${pos.depth_index || "-"}`,
    ];
    return lines.join("\n");
  }

  window.WarehouseLayout.registerRenderer("shelf_strips", function ($canvas, model, meta, helpers) {
    const { cols, nivs, byKey } = model;

    $canvas.empty();

    const $wrap = $('<div class="rack-shelf-strips"></div>');

    if (!cols.length || !nivs.length) {
      $wrap.append(`
        <div class="alert alert-info mb-0">
          No hay posiciones para mostrar. Seleccioná otro rack o depósito.
        </div>
      `);
      $canvas.append($wrap);
      return;
    }

    const depCode  = (meta && meta.deposito && meta.deposito.code) || "";
    const rackCode = (meta && meta.rack && meta.rack.code) || "";

    const $info = $(`
      <div class="d-flex justify-content-between align-items-center mb-2">
        <div class="small text-muted">
          <span class="me-2">Depósito: <span class="badge text-bg-light">${depCode || "-"}</span></span>
          <span>Rack: <span class="badge text-bg-secondary">${rackCode || "-"}</span></span>
        </div>
        <div class="small text-muted">Shelf strips</div>
      </div>
    `);

    const $table = $(`
      <table class="table table-sm table-bordered align-middle mb-0">
        <thead>
          <tr>
            <th style="width:120px">Nivel \\ Col</th>
          </tr>
        </thead>
        <tbody></tbody>
      </table>
    `);

    // Encabezado de columnas
    cols.forEach(c => $table.find("thead tr").append(`<th class="text-center">${c}</th>`));

    // Filas por nivel (mayor arriba)
    nivs.forEach(nv => {
      const $tr = $(`<tr><th class="text-center">${nv}</th></tr>`);

      cols.forEach(c => {
        const k = c + "#" + nv;
        const cell = byKey[k] || { fondos: [] };

        const $td = $('<td class="p-1"></td>');
        const $strip = $('<div class="shelf-strip d-flex flex-row flex-wrap gap-1 justify-content-center align-items-center"></div>');

        if (cell.fondos.length === 0) {
          $strip.append('<div class="text-muted small px-2">—</div>');
        } else {
          cell.fondos.forEach(({ f, pos }) => {
            const clsState = cssState(pos.status);
            const pick  = Number(pos.is_pick_face || 0) === 1;
            const title = tileInfo(pos);
            const label = "F" + String(f).padStart(2, "0");

            // Clase común "pos-tile" para que el core pueda filtrar
            const $tile = $(`<div class="pos-tile shelf-strip-cell ${clsState}" title="${title}">${label}</div>`);
            if (pick) $tile.attr("data-pick", "1");
            $tile.data("pos", pos);
            $strip.append($tile);
          });
        }

        $td.append($strip);
        $tr.append($td);
      });

      $table.find("tbody").append($tr);
    });

    $wrap.append($info, $table);
    $canvas.append($wrap);

    // Click sobre cada fondo → detalle
    $canvas.off("click", ".pos-tile").on("click", ".pos-tile", function () {
      const pos = $(this).data("pos");
      if (!pos) return;
      if (helpers && typeof helpers.notify === "function") {
        helpers.notify("info", "Posición", tileInfo(pos));
      } else {
        alert(tileInfo(pos));
      }
    });

    // Re-aplicar filtro si existe
    if (helpers && typeof helpers.reapplyFilter === "function") {
      helpers.reapplyFilter();
    }
  });

})(window, jQuery);
