<?php
declare(strict_types=1);

// public/tools/layout_builder.php
// Simple admin tool to create/update rows in wh_posicion using ranges (racks/cols/niveles/fondos)

$ROOT = dirname(__DIR__, 1); // public
$BASE = dirname($ROOT);      // project root
require_once $BASE . '/config/db.php';
// Composer autoloader for App\LayoutBuilder used in tests and dry-run
if (file_exists($BASE . '/vendor/autoload.php')) {
  require_once $BASE . '/vendor/autoload.php';
}
if (file_exists($BASE . '/app/Support/ViewHelpers.php')) {
  require_once $BASE . '/app/Support/ViewHelpers.php';
}
if (file_exists($BASE . '/app/Support/Auth.php')) {
  require_once $BASE . '/app/Support/Auth.php';
}

session_start();
// require minimal auth - disable writes if no session
$loggedIn = !empty($_SESSION['usuario_id']);

try {
  $pdo = getPDO();
} catch (Throwable $e) {
  echo "<pre>DB error: " . htmlspecialchars($e->getMessage()) . "</pre>";
  exit;
}

$deps = $pdo->query('SELECT id, code, nombre FROM wh_deposito ORDER BY id')->fetchAll(PDO::FETCH_ASSOC);
$ambs = $pdo->query('SELECT id, code, nombre FROM wh_ambiente ORDER BY id')->fetchAll(PDO::FETCH_ASSOC);

// helper: check if a column exists in a table (safe for multiple calls)
if (!function_exists('columnExists')) {
  function columnExists(PDO $pdo, string $table, string $column): bool
  {
    static $cache = [];
    $key = $table . '.' . $column;
    if (array_key_exists($key, $cache))
      return $cache[$key];
    try {
      $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?");
      $st->execute([$table, $column]);
      $res = (int) $st->fetchColumn();
      $cache[$key] = $res > 0;
      return $cache[$key];
    } catch (Throwable $e) {
      $cache[$key] = false;
      return false;
    }
  }
}

// helper: get column data type (from INFORMATION_SCHEMA) — returns data_type or null
if (!function_exists('columnType')) {
  function columnType(PDO $pdo, string $table, string $column): ?string
  {
    try {
      $st = $pdo->prepare("SELECT DATA_TYPE FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ? LIMIT 1");
      $st->execute([$table, $column]);
      $res = $st->fetchColumn();
      return $res ?: null;
    } catch (Throwable $e) {
      return null;
    }
  }
}

// helper: check if a table exists in the current database
if (!function_exists('tableExists')) {
  function tableExists(PDO $pdo, string $table): bool
  {
    static $cache = [];
    if (array_key_exists($table, $cache))
      return $cache[$table];
    try {
      $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?");
      $st->execute([$table]);
      $res = (int) $st->fetchColumn();
      $cache[$table] = $res > 0;
      return $cache[$table];
    } catch (Throwable $e) {
      $cache[$table] = false;
      return false;
    }
  }
}

// helpers: convert between 1-based index and letters (A, B, ..., Z, AA, AB, ...)
if (!function_exists('indexToLetters')) {
  function indexToLetters(int $n): string
  {
    $s = '';
    while ($n > 0) {
      $n--;
      $s = chr(65 + ($n % 26)) . $s;
      $n = intdiv($n, 26);
    }
    return $s;
  }
}
if (!function_exists('lettersToIndex')) {
  function lettersToIndex(string $s): int
  {
    $s = strtoupper(trim($s));
    if ($s === '') return 0;
    $val = 0;
    $len = strlen($s);
    for ($i = 0; $i < $len; $i++) {
      $ch = ord($s[$i]);
      if ($ch < 65 || $ch > 90) return 0; // non A-Z
      $val = $val * 26 + ($ch - 64);
    }
    return $val;
  }
}

$feedback = null;
$preview = [];
// optional: show current positions as JSON for reference (GET request)
$currentPositionsJson = null;
if (!empty($_GET['show_pos'])) {
  $showDeposito = isset($_GET['show_pos_deposito']) ? (int) $_GET['show_pos_deposito'] : 0;
  $limit = min(1000, max(1, (int) ($_GET['show_pos_limit'] ?? 100)));
  // pick a safe list of columns to include
  $colsToTry = ['id', 'deposito_id', 'rack', 'columna', 'nivel', 'fondo', 'code', 'code_full', 'pos_code', 'pos_code_full', 'capacidad_pallets', 'ambiente_id', 'activo', 'created_at', 'created_by', 'title'];
  $existingCols = [];
  foreach ($colsToTry as $c) {
    if (columnExists($pdo, 'wh_posicion', $c))
      $existingCols[] = $c;
  }
  if (empty($existingCols)) {
    $currentPositionsJson = json_encode(['error' => 'no compatible columns found in wh_posicion'], JSON_PRETTY_PRINT);
  } else {
    $colSql = implode(', ', array_map(function ($c) {
      return "`$c`";
    }, $existingCols));
    try {
      if ($showDeposito > 0) {
        $st = $pdo->prepare("SELECT $colSql FROM `wh_posicion` WHERE deposito_id = ? ORDER BY id LIMIT ?");
        $st->execute([$showDeposito, $limit]);
      } else {
        $st = $pdo->prepare("SELECT $colSql FROM `wh_posicion` ORDER BY id LIMIT ?");
        $st->execute([$limit]);
      }
      $rows = $st->fetchAll(PDO::FETCH_ASSOC);
      $currentPositionsJson = json_encode($rows, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    } catch (Throwable $e) {
      $currentPositionsJson = json_encode(['error' => $e->getMessage()], JSON_PRETTY_PRINT);
    }
  }
}
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $loggedIn) {
  $dryRun = !empty($_POST['dry_run']);
  $deposito_id = (int) ($_POST['deposito_id'] ?? 0);
  $racks = max(1, (int) ($_POST['racks'] ?? 1));
  $cols = max(1, (int) ($_POST['cols'] ?? 1));
  $niveles = max(1, (int) ($_POST['niveles'] ?? 1));
  $fondos = max(1, (int) ($_POST['fondos'] ?? 1));
  $cap = max(1, (int) ($_POST['capacidad'] ?? 1));
  $orient = strtoupper(substr((string) ($_POST['orientacion'] ?? 'N'), 0, 1));
  $ambiente_id = isset($_POST['ambiente_id']) && (int) $_POST['ambiente_id'] > 0 ? (int) $_POST['ambiente_id'] : null;
  $useAmbCap = !empty($_POST['use_amb_capacity']);
  $ambCapacityOverride = isset($_POST['amb_capacity_override']) && $_POST['amb_capacity_override'] !== '' ? (int) $_POST['amb_capacity_override'] : null;
  $saveAmbPreset = !empty($_POST['save_amb_preset']);
  // whether to update existing positions (if false, use INSERT IGNORE to skip existing)
  $updateExisting = isset($_POST['update_existing']) ? !empty($_POST['update_existing']) : true;
  $nota = isset($_POST['nota']) ? substr((string) $_POST['nota'], 0, 255) : null;
  $uid = (int) $_SESSION['usuario_id'];

  if ($racks * $cols * $niveles * $fondos > 5000) {
    $feedback = ['ok' => false, 'msg' => 'Volumen muy grande. Reduce los rangos.'];
  } else {
    $total = $racks * $cols * $niveles * $fondos;
    $inserted = $updated = $skipped = $errs = 0;
    $errsMsg = [];

    // prepare deposit code
    $depCode = 'DEP';
    $r = $pdo->prepare('SELECT code FROM wh_deposito WHERE id=? LIMIT 1');
    $r->execute([$deposito_id]);
    $tmp = $r->fetchColumn();
    if ($tmp)
      $depCode = $tmp;

    // determine ambiente-based capacity if requested
    $ambienteCode = null;
    $ambienteName = null;
    $ambPreset = null;
    if ($ambiente_id) {
      $s = $pdo->prepare('SELECT code, nombre FROM wh_ambiente WHERE id=? LIMIT 1');
      $s->execute([$ambiente_id]);
      $row = $s->fetch(PDO::FETCH_ASSOC);
      if ($row) {
        $ambienteCode = strtoupper((string) $row['code']);
        $ambienteName = strtoupper((string) $row['nombre']);
      }
      // read preset if exists (only if table available)
  $ambPreset = null;
      if (tableExists($pdo, 'wh_ambiente_capacity')) {
        try {
          $ps = $pdo->prepare('SELECT capacidad_pallets FROM wh_ambiente_capacity WHERE ambiente_id=? LIMIT 1');
          $ps->execute([$ambiente_id]);
          $ambPreset = $ps->fetchColumn();
        } catch (Throwable $e) {
          $ambPreset = null; // ignore and continue
        }
      }
    }

    // helper to compute per-position capacity (respects checkbox use_amb_capacity)
    $computePosCap = function ($override = null) use ($cap, $useAmbCap, $ambienteCode, $ambienteName, $ambPreset) {
      // explicit override wins
      if ($override !== null && $override > 0) {
        return (int) $override;
      }
      // If ambiente-based capacity is enabled
      if ($useAmbCap) {
        if ($ambPreset !== null && (int) $ambPreset > 0) {
          return (int) $ambPreset;
        }
        $posCap = max(1, (int) $cap);
        if ($ambienteCode !== null || $ambienteName !== null) {
          $hay = strtoupper(($ambienteCode ?? '') . ' ' . ($ambienteName ?? ''));
          if (strpos($hay, 'PICK') !== false || strpos($hay, 'PICKING') !== false) {
            $posCap = 2; // picking zones may hold 2 pallets
          }
          if (strpos($hay, 'CUAR') !== false || strpos($hay, 'CUARENTENA') !== false) {
            $posCap = 2; // quarantine may allow more
          }
        }
        return $posCap;
      }
      // Ambiente-based disabled: honor input capacidad
      return max(1, (int) $cap);
    };

    // DRY RUN: generate preview list and counts without writing to DB
    if ($dryRun) {
      // Use shared generator
      $previewLimit = 200;
      $preview = \App\LayoutBuilder::generatePositions($depCode, $racks, $cols, $niveles, $fondos, $previewLimit);
      // add pos capacity to preview entries (respect override and checkbox)
      foreach ($preview as &$pp) {
        $pp['capacidad_pallets'] = $computePosCap($ambCapacityOverride);
        // convert numeric rack/columna from generator to letters for preview
        if (isset($pp['rack']) && is_numeric($pp['rack'])) {
          $pp['rack'] = indexToLetters((int) $pp['rack']);
        }
        if (isset($pp['columna']) && is_numeric($pp['columna'])) {
          $pp['columna'] = indexToLetters((int) $pp['columna']);
        }
        // ensure lado present as numeric default
        if (!isset($pp['lado']))
          $pp['lado'] = 1;
      }
      unset($pp);
      // Prepend two quarantine positions (recepción, dañados)
      try {
        // detect cuarentena ambiente id if exists
        $cuarAmbId = null;
        try {
          $q = $pdo->prepare("SELECT id FROM wh_ambiente WHERE code LIKE '%CUAR%' OR nombre LIKE '%CUARENTENA%' LIMIT 1");
          $q->execute();
          $cuarAmbId = $q->fetchColumn() ?: null;
        } catch (Throwable $e) {
          $cuarAmbId = null;
        }
        $cuar1 = ['rack' => 0, 'columna' => 0, 'nivel' => 0, 'fondo' => 0, 'code' => 'CUAR-REC', 'code_full' => "{$depCode}-CUAR-REC", "capacidad_pallets" => $computePosCap($ambCapacityOverride), 'ambiente_id' => $cuarAmbId];
        $cuar2 = ['rack' => 0, 'columna' => 0, 'nivel' => 0, 'fondo' => 0, 'code' => 'CUAR-DAN', 'code_full' => "{$depCode}-CUAR-DAN", "capacidad_pallets" => $computePosCap($ambCapacityOverride), 'ambiente_id' => $cuarAmbId];
        array_unshift($preview, $cuar2);
        array_unshift($preview, $cuar1);
      } catch (Throwable $e) { /* ignore preview enrichment errors */
      }
      $feedback = ['ok' => true, 'dry_run' => true, 'total' => $total, 'preview_count' => count($preview), 'preview' => $preview];
    } else {
      // actual write path
      try {
        $pdo->beginTransaction();
        // Build column list dynamically based on existing columns in wh_posicion
        $table = 'wh_posicion';
        $wantedCols = [
          'deposito_id',
          'rack',
          'columna',
          'nivel',
          'fondo',
          'lado',
          'orientacion',
          'ambiente_id',
          'capacidad_pallets',
          'activo',
          'title',
          'code',
          'code_full',
          'pos_code',
          'pos_code_full',
          'created_at',
          'created_by'
        ];
        $existing = [];
        foreach ($wantedCols as $c) {
          if (columnExists($pdo, $table, $c))
            $existing[] = $c;
        }

        // Always ensure some minimal set exists
        if (empty($existing)) {
          throw new RuntimeException('Tabla wh_posicion no tiene las columnas esperadas.');
        }

        // Prepare INSERT clause
        $colsSql = implode(', ', array_map(function ($c) {
          return "`$c`";
        }, $existing));
        // Prepare VALUES placeholders (use NOW() for created_at if present)
        $placeholders = [];
        foreach ($existing as $c) {
          if ($c === 'created_at') {
            $placeholders[] = 'NOW()';
            continue;
          }
          $placeholders[] = ':' . $c;
        }
        $valsSql = implode(', ', $placeholders);

        if ($updateExisting) {
          // Build ON DUPLICATE KEY UPDATE only for writable columns (avoid created_by/created_at)
          $upSets = [];
          foreach ($existing as $c) {
            if (in_array($c, ['created_at', 'created_by']))
              continue;
            // updated_at special handling if exists
            if ($c === 'updated_at') {
              $upSets[] = "updated_at = NOW()";
              continue;
            }
            $upSets[] = "`$c` = VALUES(`$c`)";
          }
          $onDup = !empty($upSets) ? 'ON DUPLICATE KEY UPDATE ' . implode(', ', $upSets) : '';
          $sql = "INSERT INTO `$table` ($colsSql) VALUES ($valsSql) $onDup";
        } else {
          // INSERT IGNORE: omit ON DUPLICATE behaviour
          $sql = "INSERT IGNORE INTO `$table` ($colsSql) VALUES ($valsSql)";
        }

        $stmt = $pdo->prepare($sql);

        // Determine whether rack/columna columns are numeric in DB; if numeric, store indices, otherwise store letters
        $rackIsNumeric = false;
        $colIsNumeric = false;
        $ladoIsNumeric = false;
        if (in_array('rack', $existing)) {
          $dt = columnType($pdo, $table, 'rack');
          if ($dt && preg_match('/^(tinyint|smallint|mediumint|int|integer|bigint)$/i', $dt))
            $rackIsNumeric = true;
        }
        if (in_array('columna', $existing)) {
          $dt = columnType($pdo, $table, 'columna');
          if ($dt && preg_match('/^(tinyint|smallint|mediumint|int|integer|bigint)$/i', $dt))
            $colIsNumeric = true;
        }
        if (in_array('lado', $existing)) {
          $dt = columnType($pdo, $table, 'lado');
          if ($dt && preg_match('/^(tinyint|smallint|mediumint|int|integer|bigint)$/i', $dt))
            $ladoIsNumeric = true;
        }

  // Compute rack start index based on existing positions for depósito/ambiente
        $rackStartIndex = 0;
        if (in_array('rack', $existing)) {
          $ambFilter = '';
          $params = [':dep' => $deposito_id];
          if (in_array('ambiente_id', $existing)) {
            if ($ambiente_id !== null) {
              $ambFilter = ' AND ambiente_id = :amb';
              $params[':amb'] = $ambiente_id;
            } else {
              $ambFilter = ' AND (ambiente_id IS NULL OR ambiente_id = 0)';
            }
          }
          if ($rackIsNumeric) {
            $q = $pdo->prepare("SELECT MAX(rack) FROM `$table` WHERE deposito_id = :dep" . $ambFilter);
            $q->execute($params);
            $rackStartIndex = (int) ($q->fetchColumn() ?: 0);
          } else {
            // textual rack values -> compute max by converting letters
            $q = $pdo->prepare("SELECT rack FROM `$table` WHERE deposito_id = :dep" . $ambFilter);
            $q->execute($params);
            $maxIdx = 0;
            while ($row = $q->fetch(PDO::FETCH_NUM)) {
              $maxIdx = max($maxIdx, lettersToIndex((string) $row[0]));
            }
            $rackStartIndex = $maxIdx;
          }
        }

        // Compute per-rack starting column index (continuity within each rack)
        $colStartByRackIndex = [];
        if (in_array('rack', $existing) && in_array('columna', $existing)) {
          $ambFilter = '';
          $params = [':dep' => $deposito_id];
          if (in_array('ambiente_id', $existing)) {
            if ($ambiente_id !== null) {
              $ambFilter = ' AND ambiente_id = :amb';
              $params[':amb'] = $ambiente_id;
            } else {
              $ambFilter = ' AND (ambiente_id IS NULL OR ambiente_id = 0)';
            }
          }
          $q = $pdo->prepare("SELECT rack, columna FROM `$table` WHERE deposito_id = :dep" . $ambFilter);
          $q->execute($params);
          while ($row = $q->fetch(PDO::FETCH_ASSOC)) {
            $rVal = $row['rack'];
            $cVal = $row['columna'];
            if ($rVal === null || $rVal === '' || $cVal === null || $cVal === '') continue;
            $rIdx = $rackIsNumeric ? (int)$rVal : lettersToIndex((string)$rVal);
            if ($rIdx <= 0) continue;
            $cIdx = $colIsNumeric ? (int)$cVal : lettersToIndex((string)$cVal);
            if ($cIdx <= 0) continue;
            if (!isset($colStartByRackIndex[$rIdx]) || $cIdx > $colStartByRackIndex[$rIdx]) {
              $colStartByRackIndex[$rIdx] = $cIdx;
            }
          }
        }

        // Attempt to insert two quarantine positions first (recepción, dañados) — only if missing
        try {
          // detect cuarentena ambiente id if exists
          $cuarAmbId = null;
          try {
            $q = $pdo->prepare("SELECT id FROM wh_ambiente WHERE code LIKE '%CUAR%' OR nombre LIKE '%CUARENTENA%' LIMIT 1");
            $q->execute();
            $cuarAmbId = $q->fetchColumn() ?: null;
          } catch (Throwable $e) {
            $cuarAmbId = null;
          }
          $cuarItems = [
            ['code' => 'CUAR-REC', 'title' => 'CUARENTENA - Recepción', 'amb' => $cuarAmbId],
            ['code' => 'CUAR-DAN', 'title' => 'CUARENTENA - Dañados', 'amb' => $cuarAmbId]
          ];
          // existence check helpers
          $hasCodeCol = in_array('code', $existing);
          $hasCodeFullCol = in_array('code_full', $existing);
          $hasDepCol = in_array('deposito_id', $existing);
          foreach ($cuarItems as $citem) {
            $shouldInsert = true;
            try {
              if ($hasCodeFullCol && $hasDepCol) {
                $chk = $pdo->prepare("SELECT 1 FROM `$table` WHERE deposito_id = ? AND code_full = ? LIMIT 1");
                $chk->execute([$deposito_id, $depCode . '-' . $citem['code']]);
                $shouldInsert = $chk->fetchColumn() ? false : true;
              } elseif ($hasCodeCol && $hasDepCol) {
                $chk = $pdo->prepare("SELECT 1 FROM `$table` WHERE deposito_id = ? AND code = ? LIMIT 1");
                $chk->execute([$deposito_id, $citem['code']]);
                $shouldInsert = $chk->fetchColumn() ? false : true;
              } elseif ($hasCodeCol) {
                $chk = $pdo->prepare("SELECT 1 FROM `$table` WHERE code = ? LIMIT 1");
                $chk->execute([$citem['code']]);
                $shouldInsert = $chk->fetchColumn() ? false : true;
              }
            } catch (Throwable $e) {
              $shouldInsert = true; // if check fails, attempt insert
            }
            if (!$shouldInsert) continue;
            $params = [];
            foreach ($existing as $c) {
              switch ($c) {
                case 'deposito_id':
                  $params[':deposito_id'] = $deposito_id;
                  break;
                // CUAR positions use empty rack/columna (letters), numeric nivel/fondo = 0
                case 'rack':
                  $params[':rack'] = $rackIsNumeric ? 0 : '';
                  break;
                case 'columna':
                  $params[':columna'] = $colIsNumeric ? 0 : '';
                  break;
                case 'nivel':
                  $params[':nivel'] = 0;
                  break;
                case 'fondo':
                  $params[':fondo'] = 0;
                  break;
                // default lado for CUAR positions (numeric if column exists)
                case 'lado':
                  $params[':lado'] = $ladoIsNumeric ? 1 : 1;
                  break;
                case 'orientacion':
                  $params[':orientacion'] = $orient;
                  break;
                case 'ambiente_id':
                  $params[':ambiente_id'] = $citem['amb'];
                  break;
                case 'capacidad_pallets':
                  $params[':capacidad_pallets'] = $computePosCap($ambCapacityOverride);
                  break;
                case 'activo':
                  $params[':activo'] = 1;
                  break;
                case 'title':
                  $params[':title'] = $citem['title'];
                  break;
                case 'code':
                  $params[':code'] = $citem['code'];
                  break;
                case 'code_full':
                  $params[':code_full'] = $code_full = $depCode . '-' . $citem['code'];
                  break;
                case 'pos_code':
                  $params[':pos_code'] = $citem['code'];
                  break;
                case 'pos_code_full':
                  $params[':pos_code_full'] = $depCode . '-' . $citem['code'];
                  break;
                case 'created_by':
                  $params[':created_by'] = $uid;
                  break;
                default:
                  break;
              }
            }
            try {
              $stmt->execute($params);
              $rc = $stmt->rowCount();
              if ($rc === 1)
                $inserted++;
              elseif ($rc === 2)
                $updated++;
              else
                $skipped++;
            } catch (Throwable $e) {
              $errs++;
              if ($errs <= 10)
                $errsMsg[] = $e->getMessage();
            }
          }
        } catch (Throwable $e) { /* ignore */
        }

        for ($ri = 1; $ri <= $racks; $ri++) {
          for ($ci = 1; $ci <= $cols; $ci++) {
            for ($ni = 1; $ni <= $niveles; $ni++) {
              for ($fi = 1; $fi <= $fondos; $fi++) {
                try {
                  // Rack/column codes (letters) for human-friendly code strings
                  $rackIndex = $rackStartIndex + $ri; // continue after existing max
                  $rackLetter = indexToLetters($rackIndex);
                  $colIndex = ($colStartByRackIndex[$rackIndex] ?? 0) + $ci;
                  $colLetter = indexToLetters($colIndex);
                  $rackCode = 'R' . $rackLetter;
                  $colCode = 'C' . $colLetter;
                  $nivCode = sprintf('N%02d', $ni);
                  $fCode = sprintf('F%02d', $fi);
                  $code = "$rackCode-$colCode-$nivCode-$fCode";
                  $code_full = "$depCode-$code";
                  $pos_code = $code;
                  $pos_code_full = $code_full;

                  $posCap = $computePosCap($ambCapacityOverride);
                  // Build params dynamically to match $existing columns
                  $params = [];
                  foreach ($existing as $c) {
                    switch ($c) {
                      case 'deposito_id':
                        $params[':deposito_id'] = $deposito_id;
                        break;
                      // store rack/columna as letters or numeric index depending on DB column type
                      case 'rack':
                        $params[':rack'] = ($rackIsNumeric ? $rackIndex : $rackLetter);
                        break;
                      case 'columna':
                        $params[':columna'] = ($colIsNumeric ? $colIndex : $colLetter);
                        break;
                      case 'nivel':
                        $params[':nivel'] = $ni;
                        break;
                      case 'fondo':
                        $params[':fondo'] = $fi;
                        break;
                      // lado numeric: default 1 (can be parameterized later)
                      case 'lado':
                        $params[':lado'] = ($ladoIsNumeric ? 1 : 1);
                        break;
                      case 'orientacion':
                        $params[':orientacion'] = $orient;
                        break;
                      case 'ambiente_id':
                        $params[':ambiente_id'] = $ambiente_id;
                        break;
                      case 'capacidad_pallets':
                        $params[':capacidad_pallets'] = $posCap;
                        break;
                      case 'activo':
                        $params[':activo'] = 1;
                        break;
                      case 'title':
                        $params[':title'] = $nota;
                        break;
                      case 'code':
                        $params[':code'] = $code;
                        break;
                      case 'code_full':
                        $params[':code_full'] = $code_full;
                        break;
                      case 'pos_code':
                        $params[':pos_code'] = $pos_code;
                        break;
                      case 'pos_code_full':
                        $params[':pos_code_full'] = $pos_code_full;
                        break;
                      case 'created_by':
                        $params[':created_by'] = $uid;
                        break;
                      // created_at handled via NOW() placeholder if present
                      default:
                        break;
                    }
                  }
                  $stmt->execute($params);
                  $rc = $stmt->rowCount();
                  if ($rc === 1)
                    $inserted++;
                  elseif ($rc === 2)
                    $updated++;
                  else
                    $skipped++;
                } catch (Throwable $e) {
                  $errs++;
                  if ($errs <= 10)
                    $errsMsg[] = $e->getMessage();
                }
              }
            }
          }
        }
        $pdo->commit();
        $feedback = ['ok' => true, 'total' => $total, 'inserted' => $inserted, 'updated' => $updated, 'skipped' => $skipped, 'errors' => $errs, 'errorsMsg' => $errsMsg];
      } catch (Throwable $e) {
        if ($pdo->inTransaction())
          $pdo->rollBack();
        $feedback = ['ok' => false, 'msg' => $e->getMessage()];
      }
    }
    // save preset if requested - admin-only
    if ($saveAmbPreset) {
      if (!function_exists('is_admin') || !is_admin()) {
        $feedback['preset_saved'] = false;
        $feedback['preset_err'] = 'permiso denegado: solo administradores pueden guardar presets';
      } elseif (!tableExists($pdo, 'wh_ambiente_capacity')) {
        $feedback['preset_saved'] = false;
        $feedback['preset_err'] = 'tabla wh_ambiente_capacity no existe en la base de datos';
      } elseif ($ambiente_id && $ambCapacityOverride && $ambCapacityOverride > 0) {
        $up = $pdo->prepare('INSERT INTO wh_ambiente_capacity (ambiente_id, capacidad_pallets, created_by) VALUES (:id, :cap, :uid) ON DUPLICATE KEY UPDATE capacidad_pallets=VALUES(capacidad_pallets), updated_by=VALUES(created_by), updated_at=NOW()');
        try {
          $up->execute([':id' => $ambiente_id, ':cap' => $ambCapacityOverride, ':uid' => $uid]);
          $feedback['preset_saved'] = true;
        } catch (Throwable $e) {
          $feedback['preset_saved'] = false;
          $feedback['preset_err'] = $e->getMessage();
        }
      }
    }
  }
}

if (!function_exists('h')) {
  function h($s)
  {
    return htmlspecialchars((string) $s, ENT_QUOTES, 'UTF-8');
  }
}

?>
<!doctype html>
<html lang="es">

<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <title>Layout Builder - wh_posicion</title>
  <link rel="stylesheet" href="<?= url('/assets/css/bootstrap.css') ?>">
  <style>
    .container {
      max-width: 980px;
      padding: 20px
    }

    body {
      background: #f7f9fb;
      color: #222
    }

    .page-header {
      display: flex;
      align-items: center;
      justify-content: space-between;
      margin-bottom: 18px
    }

    .page-title {
      font-size: 1.25rem;
      margin: 0
    }

    .card-ghost {
      background: linear-gradient(180deg, #ffffffcc, #f8fbff);
      border: 1px solid #e6eef8
    }

    .preview-code {
      font-family: monospace;
      background: #f4f7fb;
      padding: 2px 6px;
      border-radius: 4px
    }

    .hint small {
      display: block;
      color: #6b7280
    }
  </style>
</head>

<body>
  <div class="container">
    <div class="page-header">
      <h3 class="page-title">Layout Builder — Generador de posiciones <small class="text-muted">(tabla
          <code>wh_posicion</code>)</small></h3>
      <div>
        <?php if (!$loggedIn): ?><span class="badge bg-warning text-dark">Sesión requerida para
            escribir</span><?php endif; ?>
      </div>
    </div>

    <?php if ($feedback): ?>
      <?php if (!empty($feedback['ok'])): ?>
        <div class="alert alert-success">Hecho. Insertados: <?= (int) $feedback['inserted'] ?> · Actualizados:
          <?= (int) $feedback['updated'] ?> · Errores: <?= (int) $feedback['errors'] ?>
        </div>
      <?php else: ?>
        <div class="alert alert-danger">Error: <?= h($feedback['msg'] ?? 'Desconocido') ?></div>
      <?php endif; ?>
    <?php endif; ?>

    <div class="card card-ghost p-3 mb-3">
      <form method="post" class="row g-3">
        <div class="col-md-3">
          <label class="form-label">Depósito</label>
          <select name="deposito_id" class="form-select">
            <?php foreach ($deps as $d): ?>
              <option value="<?= (int) $d['id'] ?>" <?= (isset($_POST['deposito_id']) && (int) $_POST['deposito_id'] == (int) $d['id']) ? 'selected' : '' ?>><?= h($d['code'] . ' — ' . $d['nombre']) ?>
              </option>
            <?php endforeach; ?>
          </select>
        </div>
        <div class="col-md-3">
          <label class="form-label">Racks</label>
          <input type="number" min="1" name="racks" class="form-control" value="<?= h($_POST['racks'] ?? '1') ?>" />
        </div>
        <div class="col-md-3">
          <label class="form-label">Columnas</label>
          <input type="number" min="1" name="cols" class="form-control" value="<?= h($_POST['cols'] ?? '1') ?>" />
        </div>
        <div class="col-md-3">
          <label class="form-label">Niveles</label>
          <input type="number" min="1" name="niveles" class="form-control" value="<?= h($_POST['niveles'] ?? '1') ?>" />
        </div>

        <div class="col-md-3">
          <label class="form-label">Fondos</label>
          <input type="number" min="1" name="fondos" class="form-control" value="<?= h($_POST['fondos'] ?? '1') ?>" />
        </div>
        <div class="col-md-3">
          <label class="form-label">Capacidad (pallets)</label>
          <input type="number" min="1" name="capacidad" class="form-control"
            value="<?= h($_POST['capacidad'] ?? '1') ?>" />
          <div class="form-text">Capacidad por posición. Cada posición creada tendrá este valor en
            <code>capacidad_pallets</code>. Por defecto suele ser <b>1</b>.
          </div>
        </div>
        <div class="col-md-6">
          <div class="form-check mt-2">
            <input class="form-check-input" type="checkbox" id="use_amb_capacity" name="use_amb_capacity"
              <?= !empty($_POST['use_amb_capacity']) ? 'checked' : '' ?> />
            <label class="form-check-label" for="use_amb_capacity">Asignar capacidad basada en el <em>ambiente</em> (p.
              ej. zonas de picking o cuarentena)</label>
            <div class="form-text">Si está activado y seleccionas un ambiente, el sistema intentará aplicar una
              capacidad por posición adecuada (p. ej. picking → 2). Puedes forzar un valor específico abajo.</div>
          </div>
          <div class="mt-2" id="amb_capacity_override_wrap"
            style="display:<?= !empty($_POST['use_amb_capacity']) ? 'block' : 'none' ?>">
            <label class="form-label">Override de capacidad por ambiente (opcional)</label>
            <div class="input-group">
              <input type="number" min="1" name="amb_capacity_override" class="form-control"
                value="<?= h($_POST['amb_capacity_override'] ?? '') ?>"
                placeholder="Ej: 2 para permitir 2 pallets por posición">
              <?php if (function_exists('is_admin') && is_admin()): ?>
                <button type="submit" name="save_amb_preset" value="1" class="btn btn-outline-success">Guardar
                  preset</button>
              <?php else: ?>
                <button type="button" class="btn btn-outline-secondary" disabled title="Solo administradores">Guardar
                  preset</button>
              <?php endif; ?>
            </div>
            <div class="form-text">Si indicas un número aquí, tendrá prioridad sobre la heurística automática. Pulsa
              "Guardar preset" para persistir este valor para el ambiente seleccionado.</div>
          </div>
        </div>
        <div class="col-md-3">
          <label class="form-label">Orientación</label>
          <select name="orientacion" class="form-select" aria-label="Orientación">
            <option value="N">N</option>
            <option value="E">E</option>
            <option value="S">S</option>
            <option value="W">W</option>
          </select>
        </div>
        <div class="col-md-3"><label class="form-label">Ambiente (opcional)</label>
          <select name="ambiente_id" class="form-select">
            <option value="0">-- (no asignar)</option>
            <?php foreach ($ambs as $a): ?>
              <option value="<?= (int) $a['id'] ?>" <?= (isset($_POST['ambiente_id']) && (int) $_POST['ambiente_id'] == (int) $a['id']) ? 'selected' : '' ?>><?= h($a['code'] . ' — ' . $a['nombre']) ?>
              </option>
            <?php endforeach; ?>
          </select>
        </div>

        <div class="col-12"><label class="form-label">Nota (opcional)</label><input name="nota" class="form-control"
            value="<?= h($_POST['nota'] ?? '') ?>"></div>

        <div class="col-6 form-check">
          <input class="form-check-input" type="checkbox" name="dry_run" id="dry_run" <?= (!empty($_POST['dry_run']) ? 'checked' : '') ?> />
          <label class="form-check-label" for="dry_run">Vista previa (no escribe en la BD)</label>
        </div>
        <div class="col-6 form-check text-end">
          <input class="form-check-input" type="checkbox" name="update_existing" id="update_existing"
            <?= (isset($_POST['update_existing']) ? (!empty($_POST['update_existing']) ? 'checked' : '') : 'checked') ?> />
          <label class="form-check-label" for="update_existing">Actualizar posiciones existentes (si no está marcado,
            las posiciones ya existentes se omiten)</label>
        </div>
        <div class="col-6 text-end">
          <button class="btn btn-primary" type="submit" <?= $loggedIn ? '' : 'disabled' ?>>Generar / Actualizar
            posiciones</button>
          <a class="btn btn-secondary" href="<?= h('/public/tools/layout_builder.php') ?>">Reset</a>
        </div>
      </form>
    </div>
    <script>
      (function () {
        var chk = document.getElementById('use_amb_capacity');
        var wrap = document.getElementById('amb_capacity_override_wrap');
        if (!chk || !wrap) return;
        chk.addEventListener('change', function () { wrap.style.display = chk.checked ? 'block' : 'none'; });
      })();
    </script>
  <!-- use project's bootstrap JS (popper + bootstrap) below -->

    <hr>
    <div class="card mb-3">
      <div class="card-body">
        <h5 class="card-title">Ver posiciones actuales (JSON)</h5>
        <p class="card-text">Consulta las filas actuales de <code>wh_posicion</code> para referencia. Útil para comparar
          antes/después.</p>
        <form method="get" class="row g-2 align-items-end">
          <input type="hidden" name="show_pos" value="1" />
          <div class="col-md-4">
            <label class="form-label">Depósito (opcional)</label>
            <select name="show_pos_deposito" class="form-select">
              <option value="0">-- Todos</option>
              <?php foreach ($deps as $d): ?>
                <option value="<?= (int) $d['id'] ?>" <?= (isset($_GET['show_pos_deposito']) && (int) $_GET['show_pos_deposito'] == (int) $d['id']) ? 'selected' : '' ?>>
                  <?= h($d['code'] . ' — ' . $d['nombre']) ?>
                </option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-3">
            <label class="form-label">Límite</label>
            <input type="number" name="show_pos_limit" min="1" max="1000" class="form-control"
              value="<?= h($_GET['show_pos_limit'] ?? 100) ?>" />
          </div>
          <div class="col-md-5 text-end">
            <button class="btn btn-outline-primary">Mostrar JSON</button>
            <a class="btn btn-link" href="<?= h('/public/tools/layout_builder.php') ?>">Reset</a>
          </div>
        </form>
      </div>
    </div>

    <?php if ($currentPositionsJson !== null): ?>
      <div class="card mb-3">
        <div class="card-body p-2">
          <h6>JSON — posiciones (primeras <?= h((int) ($_GET['show_pos_limit'] ?? 100)) ?> filas)</h6>
          <pre
            style="max-height:420px;overflow:auto;background:#0b0b0b;color:#e6ffed;padding:12px;border-radius:6px;font-size:0.85rem;"><?= h($currentPositionsJson) ?></pre>
        </div>
      </div>
    <?php endif; ?>
    <?php if (!empty($feedback) && !empty($feedback['dry_run'])): ?>
      <div class="mb-3">
        <div class="alert alert-info mb-2">
          <strong>Vista previa:</strong> Total posiciones calculadas: <?= (int) $feedback['total'] ?>. Mostrando
          <?= (int) $feedback['preview_count'] ?> de <?= (int) $feedback['total'] ?> (máx 200).
        </div>
        <div class="card">
          <div class="card-body p-2">
            <div class="table-responsive">
              <table class="table table-sm table-striped mb-0">
                <thead class="table-light">
                  <tr>
                    <th style="width:48px">#</th>
                    <th>Rack</th>
                    <th>Col</th>
                    <th>Lado</th>
                    <th>Niv</th>
                    <th>F</th>
                    <th>Code</th>
                    <th>Code full</th>
                  </tr>
                </thead>
                <tbody>
                  <?php foreach ($feedback['preview'] as $i => $p): ?>
                    <tr>
                      <td><?= $i + 1 ?></td>
                      <td><?= h($p['rack']) ?></td>
                      <td><?= h($p['columna']) ?></td>
                      <td><?= isset($p['lado']) ? (int) $p['lado'] : 1 ?></td>
                      <td><?= (int) $p['nivel'] ?></td>
                      <td><?= (int) $p['fondo'] ?></td>
                      <td><span class="preview-code"><?= h($p['code']) ?></span></td>
                      <td><span class="preview-code"><?= h($p['code_full']) ?></span></td>
                    </tr>
                  <?php endforeach; ?>
                </tbody>
              </table>
            </div>
          </div>
        </div>
      </div>
    <?php endif; ?>
    <p class="small text-muted">Esta herramienta modifica la tabla <code>wh_posicion</code>. Haz backup antes de usarla
      en producción.</p>

  <!-- jQuery primero -->
  <script src="<?= url('/assets/js/jquery-3.5.1.min.js') ?>"></script>

  <!-- Select2 (SOLO una vez) -->
  <script src="https://cdnjs.cloudflare.com/ajax/libs/select2/4.0.13/js/select2.min.js"></script>

  <!-- Auto-init para .form-select (tu archivo) -->
  <script src="<?= url('/assets/js/theme/select2-init.js') ?>"></script>

  <script src="<?= url('/assets/js/bootstrap/popper.min.js') ?>"></script>
  <script src="<?= url('/assets/js/bootstrap/bootstrap.min.js') ?>"></script>

  </div>
</body>

</html>