<?php
// scripts/apply_sp_preparar_auto.php
// CLI: php scripts/apply_sp_preparar_auto.php
declare(strict_types=1);

require_once __DIR__ . '/../config/db.php';

function outln(string $s): void { echo $s . PHP_EOL; }

try {
    $pdo = getPDO();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec("SET NAMES utf8mb4");

    // Ensure log table exists for diagnostics
    outln('Ensuring log table exists...');
    $pdo->exec(<<<'SQL'
CREATE TABLE IF NOT EXISTS so_preparar_auto_log (
  id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  ts DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
  pedido_codigo VARCHAR(64) NULL,
  pedido_id BIGINT UNSIGNED NULL,
  pre_id BIGINT UNSIGNED NULL,
  fase VARCHAR(20) NULL,
  item_id BIGINT UNSIGNED NULL,
  producto_id BIGINT UNSIGNED NULL,
  lote_id BIGINT UNSIGNED NULL,
  lote_code VARCHAR(64) NULL,
  from_pos_id BIGINT UNSIGNED NULL,
  to_pos_id BIGINT UNSIGNED NULL,
  pallet_id BIGINT UNSIGNED NULL,
  take_uv INT NULL,
  take_uc INT NULL,
  need_uv_post INT NULL,
  need_uc_post INT NULL,
  venc DATE NULL,
  msg VARCHAR(255) NULL,
  INDEX idx_prelog_ped (pedido_id),
  INDEX idx_prelog_pre (pre_id),
  INDEX idx_prelog_ts (ts)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
SQL);

    outln('Dropping procedure if exists...');
    $pdo->exec("DROP PROCEDURE IF EXISTS sp_so_preparar_auto");

    outln('Creating procedure sp_so_preparar_auto ...');
    $sql = <<<'SQL'
CREATE PROCEDURE sp_so_preparar_auto(
  IN p_pedido_codigo   VARCHAR(64),
  IN p_deposito_code   VARCHAR(32),
  IN p_pos_prep_code   VARCHAR(64),
  IN p_simulate        TINYINT
)
BEGIN
  /* Logging: controlled by session variable @so_pre_log (0 by default) */
  DECLARE v_do_log TINYINT DEFAULT 0;
  DECLARE v_pedido_id    BIGINT UNSIGNED;
  DECLARE v_dep_id       INT UNSIGNED;
  DECLARE v_pos_prep_id  BIGINT UNSIGNED;
  DECLARE v_pos_pick_id  BIGINT UNSIGNED;
  DECLARE v_pre_id       BIGINT UNSIGNED;
  DECLARE v_pre_code     VARCHAR(128);
  DECLARE v_pre_est_id   INT UNSIGNED;
  DECLARE v_done         INT DEFAULT 0;

  DECLARE v_item_id      BIGINT UNSIGNED;
  DECLARE v_prod_id      BIGINT UNSIGNED;
  DECLARE v_lote_code    VARCHAR(64);
  DECLARE v_dest_id      BIGINT UNSIGNED;
  DECLARE v_need_uv      INT;
  DECLARE v_need_uc      INT;

  DECLARE v_pos_from     BIGINT UNSIGNED;
  DECLARE v_lote_id      BIGINT UNSIGNED;
  DECLARE v_pallet_id    BIGINT UNSIGNED;
  DECLARE v_take_uv      INT;
  DECLARE v_take_uc      INT;
  DECLARE v_move_id      BIGINT UNSIGNED;
  DECLARE v_direct       TINYINT;
  DECLARE v_max_cands    INT DEFAULT 0;
  DECLARE v_max_iters    INT DEFAULT 0;
  DECLARE v_it_pick      INT DEFAULT 0;
  DECLARE v_it_repo      INT DEFAULT 0;
  DECLARE v_it_pick2     INT DEFAULT 0;
  /* UC->UV conversion controls */
  DECLARE v_uc_to_uv     TINYINT DEFAULT 0;
  /* UV->UC conversion controls (abrir cajas) */
  DECLARE v_uv_to_uc     TINYINT DEFAULT 0;
  DECLARE v_uc_per_uv    INT DEFAULT 0;
  /* Movement deltas (what we actually move in wh_move) */
  DECLARE v_mv_uv        INT DEFAULT 0;
  DECLARE v_mv_uc        INT DEFAULT 0;
  /* Row-level captured quantities for repo/pick2 */
  DECLARE v_row_uv       INT DEFAULT 0;
  DECLARE v_row_uc       INT DEFAULT 0;
  /* Base picks before conversion */
  DECLARE v_take_uv_base INT DEFAULT 0;
  DECLARE v_take_uc_base INT DEFAULT 0;
    DECLARE v_break_uv     INT DEFAULT 0;
    DECLARE v_break_uc     INT DEFAULT 0;

  /* Now safe to set variables after all DECLAREs */
  SET v_do_log = IFNULL(@so_pre_log, 0);
  SET v_direct = IFNULL(@so_pre_direct_to_prep, 0);
  SET v_max_cands = IFNULL(@so_pre_max_cands, 0);
  SET v_max_iters = IFNULL(@so_pre_max_iters, 2000);
  SET v_uc_to_uv = IFNULL(@so_pre_uc_to_uv, 0);
  SET v_uv_to_uc = IFNULL(@so_pre_uv_to_uc, 1);

  /* Nota: sin EXIT HANDLER explícito para compatibilidad; los errores propagan */

  /* Log parameters received */
  IF v_do_log = 1 THEN
    INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, msg)
    VALUES (p_pedido_codigo, 0, 0, 'INIT', CONCAT('Parameters: uc_to_uv=', v_uc_to_uv, ', uv_to_uc=', v_uv_to_uc, ', log=', v_do_log));
  END IF;

  IF p_pedido_codigo IS NULL OR p_pedido_codigo = '' THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='p_pedido_codigo es requerido.';
  END IF;

  SELECT id INTO v_pedido_id FROM so_pedido WHERE codigo=p_pedido_codigo LIMIT 1;
  IF v_pedido_id IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='Pedido no encontrado.';
  END IF;

  SELECT id INTO v_dep_id FROM wh_deposito WHERE code=p_deposito_code LIMIT 1;
  IF v_dep_id IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='Depósito no encontrado.';
  END IF;

  /* PREP position */
  IF p_pos_prep_code IS NOT NULL AND p_pos_prep_code <> '' THEN
    SELECT id INTO v_pos_prep_id
      FROM wh_posicion
     WHERE deposito_id=v_dep_id
       AND (code=p_pos_prep_code OR code_full=p_pos_prep_code OR pos_code=p_pos_prep_code OR pos_code_full=p_pos_prep_code)
     LIMIT 1;
    IF v_pos_prep_id IS NULL THEN
      SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='Posición PREP indicada no existe en el depósito.';
    END IF;
  ELSE
    SELECT p.id INTO v_pos_prep_id
      FROM wh_posicion p
      JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code='PREP'
      LEFT JOIN wh_stock s ON s.posicion_id=p.id AND s.deposito_id=v_dep_id
     WHERE p.deposito_id=v_dep_id AND p.activo=1
     GROUP BY p.id
     ORDER BY COALESCE(SUM(s.qty_uv+s.qty_uc),0) ASC, p.id ASC
     LIMIT 1;
    IF v_pos_prep_id IS NULL THEN
      SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='No hay posiciones PREP en el depósito.';
    END IF;
  END IF;

  /* PICKING position (menos ocupada) */
  SELECT p.id INTO v_pos_pick_id
    FROM wh_posicion p
    JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code='PICKING'
    LEFT JOIN wh_stock s ON s.posicion_id=p.id AND s.deposito_id=v_dep_id
   WHERE p.deposito_id=v_dep_id AND p.activo=1
   GROUP BY p.id
   ORDER BY COALESCE(SUM(s.qty_uv+s.qty_uc),0) ASC, p.id ASC
   LIMIT 1;
  IF v_pos_pick_id IS NULL THEN
    SIGNAL SQLSTATE '45000' SET MESSAGE_TEXT='No hay posiciones PICKING en el depósito.';
  END IF;

  /* Crear/obtener pre-embarque */
  SET v_pre_code = CONCAT('PRE-', p_pedido_codigo);
  SELECT id INTO v_pre_est_id FROM so_preembarque_estado WHERE code='PENDIENTE' LIMIT 1;
  SELECT id INTO v_pre_id FROM so_preembarque WHERE codigo=v_pre_code LIMIT 1;
  IF v_pre_id IS NULL THEN
    INSERT INTO so_preembarque (codigo, pedido_id, deposito_id, estado_id, zona_posicion_id, asignado_at, inicio_at)
    VALUES (v_pre_code, v_pedido_id, v_dep_id, v_pre_est_id, v_pos_prep_id, NOW(), NOW());
    SET v_pre_id = LAST_INSERT_ID();
  ELSE
    UPDATE so_preembarque SET zona_posicion_id=COALESCE(zona_posicion_id, v_pos_prep_id) WHERE id=v_pre_id;
  END IF;

  /* log posiciones iniciales (antes de la transacción) */
  IF v_do_log = 1 THEN
    INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, from_pos_id, to_pos_id, msg)
    VALUES (p_pedido_codigo, v_pedido_id, v_pre_id, 'INIT', v_pos_pick_id, v_pos_prep_id, 'INIT POS PICK/PREP');
  END IF;

  /* Nota: se evita transacción explícita larga para reducir locks; usar autocommit */

  /* Ítems pendientes */
  IF p_simulate = 1 THEN
    DROP TEMPORARY TABLE IF EXISTS _plan;
    CREATE TEMPORARY TABLE _plan (
      fase VARCHAR(16),
      item_id BIGINT UNSIGNED,
      producto_id BIGINT UNSIGNED,
      lote_id BIGINT UNSIGNED NULL,
      lote_code VARCHAR(64) NULL,
      from_pos_id BIGINT UNSIGNED NULL,
      to_pos_id BIGINT UNSIGNED NULL,
      pallet_id BIGINT UNSIGNED NULL,
      take_uv INT,
      take_uc INT,
      venc DATE NULL
    ) ENGINE=MEMORY;
  END IF;
  DROP TEMPORARY TABLE IF EXISTS _pend;
  CREATE TEMPORARY TABLE _pend (
    pedido_dest_item_id BIGINT UNSIGNED,
    pedido_dest_id      BIGINT UNSIGNED,
    destinatario_id     BIGINT UNSIGNED,
    producto_id         BIGINT UNSIGNED,
    lote_codigo         VARCHAR(64),
    need_uv             INT,
    need_uc             INT
  ) ENGINE=InnoDB;

  INSERT INTO _pend
  SELECT i.id, d.id, d.destinatario_id, i.producto_id, i.lote_codigo,
         GREATEST(i.expected_uv - i.prepared_uv,0) AS need_uv,
         GREATEST(i.expected_uc - i.prepared_uc,0) AS need_uc
  FROM so_pedido_dest_item i
  JOIN so_pedido_dest d ON d.id=i.pedido_dest_id
  WHERE d.pedido_id = v_pedido_id
    AND (GREATEST(i.expected_uv - i.prepared_uv,0) > 0
      OR GREATEST(i.expected_uc - i.prepared_uc,0) > 0);

  /* Cursor por item */
  BEGIN
    DECLARE cur_it CURSOR FOR
      SELECT pedido_dest_item_id, producto_id, lote_codigo, destinatario_id, need_uv, need_uc
      FROM _pend;
    DECLARE CONTINUE HANDLER FOR NOT FOUND SET v_done=1;

    SET v_done=0;
    OPEN cur_it;
    it_loop: LOOP
      FETCH cur_it INTO v_item_id, v_prod_id, v_lote_code, v_dest_id, v_need_uv, v_need_uc;
      IF v_done=1 THEN LEAVE it_loop; END IF;

      /* Log inicio de item y necesidades */
      IF v_do_log = 1 THEN
        INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, take_uv, take_uc, msg)
        VALUES (p_pedido_codigo, v_pedido_id, v_pre_id, 'ITEM', v_item_id, v_prod_id, NULL, v_lote_code, v_need_uv, v_need_uc, 'NEEDS');
      END IF;

      /* Obtener factor UC->UV para el producto (0 si desconocido) */
  SELECT COALESCE(MAX(unidades_por_uv),0) INTO v_uc_per_uv FROM para_producto_pack WHERE producto_id = v_prod_id;

      /* 1) Consumir primero desde PICKING (FEFO) */
      DROP TEMPORARY TABLE IF EXISTS _cand_pick;
      CREATE TEMPORARY TABLE _cand_pick (
        posicion_id BIGINT UNSIGNED,
        lote_id     BIGINT UNSIGNED,
        pallet_id   BIGINT UNSIGNED,
        qty_uv      INT,
        qty_uc      INT,
        venc        DATE
      ) ENGINE=MEMORY;

      IF v_max_cands > 0 THEN
        INSERT INTO _cand_pick (posicion_id,lote_id,pallet_id,qty_uv,qty_uc,venc)
        SELECT s.posicion_id, s.lote_id, s.pallet_id, s.qty_uv, s.qty_uc, l.fecha_vencimiento
          FROM wh_stock s
          JOIN wh_posicion p   ON p.id=s.posicion_id
          JOIN wh_ambiente a   ON a.id=p.ambiente_id AND a.code='PICKING'
          LEFT JOIN wh_lote l  ON l.id=s.lote_id
         WHERE s.deposito_id = v_dep_id
           AND s.producto_id = v_prod_id
           AND (s.qty_uv > 0 OR s.qty_uc > 0)
         ORDER BY 
           /* Priorizar lote específico si está pedido */
           CASE WHEN v_lote_code IS NOT NULL AND v_lote_code <> '' AND l.codigo = v_lote_code THEN 1 ELSE 2 END,
           /* Luego FEFO (vencimiento) */
           ISNULL(l.fecha_vencimiento) ASC, l.fecha_vencimiento ASC
         LIMIT v_max_cands;
      ELSE
        INSERT INTO _cand_pick (posicion_id,lote_id,pallet_id,qty_uv,qty_uc,venc)
        SELECT s.posicion_id, s.lote_id, s.pallet_id, s.qty_uv, s.qty_uc, l.fecha_vencimiento
          FROM wh_stock s
          JOIN wh_posicion p   ON p.id=s.posicion_id
          JOIN wh_ambiente a   ON a.id=p.ambiente_id AND a.code='PICKING'
          LEFT JOIN wh_lote l  ON l.id=s.lote_id
         WHERE s.deposito_id = v_dep_id
           AND s.producto_id = v_prod_id
           AND (s.qty_uv > 0 OR s.qty_uc > 0)
         ORDER BY 
           /* Priorizar lote específico si está pedido */
           CASE WHEN v_lote_code IS NOT NULL AND v_lote_code <> '' AND l.codigo = v_lote_code THEN 1 ELSE 2 END,
           /* Luego FEFO (vencimiento) */
           ISNULL(l.fecha_vencimiento) ASC, l.fecha_vencimiento ASC;
      END IF;

  /* Log cantidad de candidatos en PICKING */
  IF v_do_log = 1 THEN
    INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, msg, take_uv, take_uc)
    SELECT p_pedido_codigo, v_pedido_id, v_pre_id, 'CAND_PICK', v_item_id, v_prod_id, NULL, v_lote_code, CONCAT('rows=',COUNT(*)), v_need_uv, v_need_uc FROM _cand_pick;
  END IF;

  SET v_it_pick = v_max_iters;
  consume_pick_loop: WHILE (v_need_uv > 0 OR v_need_uc > 0) DO
        SET v_it_pick = v_it_pick - 1;
        IF v_it_pick <= 0 THEN
          IF v_do_log = 1 THEN
            INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, msg)
            VALUES (p_pedido_codigo, v_pedido_id, v_pre_id, 'WARN', v_item_id, v_prod_id, NULL, v_lote_code, 'MAX_ITERS_PICK');
          END IF;
          LEAVE consume_pick_loop;
        END IF;
        SELECT posicion_id, lote_id, pallet_id, qty_uv, qty_uc
          INTO v_pos_from, v_lote_id, v_pallet_id, v_take_uv, v_take_uc
        FROM _cand_pick ORDER BY COALESCE(venc,'9999-12-31') ASC LIMIT 1;

        IF v_pos_from IS NULL THEN LEAVE consume_pick_loop; END IF;

          /* Revalidar disponibilidad exacta en la fila de stock */
          BEGIN
            DECLARE cur_uv INT DEFAULT 0; DECLARE cur_uc INT DEFAULT 0;
            SELECT qty_uv, qty_uc INTO cur_uv, cur_uc
              FROM wh_stock
             WHERE deposito_id=v_dep_id AND posicion_id=v_pos_from AND producto_id=v_prod_id AND lote_id=v_lote_id
               AND (pallet_id <=> v_pallet_id)
             LIMIT 1;
            /* Base amounts */
            SET v_take_uv_base = LEAST(COALESCE(cur_uv,0), v_need_uv);
            SET v_take_uc_base = LEAST(COALESCE(cur_uc,0), v_need_uc);
            SET v_take_uv = v_take_uv_base;
            SET v_take_uc = v_take_uc_base;
            /* Movement deltas start equal to base (no conversion yet) */
            SET v_mv_uv = v_take_uv_base;
            SET v_mv_uc = v_take_uc_base;
            SET v_break_uv = 0;
            SET v_break_uc = 0;
            /* If allowed, convert UC to UV to cover UV need when no UV available */
            IF v_uc_to_uv = 1 AND v_uc_per_uv > 0 AND (v_need_uv - v_take_uv) > 0 AND (COALESCE(cur_uv,0) - v_take_uv_base) = 0 AND COALESCE(cur_uc,0) > 0 THEN
              /* How many UV can we form from remaining UC? */
              SET @rem_uc := GREATEST(COALESCE(cur_uc,0) - v_take_uc_base, 0);
              SET @conv_uv := LEAST(v_need_uv - v_take_uv, FLOOR(@rem_uc / v_uc_per_uv));
              IF @conv_uv > 0 THEN
                SET v_take_uv = v_take_uv + @conv_uv;
                /* We move UC units for conversion, not UV */
                SET v_mv_uc = v_mv_uc + (@conv_uv * v_uc_per_uv);
              END IF;
            END IF;
            /* If allowed, convert UV to UC to cover UC need when no UC available (abrir cajas) */
            /* Modified UV→UC logic: Allow conversion when we still need UC and UV is available (less restrictive) */
            IF v_do_log = 1 THEN
              INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, msg)
              VALUES (p_pedido_codigo, v_pedido_id, v_pre_id, 'PICKING', v_item_id, v_prod_id, v_lote_id, v_lote_code, CONCAT('UV→UC DEBUG: v_uv_to_uc=', v_uv_to_uc, ', v_uc_per_uv=', v_uc_per_uv, ', need_uc_deficit=', (v_need_uc - v_take_uc), ', cur_uv=', COALESCE(cur_uv,0), ', v_take_uv_base=', v_take_uv_base));
            END IF;
            IF v_uv_to_uc = 1 AND v_uc_per_uv > 0 AND (v_need_uc - v_take_uc) > 0 AND COALESCE(cur_uv,0) > 0 THEN
              /* How many UV can we break to get UC? */
              SET @rem_uv := GREATEST(COALESCE(cur_uv,0) - v_take_uv_base, 0);
              SET @break_uv := LEAST(@rem_uv, CEIL((v_need_uc - v_take_uc) / v_uc_per_uv));
              IF v_do_log = 1 THEN
                INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, msg)
                VALUES (p_pedido_codigo, v_pedido_id, v_pre_id, 'PICKING', v_item_id, v_prod_id, v_lote_id, v_lote_code, CONCAT('UV→UC CALC: rem_uv=', @rem_uv, ', break_uv=', @break_uv, ', formula=CEIL(', (v_need_uc - v_take_uc), '/', v_uc_per_uv, ')'));
              END IF;
              IF @break_uv > 0 THEN
                SET v_break_uv = @break_uv;
                SET v_break_uc = @break_uv * v_uc_per_uv;
                SET v_take_uc = v_take_uc + v_break_uc;
                /* We move UV units for breaking into UC */
                SET v_mv_uv = v_mv_uv + v_break_uv;
                /* Log UV→UC conversion */
                IF v_do_log = 1 THEN
                  INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, from_pos_id, pallet_id, take_uv, take_uc, msg)
                  VALUES (p_pedido_codigo, v_pedido_id, v_pre_id, 'PICKING', v_item_id, v_prod_id, v_lote_id, v_lote_code, v_pos_from, v_pallet_id, v_break_uv, v_break_uc, CONCAT('UV→UC: Breaking ', v_break_uv, ' UV to get ', v_break_uc, ' UC'));
                END IF;
              ELSE
                IF v_do_log = 1 THEN
                  INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, msg)
                  VALUES (p_pedido_codigo, v_pedido_id, v_pre_id, 'PICKING', v_item_id, v_prod_id, v_lote_id, v_lote_code, 'UV→UC: Conditions not met for conversion');
                END IF;
              END IF;
            ELSE
              IF v_do_log = 1 THEN
                INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, msg)
                VALUES (p_pedido_codigo, v_pedido_id, v_pre_id, 'PICKING', v_item_id, v_prod_id, v_lote_id, v_lote_code, 'UV→UC: Conditions not met for conversion');
              END IF;
            END IF;
          END;

        IF v_take_uv=0 AND v_mv_uc=0 THEN
          DELETE FROM _cand_pick LIMIT 1; ITERATE consume_pick_loop;
        END IF;

        /* TRY pick: log antes del movimiento */
        IF v_do_log = 1 THEN
          INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, from_pos_id, to_pos_id, pallet_id, take_uv, take_uc, need_uv_post, need_uc_post, venc)
          VALUES (p_pedido_codigo, v_pedido_id, v_pre_id, 'TRY_PICK', v_item_id, v_prod_id, v_lote_id, v_lote_code, v_pos_from, v_pos_prep_id, v_pallet_id, v_take_uv, v_take_uc, v_need_uv, v_need_uc,
            (SELECT fecha_vencimiento FROM wh_lote WHERE id=v_lote_id LIMIT 1));
        END IF;

        IF p_simulate = 0 THEN
          INSERT INTO wh_move (deposito_id, tipo, motivo, pallet_id, producto_id, lote_id, from_pos_id, to_pos_id, delta_uv, delta_uc, referencia)
          VALUES (v_dep_id, 'MOVE', 'PREPARACION', v_pallet_id, v_prod_id, v_lote_id, v_pos_from, v_pos_prep_id, v_mv_uv, v_mv_uc, CONCAT('PRE-', v_pedido_id));
          SET v_move_id = LAST_INSERT_ID();
          INSERT INTO so_pre_pick (preembarque_id, pedido_dest_item_id, from_pos_id, to_pos_id, pallet_id, lote_id, uv_cajas, uc_unidades, creado_por)
          VALUES (v_pre_id, v_item_id, v_pos_from, v_pos_prep_id, v_pallet_id, v_lote_id, v_take_uv, v_take_uc, NULL);
      UPDATE so_pedido_dest_item
       SET prepared_uv = LEAST(COALESCE(expected_uv,0), COALESCE(prepared_uv,0) + v_take_uv),
         prepared_uc = LEAST(COALESCE(expected_uc,0), COALESCE(prepared_uc,0) + v_take_uc)
       WHERE id = v_item_id;
          IF v_break_uv > 0 THEN
            CALL sp_wh_stock_apply(v_dep_id, v_pos_prep_id, v_prod_id, v_lote_id, v_pallet_id, -v_break_uv, v_break_uc, 1);
            SET v_break_uv = 0;
            SET v_break_uc = 0;
          END IF;
        ELSE
          INSERT INTO _plan (fase,item_id,producto_id,lote_id,lote_code,from_pos_id,to_pos_id,pallet_id,take_uv,take_uc,venc)
          VALUES ('PICK',v_item_id,v_prod_id,v_lote_id,v_lote_code,v_pos_from,v_pos_prep_id,v_pallet_id,v_take_uv,v_mv_uc,(SELECT fecha_vencimiento FROM wh_lote WHERE id=v_lote_id LIMIT 1));
        END IF;

        /* log pick */
        IF v_do_log = 1 THEN
          INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, from_pos_id, to_pos_id, pallet_id, take_uv, take_uc, need_uv_post, need_uc_post, venc)
          VALUES (p_pedido_codigo, v_pedido_id, v_pre_id, 'PICK', v_item_id, v_prod_id, v_lote_id, v_lote_code, v_pos_from, v_pos_prep_id, v_pallet_id, v_take_uv, v_take_uc, v_need_uv - v_take_uv, v_need_uc - v_take_uc,
            (SELECT fecha_vencimiento FROM wh_lote WHERE id=v_lote_id LIMIT 1));
        END IF;

  SET v_need_uv = v_need_uv - v_take_uv;
  SET v_need_uc = v_need_uc - v_take_uc;

  UPDATE _cand_pick SET qty_uv=qty_uv - v_mv_uv, qty_uc=qty_uc - v_mv_uc
         WHERE posicion_id=v_pos_from AND lote_id=v_lote_id AND COALESCE(pallet_id,0)=COALESCE(v_pallet_id,0);
  DELETE FROM _cand_pick WHERE qty_uv<=0 AND qty_uc<=0 LIMIT 1;
      END WHILE;
      DROP TEMPORARY TABLE IF EXISTS _cand_pick;

  /* Fin consumo PICKING */
  IF v_do_log = 1 THEN
    INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, take_uv, take_uc, msg)
    VALUES (p_pedido_codigo, v_pedido_id, v_pre_id, 'END_PICK', v_item_id, v_prod_id, NULL, v_lote_code, v_need_uv, v_need_uc, 'REMAIN_AFTER_PICK');
  END IF;

      /* 2) Si falta, reponer pallet entero de otros ambientes hacia PICKING y volver a intentar */
  /* Log inicio reposición si aún hay necesidades */
  IF v_do_log = 1 THEN
    INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, take_uv, take_uc, msg)
    VALUES (p_pedido_codigo, v_pedido_id, v_pre_id, 'REPO_START', v_item_id, v_prod_id, NULL, v_lote_code, v_need_uv, v_need_uc, 'NEEDS_BEFORE_REPO');
  END IF;

  SET v_it_repo = v_max_iters;
  reponer_loop: WHILE (v_need_uv > 0 OR v_need_uc > 0) DO
        SET v_it_repo = v_it_repo - 1;
        IF v_it_repo <= 0 THEN
          IF v_do_log = 1 THEN
            INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, msg)
            VALUES (p_pedido_codigo, v_pedido_id, v_pre_id, 'WARN', v_item_id, v_prod_id, NULL, v_lote_code, 'MAX_ITERS_REPO');
          END IF;
          LEAVE reponer_loop;
        END IF;
        /* Buscar filas de stock en otros ambientes (no PREP/CUARENTENA/PICKING) por FEFO */
        DROP TEMPORARY TABLE IF EXISTS _src_rows;
        CREATE TEMPORARY TABLE _src_rows (
          from_pos_id BIGINT UNSIGNED,
          pallet_id   BIGINT UNSIGNED,
          lote_id     BIGINT UNSIGNED,
          qty_uv      INT,
          qty_uc      INT,
          venc        DATE
        ) ENGINE=MEMORY;

        IF v_max_cands > 0 THEN
          INSERT INTO _src_rows (from_pos_id, pallet_id, lote_id, qty_uv, qty_uc, venc)
          SELECT s.posicion_id, s.pallet_id, s.lote_id, s.qty_uv, s.qty_uc, l.fecha_vencimiento
            FROM wh_stock s
            JOIN wh_posicion p ON p.id=s.posicion_id
            JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code NOT IN ('PREP','CUARENTENA','PICKING')
            LEFT JOIN wh_lote l ON l.id=s.lote_id
           WHERE s.deposito_id=v_dep_id AND s.producto_id=v_prod_id
             AND (s.qty_uv>0 OR s.qty_uc>0)
           ORDER BY 
             /* Priorizar lote específico si está pedido */
             CASE WHEN v_lote_code IS NOT NULL AND v_lote_code <> '' AND l.codigo = v_lote_code THEN 1 ELSE 2 END,
             /* Luego FEFO (vencimiento) */
             ISNULL(l.fecha_vencimiento) ASC, l.fecha_vencimiento ASC
           LIMIT v_max_cands;
        ELSE
          INSERT INTO _src_rows (from_pos_id, pallet_id, lote_id, qty_uv, qty_uc, venc)
          SELECT s.posicion_id, s.pallet_id, s.lote_id, s.qty_uv, s.qty_uc, l.fecha_vencimiento
            FROM wh_stock s
            JOIN wh_posicion p ON p.id=s.posicion_id
            JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code NOT IN ('PREP','CUARENTENA','PICKING')
            LEFT JOIN wh_lote l ON l.id=s.lote_id
           WHERE s.deposito_id=v_dep_id AND s.producto_id=v_prod_id
             AND (s.qty_uv>0 OR s.qty_uc>0)
           ORDER BY 
             /* Priorizar lote específico si está pedido */
             CASE WHEN v_lote_code IS NOT NULL AND v_lote_code <> '' AND l.codigo = v_lote_code THEN 1 ELSE 2 END,
             /* Luego FEFO (vencimiento) */
             ISNULL(l.fecha_vencimiento) ASC, l.fecha_vencimiento ASC;
        END IF;

  /* Log cantidad de candidatos para reposición */
  IF v_do_log = 1 THEN
    INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, msg, take_uv, take_uc)
  SELECT p_pedido_codigo, v_pedido_id, v_pre_id, 'CAND_REPO', v_item_id, v_prod_id, NULL, v_lote_code, CONCAT('rows=',COUNT(*)), v_need_uv, v_need_uc FROM _src_rows;
  END IF;

        /* Tomar la mejor fila y mover parcial hacia PICKING */
        SELECT from_pos_id, pallet_id, lote_id, qty_uv, qty_uc
          INTO v_pos_from, v_pallet_id, v_lote_id, v_row_uv, v_row_uc
        FROM _src_rows LIMIT 1;
        DROP TEMPORARY TABLE IF EXISTS _src_rows;

        IF v_pos_from IS NULL THEN
          INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, msg)
          VALUES (p_pedido_codigo, v_pedido_id, v_pre_id, 'INFO', v_item_id, v_prod_id, NULL, v_lote_code, 'NO_MORE_SOURCE_FOR_REPOSICION');
          LEAVE reponer_loop; /* no hay más stock para reponer ESTE ítem, continuar con el siguiente */
        END IF;

        SET v_take_uv = LEAST(COALESCE(v_row_uv,0), v_need_uv);
        SET v_take_uc = LEAST(COALESCE(v_row_uc,0), v_need_uc);
        SET v_mv_uv = v_take_uv;
        SET v_mv_uc = v_take_uc;
        SET v_break_uv = 0;
        SET v_break_uc = 0;
        /* Allow conversion from remaining UC at source row */
        IF v_uc_to_uv = 1 AND v_uc_per_uv > 0 AND (v_need_uv - v_take_uv) > 0 THEN
          SET @rem_uc2 := GREATEST(COALESCE(v_row_uc,0) - v_take_uc, 0);
          SET @conv_uv2 := LEAST(v_need_uv - v_take_uv, FLOOR(@rem_uc2 / v_uc_per_uv));
          IF @conv_uv2 > 0 THEN
            SET v_take_uv = v_take_uv + @conv_uv2;
            SET v_mv_uc = v_mv_uc + (@conv_uv2 * v_uc_per_uv);
          END IF;
        END IF;
        /* Allow conversion from remaining UV at source row (abrir cajas) */
        IF v_do_log = 1 THEN
          INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, msg)
          VALUES (p_pedido_codigo, v_pedido_id, v_pre_id, 'REPOSICION', v_item_id, v_prod_id, v_lote_id, v_lote_code, CONCAT('UV→UC Check: v_uv_to_uc=', v_uv_to_uc, ', v_uc_per_uv=', v_uc_per_uv, ', need_uc=', (v_need_uc - v_take_uc), ', row_uv=', COALESCE(v_row_uv,0)));
        END IF;
        IF v_uv_to_uc = 1 AND v_uc_per_uv > 0 AND (v_need_uc - v_take_uc) > 0 THEN
          SET @rem_uv2 := GREATEST(COALESCE(v_row_uv,0) - v_take_uv, 0);
          SET @break_uv2 := LEAST(@rem_uv2, CEIL((v_need_uc - v_take_uc) / v_uc_per_uv));
          IF @break_uv2 > 0 THEN
            SET v_break_uv = @break_uv2;
            SET v_break_uc = @break_uv2 * v_uc_per_uv;
            SET v_take_uc = v_take_uc + v_break_uc;
            SET v_mv_uv = v_mv_uv + v_break_uv;
            /* Log UV→UC conversion */
            IF v_do_log = 1 THEN
              INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, from_pos_id, to_pos_id, pallet_id, take_uv, take_uc, msg)
              VALUES (p_pedido_codigo, v_pedido_id, v_pre_id, 'REPOSICION', v_item_id, v_prod_id, v_lote_id, v_lote_code, v_pos_from, v_pos_prep_id, v_pallet_id, v_break_uv, v_break_uc, CONCAT('UV→UC: Breaking ', v_break_uv, ' UV to get ', v_break_uc, ' UC'));
            END IF;
          END IF;
        END IF;
        IF v_take_uv=0 AND v_take_uc=0 THEN
          LEAVE reponer_loop; /* no se puede tomar nada de este ítem, continuar con el siguiente */
        END IF;

        IF v_direct = 1 THEN
          /* Directo a PREP: un solo move + registro de pick, sin pasar por PICKING */
          IF v_do_log = 1 THEN
            INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, from_pos_id, to_pos_id, pallet_id, take_uv, take_uc, need_uv_post, need_uc_post, venc)
            VALUES (p_pedido_codigo, v_pedido_id, v_pre_id, 'TRY_DIRECT', v_item_id, v_prod_id, v_lote_id, v_lote_code, v_pos_from, v_pos_prep_id, v_pallet_id, v_take_uv, v_take_uc, v_need_uv, v_need_uc,
              (SELECT fecha_vencimiento FROM wh_lote WHERE id=v_lote_id LIMIT 1));
          END IF;

          IF p_simulate = 0 THEN
            INSERT INTO wh_move (deposito_id, tipo, motivo, pallet_id, producto_id, lote_id, from_pos_id, to_pos_id, delta_uv, delta_uc, referencia)
            VALUES (v_dep_id, 'MOVE', 'PREPARACION', v_pallet_id, v_prod_id, v_lote_id, v_pos_from, v_pos_prep_id, v_mv_uv, v_mv_uc, CONCAT('PRE-', v_pedido_id));
            SET v_move_id = LAST_INSERT_ID();
            INSERT INTO so_pre_pick (preembarque_id, pedido_dest_item_id, from_pos_id, to_pos_id, pallet_id, lote_id, uv_cajas, uc_unidades, creado_por)
            VALUES (v_pre_id, v_item_id, v_pos_from, v_pos_prep_id, v_pallet_id, v_lote_id, v_take_uv, v_mv_uc, NULL);
          UPDATE so_pedido_dest_item
            SET prepared_uv = LEAST(COALESCE(expected_uv,0), COALESCE(prepared_uv,0) + v_take_uv),
               prepared_uc = LEAST(COALESCE(expected_uc,0), COALESCE(prepared_uc,0) + v_take_uc)
           WHERE id = v_item_id;
            IF v_break_uv > 0 THEN
              CALL sp_wh_stock_apply(v_dep_id, v_pos_prep_id, v_prod_id, v_lote_id, v_pallet_id, -v_break_uv, v_break_uc, 1);
              SET v_break_uv = 0;
              SET v_break_uc = 0;
            END IF;
          ELSE
            INSERT INTO _plan (fase,item_id,producto_id,lote_id,lote_code,from_pos_id,to_pos_id,pallet_id,take_uv,take_uc,venc)
            VALUES ('DIRECT',v_item_id,v_prod_id,v_lote_id,v_lote_code,v_pos_from,v_pos_prep_id,v_pallet_id,v_take_uv,v_mv_uc,(SELECT fecha_vencimiento FROM wh_lote WHERE id=v_lote_id LIMIT 1));
          END IF;

          IF v_do_log = 1 THEN
            INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, from_pos_id, to_pos_id, pallet_id, take_uv, take_uc, need_uv_post, need_uc_post, venc)
            VALUES (p_pedido_codigo, v_pedido_id, v_pre_id, 'DIRECT', v_item_id, v_prod_id, v_lote_id, v_lote_code, v_pos_from, v_pos_prep_id, v_pallet_id, v_take_uv, v_take_uc, v_need_uv - v_take_uv, v_need_uc - v_take_uc,
              (SELECT fecha_vencimiento FROM wh_lote WHERE id=v_lote_id LIMIT 1));
          END IF;

          SET v_need_uv = v_need_uv - v_take_uv;
          SET v_need_uc = v_need_uc - v_take_uc;
        ELSE
          /* Modo clásico: reponer a PICKING y luego hacer PICK2 hacia PREP */
          IF v_do_log = 1 THEN
            INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, from_pos_id, to_pos_id, pallet_id, take_uv, take_uc, need_uv_post, need_uc_post, venc)
            VALUES (p_pedido_codigo, v_pedido_id, v_pre_id, 'TRY_REPO', v_item_id, v_prod_id, v_lote_id, v_lote_code, v_pos_from, v_pos_pick_id, v_pallet_id, v_take_uv, v_take_uc, v_need_uv, v_need_uc,
              (SELECT fecha_vencimiento FROM wh_lote WHERE id=v_lote_id LIMIT 1));
          END IF;

          IF p_simulate = 0 THEN
            INSERT INTO wh_move (deposito_id, tipo, motivo, pallet_id, producto_id, lote_id, from_pos_id, to_pos_id, delta_uv, delta_uc, referencia)
            VALUES (v_dep_id, 'MOVE', 'REPOSICION_PICKING', v_pallet_id, v_prod_id, v_lote_id, v_pos_from, v_pos_pick_id, v_mv_uv, v_mv_uc, CONCAT('PREP-REP-', v_pedido_id));
            SET v_move_id = LAST_INSERT_ID();
            IF v_do_log = 1 THEN
              INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, from_pos_id, to_pos_id, pallet_id, take_uv, take_uc, need_uv_post, need_uc_post, venc)
              VALUES (p_pedido_codigo, v_pedido_id, v_pre_id, 'REPO', v_item_id, v_prod_id, v_lote_id, v_lote_code, v_pos_from, v_pos_pick_id, v_pallet_id, v_take_uv, v_take_uc, v_need_uv, v_need_uc,
                (SELECT fecha_vencimiento FROM wh_lote WHERE id=v_lote_id LIMIT 1));
            END IF;
          ELSE
            INSERT INTO _plan (fase,item_id,producto_id,lote_id,lote_code,from_pos_id,to_pos_id,pallet_id,take_uv,take_uc,venc)
            VALUES ('REPO',v_item_id,v_prod_id,v_lote_id,v_lote_code,v_pos_from,v_pos_pick_id,v_pallet_id,v_take_uv,v_take_uc,(SELECT fecha_vencimiento FROM wh_lote WHERE id=v_lote_id LIMIT 1));
          END IF;

          /* Intentar consumir nuevamente desde PICKING hacia PREP */
          DROP TEMPORARY TABLE IF EXISTS _cand_pick2;
          CREATE TEMPORARY TABLE _cand_pick2 (
            posicion_id BIGINT UNSIGNED,
            lote_id     BIGINT UNSIGNED,
            pallet_id   BIGINT UNSIGNED,
            qty_uv      INT,
            qty_uc      INT,
            venc        DATE
          ) ENGINE=MEMORY;

          IF v_max_cands > 0 THEN
            INSERT INTO _cand_pick2 (posicion_id,lote_id,pallet_id,qty_uv,qty_uc,venc)
            SELECT s.posicion_id, s.lote_id, s.pallet_id, s.qty_uv, s.qty_uc, l.fecha_vencimiento
              FROM wh_stock s
              JOIN wh_posicion p   ON p.id=s.posicion_id
              JOIN wh_ambiente a   ON a.id=p.ambiente_id AND a.code='PICKING'
              LEFT JOIN wh_lote l  ON l.id=s.lote_id
             WHERE s.deposito_id = v_dep_id
               AND s.producto_id = v_prod_id
               AND (s.qty_uv > 0 OR s.qty_uc > 0)
             ORDER BY 
               /* Priorizar lote específico si está pedido */
               CASE WHEN v_lote_code IS NOT NULL AND v_lote_code <> '' AND l.codigo = v_lote_code THEN 1 ELSE 2 END,
               /* Luego FEFO (vencimiento) */
               ISNULL(l.fecha_vencimiento) ASC, l.fecha_vencimiento ASC
             LIMIT v_max_cands;
          ELSE
            INSERT INTO _cand_pick2 (posicion_id,lote_id,pallet_id,qty_uv,qty_uc,venc)
            SELECT s.posicion_id, s.lote_id, s.pallet_id, s.qty_uv, s.qty_uc, l.fecha_vencimiento
              FROM wh_stock s
              JOIN wh_posicion p   ON p.id=s.posicion_id
              JOIN wh_ambiente a   ON a.id=p.ambiente_id AND a.code='PICKING'
              LEFT JOIN wh_lote l  ON l.id=s.lote_id
             WHERE s.deposito_id = v_dep_id
               AND s.producto_id = v_prod_id
               AND (s.qty_uv > 0 OR s.qty_uc > 0)
             ORDER BY 
               /* Priorizar lote específico si está pedido */
               CASE WHEN v_lote_code IS NOT NULL AND v_lote_code <> '' AND l.codigo = v_lote_code THEN 1 ELSE 2 END,
               /* Luego FEFO (vencimiento) */
               ISNULL(l.fecha_vencimiento) ASC, l.fecha_vencimiento ASC;
          END IF;

  SET v_it_pick2 = v_max_iters;
  consume_pick2_loop: WHILE (v_need_uv > 0 OR v_need_uc > 0) DO
            SET v_it_pick2 = v_it_pick2 - 1;
            IF v_it_pick2 <= 0 THEN
              IF v_do_log = 1 THEN
                INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, msg)
                VALUES (p_pedido_codigo, v_pedido_id, v_pre_id, 'WARN', v_item_id, v_prod_id, NULL, v_lote_code, 'MAX_ITERS_PICK2');
              END IF;
              LEAVE consume_pick2_loop;
            END IF;
            SELECT posicion_id, lote_id, pallet_id, qty_uv, qty_uc
              INTO v_pos_from, v_lote_id, v_pallet_id, v_row_uv, v_row_uc
            FROM _cand_pick2 ORDER BY COALESCE(venc,'9999-12-31') ASC LIMIT 1;

            IF v_pos_from IS NULL THEN LEAVE consume_pick2_loop; END IF;

            SET v_take_uv = LEAST(COALESCE(v_row_uv,0), v_need_uv);
            SET v_take_uc = LEAST(COALESCE(v_row_uc,0), v_need_uc);
            SET v_mv_uv = v_take_uv;
            SET v_mv_uc = v_take_uc;
            SET v_break_uv = 0;
            SET v_break_uc = 0;
            IF v_uc_to_uv = 1 AND v_uc_per_uv > 0 AND (v_need_uv - v_take_uv) > 0 THEN
              SET @rem_uc3 := GREATEST(COALESCE(v_row_uc,0) - v_take_uc, 0);
              SET @conv_uv3 := LEAST(v_need_uv - v_take_uv, FLOOR(@rem_uc3 / v_uc_per_uv));
              IF @conv_uv3 > 0 THEN
                SET v_take_uv = v_take_uv + @conv_uv3;
                SET v_mv_uc = v_mv_uc + (@conv_uv3 * v_uc_per_uv);
              END IF;
            END IF;
            /* Allow conversion from remaining UV at source row (abrir cajas) */
            IF v_do_log = 1 THEN
              INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, msg)
              VALUES (p_pedido_codigo, v_pedido_id, v_pre_id, 'PICK2', v_item_id, v_prod_id, v_lote_id, v_lote_code, CONCAT('UV→UC Check: v_uv_to_uc=', v_uv_to_uc, ', v_uc_per_uv=', v_uc_per_uv, ', need_uc=', (v_need_uc - v_take_uc), ', row_uv=', COALESCE(v_row_uv,0)));
            END IF;
            IF v_uv_to_uc = 1 AND v_uc_per_uv > 0 AND (v_need_uc - v_take_uc) > 0 THEN
              SET @rem_uv3 := GREATEST(COALESCE(v_row_uv,0) - v_take_uv, 0);
              SET @break_uv3 := LEAST(@rem_uv3, CEIL((v_need_uc - v_take_uc) / v_uc_per_uv));
              IF @break_uv3 > 0 THEN
                SET v_break_uv = @break_uv3;
                SET v_break_uc = @break_uv3 * v_uc_per_uv;
                SET v_take_uc = v_take_uc + v_break_uc;
                SET v_mv_uv = v_mv_uv + v_break_uv;
                /* Log UV→UC conversion */
                IF v_do_log = 1 THEN
                  INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, from_pos_id, to_pos_id, pallet_id, take_uv, take_uc, msg)
                  VALUES (p_pedido_codigo, v_pedido_id, v_pre_id, 'PICK2', v_item_id, v_prod_id, v_lote_id, v_lote_code, v_pos_from, v_pos_prep_id, v_pallet_id, v_break_uv, v_break_uc, CONCAT('UV→UC: Breaking ', v_break_uv, ' UV to get ', v_break_uc, ' UC'));
                END IF;
              END IF;
            END IF;

            IF v_take_uv=0 AND v_take_uc=0 THEN
              DELETE FROM _cand_pick2 LIMIT 1; ITERATE consume_pick2_loop;
            END IF;

            IF v_do_log = 1 THEN
              INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, from_pos_id, to_pos_id, pallet_id, take_uv, take_uc, need_uv_post, need_uc_post, venc)
              VALUES (p_pedido_codigo, v_pedido_id, v_pre_id, 'TRY_PICK2', v_item_id, v_prod_id, v_lote_id, v_lote_code, v_pos_from, v_pos_prep_id, v_pallet_id, v_take_uv, v_take_uc, v_need_uv, v_need_uc,
                (SELECT fecha_vencimiento FROM wh_lote WHERE id=v_lote_id LIMIT 1));
            END IF;

            IF p_simulate = 0 THEN
              INSERT INTO wh_move (deposito_id, tipo, motivo, pallet_id, producto_id, lote_id, from_pos_id, to_pos_id, delta_uv, delta_uc, referencia)
              VALUES (v_dep_id, 'MOVE', 'PREPARACION', v_pallet_id, v_prod_id, v_lote_id, v_pos_from, v_pos_prep_id, v_mv_uv, v_mv_uc, CONCAT('PRE-', v_pedido_id));
              SET v_move_id = LAST_INSERT_ID();
              INSERT INTO so_pre_pick (preembarque_id, pedido_dest_item_id, from_pos_id, to_pos_id, pallet_id, lote_id, uv_cajas, uc_unidades, creado_por)
              VALUES (v_pre_id, v_item_id, v_pos_from, v_pos_prep_id, v_pallet_id, v_lote_id, v_take_uv, v_take_uc, NULL);
            UPDATE so_pedido_dest_item
              SET prepared_uv = LEAST(COALESCE(expected_uv,0), COALESCE(prepared_uv,0) + v_take_uv),
                 prepared_uc = LEAST(COALESCE(expected_uc,0), COALESCE(prepared_uc,0) + v_take_uc)
             WHERE id = v_item_id;
              IF v_break_uv > 0 THEN
                CALL sp_wh_stock_apply(v_dep_id, v_pos_prep_id, v_prod_id, v_lote_id, v_pallet_id, -v_break_uv, v_break_uc, 1);
                SET v_break_uv = 0;
                SET v_break_uc = 0;
              END IF;
            ELSE
              INSERT INTO _plan (fase,item_id,producto_id,lote_id,lote_code,from_pos_id,to_pos_id,pallet_id,take_uv,take_uc,venc)
              VALUES ('PICK2',v_item_id,v_prod_id,v_lote_id,v_lote_code,v_pos_from,v_pos_prep_id,v_pallet_id,v_take_uv,v_mv_uc,(SELECT fecha_vencimiento FROM wh_lote WHERE id=v_lote_id LIMIT 1));
            END IF;

            IF v_do_log = 1 THEN
              INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, pre_id, fase, item_id, producto_id, lote_id, lote_code, from_pos_id, to_pos_id, pallet_id, take_uv, take_uc, need_uv_post, need_uc_post, venc)
              VALUES (p_pedido_codigo, v_pedido_id, v_pre_id, 'PICK2', v_item_id, v_prod_id, v_lote_id, v_lote_code, v_pos_from, v_pos_prep_id, v_pallet_id, v_take_uv, v_take_uc, v_need_uv - v_take_uv, v_need_uc - v_take_uc,
                (SELECT fecha_vencimiento FROM wh_lote WHERE id=v_lote_id LIMIT 1));
            END IF;

            SET v_need_uv = v_need_uv - v_take_uv;
            SET v_need_uc = v_need_uc - v_take_uc;

            UPDATE _cand_pick2 SET qty_uv=qty_uv - v_mv_uv, qty_uc=qty_uc - v_mv_uc LIMIT 1;
            DELETE FROM _cand_pick2 WHERE qty_uv<=0 AND qty_uc<=0 LIMIT 1;
          END WHILE;
          DROP TEMPORARY TABLE IF EXISTS _cand_pick2;
        END IF;

        IF NOT (v_need_uv > 0 OR v_need_uc > 0) THEN LEAVE reponer_loop; END IF;
      END WHILE;
    END LOOP;
    CLOSE cur_it;
  END;

  /* fin */

  IF p_simulate = 1 THEN
    SELECT * FROM _plan;
  END IF;

  SELECT d.pedido_id, i.id AS pedido_dest_item_id, i.producto_id, i.lote_codigo,
         i.expected_uv, i.prepared_uv, (i.expected_uv - i.prepared_uv) AS pend_uv,
         i.expected_uc, i.prepared_uc, (i.expected_uc - i.prepared_uc) AS pend_uc
  FROM so_pedido_dest_item i
  JOIN so_pedido_dest d ON d.id=i.pedido_dest_id
  WHERE d.pedido_id=v_pedido_id;
END
SQL;

    $pdo->exec($sql);

    // Ensure application users retain EXECUTE privilege even if the procedure
    // is recreated under a privileged definer.
    outln('Granting EXECUTE privilege on sp_so_preparar_auto ...');
    $dbName = '';
    try {
      $dbName = (string)$pdo->query('SELECT DATABASE()')->fetchColumn();
    } catch (Throwable $e) {
      $dbName = (string)($_ENV['DB_NAME'] ?? 'sol');
    }
    if ($dbName === '') {
      $dbName = (string)($_ENV['DB_NAME'] ?? 'sol');
    }

    $grantCombos = [];
    $appUser = trim((string)($_ENV['DB_USER'] ?? ''));
    if ($appUser !== '') {
      $hosts = [
        trim((string)($_ENV['DB_HOST'] ?? '')),
        'localhost',
        '127.0.0.1',
        '%'
      ];
      $hosts = array_values(array_unique(array_filter($hosts, static function ($v) {
        return $v !== '';
      })));
      foreach ($hosts as $host) {
        $grantCombos[$appUser . '@' . $host] = [$appUser, $host];
      }
    }

    $extraUsers = trim((string)($_ENV['SO_PREP_GRANT_USERS'] ?? ''));
    if ($extraUsers !== '') {
      foreach (explode(',', $extraUsers) as $raw) {
        $raw = trim($raw);
        if ($raw === '') {
          continue;
        }
        $parts = explode('@', $raw, 2);
        $user = trim($parts[0]);
        $host = isset($parts[1]) && trim($parts[1]) !== '' ? trim($parts[1]) : '%';
        if ($user === '') {
          continue;
        }
        $grantCombos[$user . '@' . $host] = [$user, $host];
      }
    }

    // Default fallback grants for legacy application user
    if ($appUser === '' || strcasecmp($appUser, 'arasa') !== 0) {
      foreach ([
        ['arasa', 'localhost'],
        ['arasa', '127.0.0.1'],
        ['arasa', '%']
      ] as [$user, $host]) {
        $grantCombos[$user . '@' . $host] = [$user, $host];
      }
    }

    foreach ($grantCombos as [$user, $host]) {
      $grantSql = sprintf(
        "GRANT EXECUTE ON PROCEDURE `%s`.`sp_so_preparar_auto` TO '%s'@'%s'",
        str_replace('`', '``', $dbName),
        str_replace("'", "''", $user),
        str_replace("'", "''", $host)
      );
      try {
        $pdo->exec($grantSql);
        outln(sprintf('  Granted EXECUTE to %s@%s', $user, $host));
      } catch (Throwable $e) {
        outln(sprintf('  Warning: could not grant to %s@%s (%s)', $user, $host, $e->getMessage()));
      }
    }

    outln('Procedure created successfully.');
    exit(0);
} catch (Throwable $e) {
    fwrite(STDERR, 'Error: ' . $e->getMessage() . PHP_EOL);
    exit(1);
}
