<?php
/**
 * Bulk auto-assign pallets in CUARENTENA to PICKING positions.
 * - Fills each PICKING position up to a capacity (default 20), then moves to the next.
 * - Only assigns pallets without a current position (posicion_id/pos_id IS NULL or 0).
 * - Recalculates wh_posicion.ocupado and picked at the end.
 *
 * Options (CLI args):
 *   --capacity=20   Max pallets per PICKING position
 *   --limit=0       Max pallets to assign (0 = no limit)
 *   --simulate=1    Don't write changes, just print the plan
 *   --include-with-position=1 Reubicar pallets que ya tienen posición (además de los sin posición)
 *   --change-state=1 Cambiar estado al salir de CUARENTENA (usa --target-state si se provee)
 *   --target-state=UBICADO Código de estado destino (por defecto se intenta UBICADO/DISPONIBLE/ALMACENADO/PICKING)
 *
 * Safe defaults: simulate off by default; use --simulate=1 to preview.
 */

declare(strict_types=1);

require_once __DIR__ . '/../config/db.php';

function getArg(string $name, $default = null) {
  foreach ($GLOBALS['argv'] ?? [] as $arg) {
    if (strpos($arg, "--{$name}=") === 0) {
      return substr($arg, strlen($name) + 3);
    }
  }
  return $default;
}

$capacity = (int) (getArg('capacity', '20'));
$limit    = (int) (getArg('limit', '0'));
$simulate = (int) (getArg('simulate', '0')) === 1;
$includeWithPos = (int) (getArg('include-with-position', '0')) === 1;
$changeState = (int) (getArg('change-state', '0')) === 1;
$targetStateCode = strtoupper(trim((string) getArg('target-state', '')));

$pdo = getPDO();
$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

// Helpers to detect schema
$hasTbl = function(PDO $pdo, string $t): bool {
  $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?");
  $st->execute([$t]);
  return (int)$st->fetchColumn() > 0;
};
$hasCol = function(PDO $pdo, string $tbl, string $col): bool {
  $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?");
  $st->execute([$tbl, $col]);
  return (int)$st->fetchColumn() > 0;
};

$palTbl = $hasTbl($pdo, 'wh_pallet') ? 'wh_pallet' : ($hasTbl($pdo, 'wh_pallets') ? 'wh_pallets' : null);
if (!$palTbl) {
  fwrite(STDERR, "No se encontró la tabla de pallets (wh_pallet/wh_pallets).\n");
  exit(1);
}
$posCol = null;
if ($hasCol($pdo, $palTbl, 'posicion_id')) $posCol = 'posicion_id';
elseif ($hasCol($pdo, $palTbl, 'pos_id'))   $posCol = 'pos_id';
if (!$posCol) {
  fwrite(STDERR, "No se encontró columna de posición en {$palTbl} (posicion_id/pos_id).\n");
  exit(1);
}
$hasDeleted = $hasCol($pdo, $palTbl, 'deleted_at');
$hasDepoPal = $hasCol($pdo, $palTbl, 'deposito_id');

// Estado CUARENTENA (legacy). Si no existe, seguimos igual: seleccionamos pallets con posicion NULL.
$cuaId = 0;
try {
  $cuaId = (int)($pdo->query("SELECT id FROM wh_pallet_estado WHERE code='CUARENTENA' LIMIT 1")->fetchColumn() ?: 0);
} catch (Throwable $e) { $cuaId = 0; }

// Posiciones de PICKING activas
$hasDepoPos = $hasCol($pdo, 'wh_posicion', 'deposito_id');
$orderExpr  = "COALESCE(pos.code_full, pos.code, LPAD(pos.id,6,'0'))";
$sqlPos = "SELECT pos.id, " . ($hasDepoPos ? "pos.deposito_id" : "NULL AS deposito_id") . " AS deposito_id
            FROM wh_posicion pos
            JOIN wh_ambiente a ON a.id = pos.ambiente_id AND a.code='PICKING'
           WHERE (pos.activo = 1 OR pos.activo IS NULL)
           ORDER BY {$orderExpr}, pos.id";
$pickPositions = $pdo->query($sqlPos)->fetchAll(PDO::FETCH_ASSOC);
if (!$pickPositions) {
  fwrite(STDERR, "No se encontraron posiciones de PICKING activas.\n");
  exit(1);
}

// Pallets candidatos: en cuarentena (si existe estado) y sin posición (a menos que se incluya reubicación)
$where = [];
$params = [];
if (!$includeWithPos) { $where[] = "(p.{$posCol} IS NULL OR p.{$posCol} = 0)"; }
if ($cuaId) {
  $where[] = "p.estado_id = ?"; $params[] = $cuaId;
}
if ($hasDeleted) {
  $where[] = "p.deleted_at IS NULL";
}
$sqlPal = "SELECT p.id" . ($hasDepoPal ? ", p.deposito_id" : ", NULL AS deposito_id") . " AS deposito_id
            FROM {$palTbl} p
           WHERE " . implode(' AND ', $where) . "
           ORDER BY p.id ASC";
$palRows = $pdo->prepare($sqlPal);
$palRows->execute($params);
$palRows = $palRows->fetchAll(PDO::FETCH_ASSOC);
if (!$palRows) {
  echo "No hay pallets candidatos (sin posición" . ($cuaId ? " y en CUARENTENA" : "") . "). Nada que hacer.\n";
  exit(0);
}

// Armar mapa de ocupación actual por posición
$stCount = $pdo->prepare("SELECT COUNT(*) FROM {$palTbl} WHERE {$posCol} = ?" . ($hasDeleted ? " AND deleted_at IS NULL" : ""));
$posInfo = [];
foreach ($pickPositions as $pos) {
  $stCount->execute([(int)$pos['id']]);
  $cur = (int)$stCount->fetchColumn();
  $posInfo[(int)$pos['id']] = [
    'depo' => $hasDepoPos ? (int)$pos['deposito_id'] : null,
    'current' => $cur,
    'remaining' => max(0, $capacity - $cur),
  ];
}

// Agrupar pallets por depósito si corresponde
$palByDepo = [];
foreach ($palRows as $r) {
  $d = $hasDepoPal ? (int)$r['deposito_id'] : -1; // -1 = sin control de depósito
  $palByDepo[$d][] = (int)$r['id'];
}

$totalAssigned = 0;
$assignments = []; // posId => [palletIds]

foreach ($pickPositions as $pos) {
  $posId = (int)$pos['id'];
  $remain = $posInfo[$posId]['remaining'];
  if ($remain <= 0) continue;
  $depo = $posInfo[$posId]['depo'];
  $bucketKey = $hasDepoPal && $hasDepoPos ? $depo : (-1);
  if (!isset($palByDepo[$bucketKey]) || empty($palByDepo[$bucketKey])) continue;

  $take = min($remain, ($limit > 0 ? $limit - $totalAssigned : PHP_INT_MAX));
  if ($take <= 0) break;

  $takeIds = array_splice($palByDepo[$bucketKey], 0, $take);
  if (!$takeIds) continue;

  $assignments[$posId] = ($assignments[$posId] ?? []);
  foreach ($takeIds as $pid) { $assignments[$posId][] = $pid; }
  $totalAssigned += count($takeIds);
  if ($limit > 0 && $totalAssigned >= $limit) break;
}

if ($totalAssigned === 0) {
  echo "No hay capacidad disponible en posiciones PICKING para los pallets candidatos.\n";
  exit(0);
}

// Mostrar plan
echo "Plan de asignación (capacidad {$capacity}, simulate=" . ($simulate ? '1' : '0') . ")\n";
foreach ($assignments as $posId => $ids) {
  echo " - Pos #{$posId}: asignar " . count($ids) . " pallets\n";
}

echo "Total pallets a asignar: {$totalAssigned}\n";

if ($simulate) {
  echo "Simulación activa: no se aplicaron cambios.\n";
  exit(0);
}

$pdo->beginTransaction();
try {
  // Aplicar updates en batch por posición
  $stUpdate = null;
  foreach ($assignments as $posId => $ids) {
    // Chunks para evitar IN enormes
    $chunks = array_chunk($ids, 500);
    foreach ($chunks as $chunk) {
      $placeholders = implode(',', array_fill(0, count($chunk), '?'));
      $sqlUpd = "UPDATE {$palTbl} SET {$posCol} = ? WHERE id IN ({$placeholders})" . ($hasDeleted ? " AND deleted_at IS NULL" : "");
      $stUpdate = $pdo->prepare($sqlUpd);
      $params = array_merge([$posId], $chunk);
      $stUpdate->execute($params);
    }
  }

  // Recalcular flags por posición tocada
  $stCnt = $pdo->prepare("SELECT COUNT(*) FROM {$palTbl} WHERE {$posCol} = ?" . ($hasDeleted ? " AND deleted_at IS NULL" : ""));
  $stUpdPos = $pdo->prepare("UPDATE wh_posicion SET ocupado = ?, picked = ? WHERE id = ?");
  foreach (array_keys($assignments) as $posId) {
    $stCnt->execute([$posId]);
    $cnt = (int)$stCnt->fetchColumn();
    $ocupado = $cnt > 0 ? 1 : 0;
    $picked  = $cnt > 1 ? 1 : 0;
    $stUpdPos->execute([$ocupado, $picked, $posId]);
  }

  // Cambiar estado si corresponde (desde CUARENTENA a destino)
  if ($changeState && $cuaId) {
    // Resolver estado destino
    $targetId = 0; $resolvedCode = $targetStateCode;
    try {
      if ($targetStateCode) {
        $st = $pdo->prepare("SELECT id FROM wh_pallet_estado WHERE UPPER(code)=? LIMIT 1");
        $st->execute([$targetStateCode]); $targetId = (int)($st->fetchColumn() ?: 0);
      }
      if ($targetId <= 0) {
        foreach (['UBICADO','DISPONIBLE','ALMACENADO','PICKING'] as $cc) {
          $st = $pdo->prepare("SELECT id FROM wh_pallet_estado WHERE code=? LIMIT 1");
          $st->execute([$cc]); $targetId = (int)($st->fetchColumn() ?: 0);
          if ($targetId>0) { $resolvedCode = $cc; break; }
        }
      }
    } catch (Throwable $e) { $targetId = 0; }
    if ($targetId>0) {
      $allIds = [];
      foreach ($assignments as $ids) { foreach ($ids as $id) $allIds[] = (int)$id; }
      if ($allIds) {
        foreach (array_chunk($allIds, 500) as $chunk) {
          $qs = implode(',', array_fill(0, count($chunk), '?'));
          $sql = "UPDATE {$palTbl} SET estado_id = ? WHERE estado_id = ? AND id IN ({$qs})" . ($hasDeleted ? " AND deleted_at IS NULL" : "");
          $st = $pdo->prepare($sql);
          $st->execute(array_merge([$targetId, $cuaId], $chunk));
        }
      }
      echo "Estados actualizados a {$resolvedCode}.\n";
    }
  }

  $pdo->commit();
  echo "Asignación aplicada correctamente. Total pallets asignados: {$totalAssigned}.\n";
} catch (Throwable $e) {
  $pdo->rollBack();
  fwrite(STDERR, "Error aplicando asignación: " . $e->getMessage() . "\n");
  exit(1);
}
