<?php
declare(strict_types=1);

/**
 * Marca pre-embarques como COMPLETADO y genera tiempos/tareas básicos.
 *
 * Uso:
 *  php scripts/complete_preembarques.php --like=SO-AUTO-%
 *  php scripts/complete_preembarques.php --like=SO-20251105-% --limit=5 --dry-run
 */

$ROOT = dirname(__DIR__);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

$options = getopt('', [
    'like::',
    'limit::',
    'dry-run',
]);

$likePattern = $options['like'] ?? 'SO-AUTO-%';
$limit = isset($options['limit']) ? max(0, (int)$options['limit']) : 0;
$dryRun = array_key_exists('dry-run', $options);

$pdo = get_pdo();
$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
$pdo->exec('SET NAMES utf8mb4');

$estadoCompletado = getPreEstadoId($pdo, 'COMPLETADO');
if ($estadoCompletado <= 0) {
    fwrite(STDERR, "No se encontró estado COMPLETADO en so_preembarque_estado.\n");
    exit(1);
}

$sql = "SELECT pre.id, pre.codigo, pre.estado_id, pre.pedido_id, pre.deposito_id, pre.asignado_at, pre.inicio_at, pre.fin_at,
               ped.codigo AS pedido_codigo, ped.fecha_pedido
        FROM so_preembarque pre
        JOIN so_pedido ped ON ped.id = pre.pedido_id
        WHERE ped.codigo LIKE :like
          AND pre.estado_id <> :estado
        ORDER BY ped.fecha_pedido ASC, pre.id ASC";
if ($limit > 0) {
    $sql .= ' LIMIT ' . $limit;
}
$st = $pdo->prepare($sql);
$st->execute([
    ':like' => $likePattern,
    ':estado' => $estadoCompletado,
]);
$pres = $st->fetchAll(PDO::FETCH_ASSOC);

if (!$pres) {
    echo "No hay pre-embarques pendientes que coincidan con {$likePattern}.\n";
    exit(0);
}

$prepTask = TaskHelper::instance($pdo);
$summary = [
    'updated' => 0,
    'skipped_already_done' => 0,
    'rows' => [],
];

foreach ($pres as $index => $pre) {
    $preId = (int)$pre['id'];
    $pedidoCodigo = (string)$pre['pedido_codigo'];

    $timeSlots = buildTimeSlots((string)$pre['fecha_pedido'], $index);
    $summary['rows'][] = [
        'pre_id' => $preId,
        'pre_codigo' => $pre['codigo'],
        'pedido_codigo' => $pedidoCodigo,
        'asignado' => $timeSlots['asignado'],
        'fin' => $timeSlots['ctrl_fin'],
    ];

    if ($dryRun) {
        continue;
    }

    // Update pre-embarque core timestamps/state
    $updatePre = $pdo->prepare('UPDATE so_preembarque
        SET estado_id = :estado,
            asignado_at = :asignado,
            inicio_at = :prep_inicio,
            fin_at = :ctrl_fin,
            observacion = :obs
        WHERE id = :id');
    $updatePre->execute([
        ':estado' => $estadoCompletado,
        ':asignado' => $timeSlots['asignado'],
        ':prep_inicio' => $timeSlots['prep_inicio'],
        ':ctrl_fin' => $timeSlots['ctrl_fin'],
        ':obs' => 'Ciclo completado automáticamente',
        ':id' => $preId,
    ]);

    // Ensure PREPARACION task
    $prepTaskId = $prepTask->ensure($preId, 'PREPARACION');
    $prepTask->updateTimes($prepTaskId, $timeSlots['prep_inicio'], $timeSlots['prep_fin']);

    // Ensure CONTROL task
    $ctrlTaskId = $prepTask->ensure($preId, 'CONTROL');
    $prepTask->updateTimes($ctrlTaskId, $timeSlots['ctrl_inicio'], $timeSlots['ctrl_fin']);

    $summary['updated']++;
}

foreach ($summary['rows'] as $row) {
    echo sprintf(
        "Pre %s (%s) -> asignado %s, fin %s\n",
        $row['pre_codigo'],
        $row['pedido_codigo'],
        $row['asignado'],
        $row['fin']
    );
}

if ($dryRun) {
    echo "Dry-run: no se aplicaron cambios.\n";
} else {
    echo "Pre-embarques completados: {$summary['updated']}\n";
}

disableQueryCache($pdo);

final class TaskHelper
{
    private PDO $pdo;

    private function __construct(PDO $pdo)
    {
        $this->pdo = $pdo;
    }

    public static function instance(PDO $pdo): self
    {
        static $instance = null;
        if ($instance === null) {
            $instance = new self($pdo);
        }
        return $instance;
    }

    public function ensure(int $preId, string $tipo): int
    {
        $tipo = strtoupper($tipo);
        $stmt = $this->pdo->prepare('SELECT id FROM so_pre_tarea WHERE preembarque_id = ? AND tipo = ? LIMIT 1');
        $stmt->execute([$preId, $tipo]);
        $id = (int)$stmt->fetchColumn();
        if ($id > 0) {
            return $id;
        }
        $ins = $this->pdo->prepare('INSERT INTO so_pre_tarea (preembarque_id, tipo) VALUES (?, ?)');
        $ins->execute([$preId, $tipo]);
        return (int)$this->pdo->lastInsertId();
    }

    public function updateTimes(int $tareaId, string $inicio, string $fin): void
    {
        $stmt = $this->pdo->prepare('UPDATE so_pre_tarea SET inicio_at = :inicio, fin_at = :fin WHERE id = :id');
        $stmt->execute([
            ':inicio' => $inicio,
            ':fin' => $fin,
            ':id' => $tareaId,
        ]);
    }
}

function getPreEstadoId(PDO $pdo, string $code): int
{
    $stmt = $pdo->prepare('SELECT id FROM so_preembarque_estado WHERE code = ? LIMIT 1');
    $stmt->execute([$code]);
    return (int)$stmt->fetchColumn();
}

function buildTimeSlots(string $fechaBase, int $index): array
{
    $base = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', $fechaBase . ' 08:00:00');
    if (!$base) {
        $base = new DateTimeImmutable('now');
    }
    $offsetMinutes = $index * 5;
    $asignado = $base->modify("+{$offsetMinutes} minutes");
    $prepInicio = $asignado->modify('+30 minutes');
    $prepFin = $prepInicio->modify('+45 minutes');
    $ctrlInicio = $prepFin->modify('+10 minutes');
    $ctrlFin = $ctrlInicio->modify('+20 minutes');

    return [
        'asignado' => $asignado->format('Y-m-d H:i:s'),
        'prep_inicio' => $prepInicio->format('Y-m-d H:i:s'),
        'prep_fin' => $prepFin->format('Y-m-d H:i:s'),
        'ctrl_inicio' => $ctrlInicio->format('Y-m-d H:i:s'),
        'ctrl_fin' => $ctrlFin->format('Y-m-d H:i:s'),
    ];
}

function disableQueryCache(PDO $pdo): void
{
    try {
        $pdo->query('SET SESSION query_cache_type = OFF');
    } catch (Throwable $e) {
        // ignore if not supported
    }
}
