<?php
declare(strict_types=1);

/**
 * Genera embarques EN_COLA a partir de pre-embarques completados.
 *
 * Uso:
 *   php scripts/create_embarques_from_pre.php --like=SO-AUTO-%
 *   php scripts/create_embarques_from_pre.php --like=SO-20251105-% --max-pre=3 --dry-run
 */

$ROOT = dirname(__DIR__);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

$options = getopt('', [
    'like::',
    'max-pre::',
    'deposit-code::',
    'dry-run',
]);

$likePattern = $options['like'] ?? 'SO-AUTO-%';
$maxPer = isset($options['max-pre']) ? max(1, (int)$options['max-pre']) : 1;
$depositCode = $options['deposit-code'] ?? 'DEP1';
$dryRun = array_key_exists('dry-run', $options);

$pdo = get_pdo();
$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
$pdo->exec('SET NAMES utf8mb4');

$estadoPreCompletado = fetchId($pdo, 'SELECT id FROM so_preembarque_estado WHERE code=? LIMIT 1', ['COMPLETADO']);
$estadoEnCola = fetchId($pdo, 'SELECT id FROM so_embarque_estado WHERE code=? LIMIT 1', ['EN_COLA']);
if ($estadoPreCompletado <= 0 || $estadoEnCola <= 0) {
    fwrite(STDERR, "Estados necesarios no encontrados (COMPLETADO o EN_COLA).\n");
    exit(1);
}

$depositoCache = new DepositoCache($pdo, $depositCode);
$preRows = fetchPreembarques($pdo, $likePattern, $estadoPreCompletado);
if (!$preRows) {
    echo "No hay pre-embarques completados pendientes de embarque para {$likePattern}.\n";
    exit(0);
}

$groups = array_chunk($preRows, $maxPer);
$summary = [];

if ($dryRun) {
    foreach ($groups as $grpIndex => $group) {
        $preCodes = array_column($group, 'pre_codigo');
        $summary[] = [
            'embarque_codigo' => '(preview)',
            'pedido_codigos' => array_column($group, 'pedido_codigo'),
            'pre_codigos' => $preCodes,
        ];
    }
} else {
    $pdo->beginTransaction();
    try {
        $codeGenerator = new EmbarqueCodeGenerator($pdo);
        foreach ($groups as $group) {
            $depositoId = resolveDepositoId($depositoCache, $group);
            $codigo = $codeGenerator->next();
            $creadoAt = buildCreadoAt($group[0]['fecha_pedido']);

            $insertEmb = $pdo->prepare('INSERT INTO so_embarque (codigo, deposito_id, estado_id, creado_at) VALUES (?,?,?,?)');
            $insertEmb->execute([$codigo, $depositoId, $estadoEnCola, $creadoAt]);
            $embarqueId = (int)$pdo->lastInsertId();

            $insertLink = $pdo->prepare('INSERT INTO so_embarque_pre (embarque_id, preembarque_id) VALUES (?, ?)');
            foreach ($group as $row) {
                $insertLink->execute([$embarqueId, (int)$row['pre_id']]);
                createParadas($pdo, $embarqueId, (int)$row['pedido_id']);
            }

            $summary[] = [
                'embarque_codigo' => $codigo,
                'pedido_codigos' => array_column($group, 'pedido_codigo'),
                'pre_codigos' => array_column($group, 'pre_codigo'),
            ];
        }
        $pdo->commit();
    } catch (Throwable $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        fwrite(STDERR, 'No se pudieron crear los embarques: ' . $e->getMessage() . "\n");
        exit(1);
    }
}

foreach ($summary as $row) {
    echo sprintf(
        "Embarque %s -> pedidos [%s], pres [%s]\n",
        $row['embarque_codigo'],
        implode(', ', $row['pedido_codigos']),
        implode(', ', $row['pre_codigos'])
    );
}

if ($dryRun) {
    echo "Dry-run: no se insertaron embarques.\n";
}

final class DepositoCache
{
    private PDO $pdo;
    private string $fallbackCode;
    private array $cache = [];

    public function __construct(PDO $pdo, string $fallbackCode)
    {
        $this->pdo = $pdo;
        $this->fallbackCode = $fallbackCode;
    }

    public function resolve(?int $depositoId): int
    {
        if ($depositoId && $depositoId > 0) {
            return $depositoId;
        }
        if (isset($this->cache[$this->fallbackCode])) {
            return $this->cache[$this->fallbackCode];
        }
        $stmt = $this->pdo->prepare('SELECT id FROM wh_deposito WHERE code = ? LIMIT 1');
        $stmt->execute([$this->fallbackCode]);
        $id = (int)$stmt->fetchColumn();
        if ($id <= 0) {
            throw new RuntimeException('No se encontró deposito con code=' . $this->fallbackCode);
        }
        $this->cache[$this->fallbackCode] = $id;
        return $id;
    }
}

final class EmbarqueCodeGenerator
{
    private PDO $pdo;
    private string $prefix;
    private int $counter;

    public function __construct(PDO $pdo)
    {
        $this->pdo = $pdo;
        $this->prefix = 'EMB-' . date('Ym') . '-';
        $this->counter = $this->fetchCurrentMax();
    }

    public function next(): string
    {
        $this->counter++;
        return $this->prefix . str_pad((string)$this->counter, 4, '0', STR_PAD_LEFT);
    }

    private function fetchCurrentMax(): int
    {
        $stmt = $this->pdo->prepare("SELECT MAX(CAST(SUBSTRING(codigo, 12) AS UNSIGNED)) FROM so_embarque WHERE codigo LIKE ?");
        $stmt->execute([$this->prefix . '%']);
        return (int)$stmt->fetchColumn();
    }
}

function fetchPreembarques(PDO $pdo, string $likePattern, int $estadoCompletado): array
{
    $stmt = $pdo->prepare(
        "SELECT pre.id AS pre_id, pre.codigo AS pre_codigo, pre.deposito_id,
                ped.id AS pedido_id, ped.codigo AS pedido_codigo, ped.fecha_pedido
         FROM so_preembarque pre
         JOIN so_pedido ped ON ped.id = pre.pedido_id
         WHERE ped.codigo LIKE :like
           AND pre.estado_id = :estado
           AND NOT EXISTS (
                 SELECT 1 FROM so_embarque_pre ep WHERE ep.preembarque_id = pre.id
           )
         ORDER BY ped.fecha_pedido ASC, pre.id ASC"
    );
    $stmt->execute([
        ':like' => $likePattern,
        ':estado' => $estadoCompletado,
    ]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
}

function resolveDepositoId(DepositoCache $cache, array $group): int
{
    foreach ($group as $row) {
        if (!empty($row['deposito_id'])) {
            return $cache->resolve((int)$row['deposito_id']);
        }
    }
    return $cache->resolve(null);
}

function createParadas(PDO $pdo, int $embarqueId, int $pedidoId): void
{
    $destQuery = $pdo->prepare('SELECT id, destinatario_id, doc_tipo, doc_numero FROM so_pedido_dest WHERE pedido_id = ? ORDER BY id');
    $destQuery->execute([$pedidoId]);
    $destinatarios = $destQuery->fetchAll(PDO::FETCH_ASSOC);
    if (!$destinatarios) {
        return;
    }

    $insertParada = $pdo->prepare('INSERT INTO so_embarque_parada (embarque_id, destinatario_id, pedido_dest_id, orden, observacion) VALUES (?,?,?,?,?)');
    $insertDoc = $pdo->prepare('INSERT INTO so_parada_doc (parada_id, doc_tipo, doc_numero) VALUES (?,?,?)');

    $orden = 1;
    foreach ($destinatarios as $dest) {
        $observacion = 'Auto-generado para ' . ($dest['doc_numero'] ?? 'DEST');
        $insertParada->execute([$embarqueId, (int)$dest['destinatario_id'], (int)$dest['id'], $orden, $observacion]);
        $paradaId = (int)$pdo->lastInsertId();
        if (!empty($dest['doc_tipo']) && !empty($dest['doc_numero'])) {
            $insertDoc->execute([$paradaId, $dest['doc_tipo'], $dest['doc_numero']]);
        }
        $orden++;
    }
}

function fetchId(PDO $pdo, string $sql, array $params): int
{
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    return (int)$stmt->fetchColumn();
}

function buildCreadoAt(string $fechaPedido): string
{
    $base = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', $fechaPedido . ' 07:30:00');
    if (!$base) {
        $base = new DateTimeImmutable('now');
    }
    return $base->format('Y-m-d H:i:s');
}
