<#
scripts/create_sanitized_export.ps1

Creates a sanitized zip of the project root excluding heavy or sensitive folders:
 - vendor/
 - node_modules/
 - public/assets/
 - public/assets1/
 - .git/
 - .env and .env.*

Usage (PowerShell):
  cd <project-root>
  .\scripts\create_sanitized_export.ps1

This will create `sol-sanitized-<timestamp>.zip` in the project root.
#>

param(
  [string]$OutputName = "sol-sanitized",
  [switch]$ShowOnly
)

$root = (Get-Location).ProviderPath
$ts = Get-Date -Format yyyyMMddHHmmss
$temp = Join-Path -Path $env:TEMP -ChildPath "sol_export_$ts"
$zipPath = Join-Path -Path $root -ChildPath "$OutputName-$ts.zip"

Write-Host "Creating temporary export folder: $temp"
New-Item -Path $temp -ItemType Directory -Force | Out-Null

# Use robocopy to mirror excluding directories and env files (robocopy exists on Windows)
$excludesDirs = @("vendor","node_modules","public\\assets","public\\assets1",".git")
$excludeFiles = @(".env",".env.*")

$robocopyArgs = @(
  '"' + $root + '"',
  '"' + $temp + '"',
  '/E',                # copy all subdirectories including empty ones
  '/NFL', '/NDL',      # reduce logging
  '/NJH', '/NJS'
)

foreach ($d in $excludesDirs) { $robocopyArgs += "/XD"; $robocopyArgs += '"' + (Join-Path $root $d) + '"' }
foreach ($f in $excludeFiles) { $robocopyArgs += "/XF"; $robocopyArgs += $f }

$cmd = "robocopy " + ($robocopyArgs -join ' ')
Write-Host "Running: $cmd"
if (-not $ShowOnly) {
  $proc = Start-Process -FilePath robocopy -ArgumentList ($robocopyArgs) -NoNewWindow -Wait -PassThru
  if ($proc.ExitCode -ge 8) {
    Write-Warning "robocopy returned exit code $($proc.ExitCode). This usually indicates some files were not copied (see robocopy docs). Continuing anyway."
  }
}
else { Write-Host "(ShowOnly) Would run: $cmd" }

# Remove any leftover .git folder just in case
$gitTemp = Join-Path $temp '.git'
if (Test-Path $gitTemp) { Remove-Item -Path $gitTemp -Recurse -Force }

# Create zip
Write-Host "Creating zip: $zipPath"
if (-not $ShowOnly) {
  if (Test-Path $zipPath) { Remove-Item $zipPath -Force }
  Compress-Archive -Path (Join-Path $temp '*') -DestinationPath $zipPath -Force
  Write-Host "Created: $zipPath"
  Write-Host "Cleaning temporary folder: $temp"
  Remove-Item -Path $temp -Recurse -Force
} else {
  Write-Host "(ShowOnly) Would compress folder: $temp -> $zipPath"
}

Write-Host "Done." -ForegroundColor Green
