<?php
declare(strict_types=1);

$ROOT = dirname(__DIR__);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

$options = getopt('', ['format::', 'output::', 'compare-schema::', 'tables::']);
$format = isset($options['format']) ? strtolower((string)$options['format']) : 'markdown';
$outputPath = $options['output'] ?? null;
$compareSchema = isset($options['compare-schema']) ? trim((string)$options['compare-schema']) : null;
$tableFilter = null;
if (isset($options['tables'])) {
    $tableFilter = array_values(array_filter(array_map(static function ($value) {
        return trim((string)$value);
    }, explode(',', (string)$options['tables'])), static function ($value) {
        return $value !== '';
    }));
    if ($tableFilter === []) {
        $tableFilter = null;
    }
}

try {
    $pdo = get_pdo();
} catch (Throwable $connectionError) {
    fwrite(STDERR, 'Connection failed: ' . $connectionError->getMessage() . PHP_EOL);
    exit(1);
}

try {
    $dbNameStmt = $pdo->query('SELECT DATABASE() as schema_name');
    $dbRow = $dbNameStmt->fetch(PDO::FETCH_ASSOC) ?: [];
    $schemaName = $dbRow['schema_name'] ?? ($_ENV['DB_NAME'] ?? '');

    if ($schemaName === '') {
        throw new RuntimeException('Unable to resolve database name. Check configuration.');
    }

    $currentSnapshot = fetchSchemaSnapshot($pdo, $schemaName, $tableFilter);

    $schemaMeta = $currentSnapshot['meta'];
    $tables = $currentSnapshot['tables'];
    $columns = $currentSnapshot['columns'];
    $constraints = $currentSnapshot['constraints'];
    $foreignKeys = $currentSnapshot['foreign_keys'];
    $columnsByTable = $currentSnapshot['columns_by_table'];
    $constraintsByTable = $currentSnapshot['constraints_by_table'];
    $foreignKeysByTable = $currentSnapshot['foreign_keys_by_table'];

    $tablesWithoutPk = [];
    foreach ($tables as $table) {
        $tableName = $table['TABLE_NAME'];
        $hasPk = false;
        if (!empty($constraintsByTable[$tableName])) {
            foreach ($constraintsByTable[$tableName] as $constraint) {
                if ($constraint['CONSTRAINT_TYPE'] === 'PRIMARY KEY') {
                    $hasPk = true;
                    break;
                }
            }
        }
        if (!$hasPk) {
            $tablesWithoutPk[] = $tableName;
        }
    }

    $collationUsage = [];
    $charsetUsage = [];
    foreach ($columns as $column) {
        $collation = $column['COLLATION_NAME'];
        $charset = $column['CHARACTER_SET_NAME'];
        if ($collation !== null) {
            $collationUsage[$collation] = ($collationUsage[$collation] ?? 0) + 1;
        }
        if ($charset !== null) {
            $charsetUsage[$charset] = ($charsetUsage[$charset] ?? 0) + 1;
        }
    }
    arsort($collationUsage);
    arsort($charsetUsage);

    $dataTypeUsage = [];
    foreach ($columns as $column) {
        $dataType = strtolower((string)$column['DATA_TYPE']);
        $dataTypeUsage[$dataType] = ($dataTypeUsage[$dataType] ?? 0) + 1;
    }
    arsort($dataTypeUsage);

    $now = date('c');

    $comparisonResults = null;
    if ($compareSchema !== null) {
        $compareSnapshot = fetchSchemaSnapshot($pdo, $compareSchema, $tableFilter);

        $currentTableMap = [];
        foreach ($tables as $tableRow) {
            $currentTableMap[$tableRow['TABLE_NAME']] = $tableRow;
        }

        $compareTableMap = [];
        foreach ($compareSnapshot['tables'] as $tableRow) {
            $compareTableMap[$tableRow['TABLE_NAME']] = $tableRow;
        }

        $missingInCurrent = array_values(array_diff(array_keys($compareTableMap), array_keys($currentTableMap)));
        $missingInCompare = array_values(array_diff(array_keys($currentTableMap), array_keys($compareTableMap)));

        $columnDifferences = [];
        $sharedTables = array_intersect(array_keys($currentTableMap), array_keys($compareTableMap));
        foreach ($sharedTables as $tableName) {
            $currentColumns = $columnsByTable[$tableName] ?? [];
            $compareColumns = $compareSnapshot['columns_by_table'][$tableName] ?? [];

            $currentColumnMap = [];
            foreach ($currentColumns as $column) {
                $currentColumnMap[$column['COLUMN_NAME']] = $column;
            }

            $compareColumnMap = [];
            foreach ($compareColumns as $column) {
                $compareColumnMap[$column['COLUMN_NAME']] = $column;
            }

            $columnsMissingInCurrent = array_values(array_diff(array_keys($compareColumnMap), array_keys($currentColumnMap)));
            $columnsMissingInCompare = array_values(array_diff(array_keys($currentColumnMap), array_keys($compareColumnMap)));

            $mismatchedColumns = [];
            $sharedColumns = array_intersect(array_keys($currentColumnMap), array_keys($compareColumnMap));
            foreach ($sharedColumns as $columnName) {
                $currentCol = $currentColumnMap[$columnName];
                $compareCol = $compareColumnMap[$columnName];

                $attributeDiffs = [];
                $attributesToCompare = [
                    'COLUMN_TYPE',
                    'IS_NULLABLE',
                    'COLUMN_DEFAULT',
                    'EXTRA',
                    'COLUMN_KEY',
                    'COLLATION_NAME',
                ];
                foreach ($attributesToCompare as $attribute) {
                    $currentValue = $currentCol[$attribute];
                    $compareValue = $compareCol[$attribute];

                    if ($currentValue !== $compareValue) {
                        $attributeDiffs[$attribute] = [
                            'current' => $currentValue,
                            'compare' => $compareValue,
                        ];
                    }
                }

                if ($attributeDiffs !== []) {
                    $mismatchedColumns[$columnName] = $attributeDiffs;
                }
            }

            if ($columnsMissingInCurrent !== [] || $columnsMissingInCompare !== [] || $mismatchedColumns !== []) {
                $columnDifferences[$tableName] = [
                    'missing_in_current' => $columnsMissingInCurrent,
                    'missing_in_compare' => $columnsMissingInCompare,
                    'mismatched' => $mismatchedColumns,
                ];
            }
        }

        $comparisonResults = [
            'schema' => $compareSchema,
            'meta' => $compareSnapshot['meta'],
            'missing_in_current' => $missingInCurrent,
            'missing_in_compare' => $missingInCompare,
            'column_differences' => $columnDifferences,
        ];
    }

    if ($format === 'json') {
        $payload = [
            'meta' => [
                'generated_at' => $now,
                'schema' => $schemaName,
                'schema_charset' => $schemaMeta['charset'],
                'schema_collation' => $schemaMeta['collation'],
            ],
            'tables' => $tables,
            'columns_by_table' => $columnsByTable,
            'constraints_by_table' => $constraintsByTable,
            'foreign_keys_by_table' => $foreignKeysByTable,
            'issues' => [
                'tables_without_primary_key' => $tablesWithoutPk,
            ],
            'stats' => [
                'collations' => $collationUsage,
                'charsets' => $charsetUsage,
                'data_types' => $dataTypeUsage,
            ],
        ];
        if ($comparisonResults !== null) {
            $payload['comparison'] = $comparisonResults;
        }
        $content = json_encode($payload, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    } else {
        $lines = [];
        $lines[] = '=== Database Schema Audit ===';
        $lines[] = 'Database: ' . $schemaName;
        $lines[] = 'Generated at: ' . $now;
        $lines[] = 'Schema charset: ' . $schemaMeta['charset'] . ' | Schema collation: ' . $schemaMeta['collation'];
        $lines[] = '';
        $lines[] = '## Table Summary';
        $lines[] = '| Table | Type | Engine | Rows | Auto Increment | Collation | Created | Updated |';
        $lines[] = '|---|---|---|---|---|---|---|---|';
        foreach ($tables as $table) {
            $lines[] = sprintf(
                '| %s | %s | %s | %s | %s | %s | %s | %s |',
                $table['TABLE_NAME'],
                $table['TABLE_TYPE'],
                $table['ENGINE'] ?? '-',
                $table['TABLE_ROWS'] !== null ? (string)$table['TABLE_ROWS'] : '-',
                $table['AUTO_INCREMENT'] !== null ? (string)$table['AUTO_INCREMENT'] : '-',
                $table['TABLE_COLLATION'] ?? '-',
                $table['CREATE_TIME'] ?? '-',
                $table['UPDATE_TIME'] ?? '-'
            );
        }
        $lines[] = '';
        $lines[] = '## Potential Issues';
        if (count($tablesWithoutPk) === 0) {
            $lines[] = '- All tables declare a primary key.';
        } else {
            $lines[] = '- Tables without primary key: ' . implode(', ', $tablesWithoutPk);
        }
        $lines[] = '';
        $lines[] = '## Collation Usage';
        if (count($collationUsage) === 0) {
            $lines[] = '- No collated columns detected.';
        } else {
            foreach ($collationUsage as $collation => $count) {
                $lines[] = sprintf('- %s: %d columns', $collation, $count);
            }
        }
        $lines[] = '';
        $lines[] = '## Character Set Usage';
        if (count($charsetUsage) === 0) {
            $lines[] = '- No character sets detected.';
        } else {
            foreach ($charsetUsage as $charset => $count) {
                $lines[] = sprintf('- %s: %d columns', $charset, $count);
            }
        }
        $lines[] = '';
        $lines[] = '## Data Type Distribution';
        foreach ($dataTypeUsage as $dataType => $count) {
            $lines[] = sprintf('- %s: %d columns', $dataType, $count);
        }
        $lines[] = '';
        $lines[] = '## Constraints Overview';
        foreach ($constraintsByTable as $tableName => $tableConstraints) {
            $constraintDetails = array_map(
                static function (array $constraint): string {
                    return $constraint['CONSTRAINT_TYPE'] . ' (' . $constraint['CONSTRAINT_NAME'] . ')';
                },
                $tableConstraints
            );
            if (count($constraintDetails) === 0) {
                $lines[] = '- ' . $tableName . ': (none)';
            } else {
                $lines[] = '- ' . $tableName . ': ' . implode(', ', $constraintDetails);
            }
        }
        $lines[] = '';
        $lines[] = '## Foreign Keys';
        if (count($foreignKeysByTable) === 0) {
            $lines[] = '- No foreign keys declared.';
        } else {
            foreach ($foreignKeysByTable as $tableName => $tableFks) {
                $lines[] = '- ' . $tableName . ':';
                foreach ($tableFks as $fk) {
                    $lines[] = sprintf(
                        '  - %s: %s -> %s.%s',
                        $fk['CONSTRAINT_NAME'],
                        $fk['COLUMN_NAME'],
                        $fk['REFERENCED_TABLE_NAME'],
                        $fk['REFERENCED_COLUMN_NAME']
                    );
                }
            }
        }
        $lines[] = '';
        $lines[] = '## Column Definitions';
        foreach ($columnsByTable as $tableName => $tableColumns) {
            $lines[] = '### ' . $tableName;
            $lines[] = '| # | Column | Type | Null | Default | Extra | Key | Collation | Comment |';
            $lines[] = '|---|---|---|---|---|---|---|---|---|';
            foreach ($tableColumns as $column) {
                $lines[] = sprintf(
                    '| %d | %s | %s | %s | %s | %s | %s | %s | %s |',
                    $column['ORDINAL_POSITION'],
                    $column['COLUMN_NAME'],
                    $column['COLUMN_TYPE'],
                    $column['IS_NULLABLE'],
                    $column['COLUMN_DEFAULT'] === null ? 'NULL' : (string)$column['COLUMN_DEFAULT'],
                    $column['EXTRA'] !== '' ? $column['EXTRA'] : '-',
                    $column['COLUMN_KEY'] !== '' ? $column['COLUMN_KEY'] : '-',
                    $column['COLLATION_NAME'] ?? '-',
                    $column['COLUMN_COMMENT'] !== '' ? $column['COLUMN_COMMENT'] : '-'
                );
            }
            $lines[] = '';
        }

        if ($comparisonResults !== null) {
            $lines[] = '## Schema Comparison (vs ' . $compareSchema . ')';
            $lines[] = '- Compared against schema: ' . $compareSchema . ' (charset: ' . $comparisonResults['meta']['charset'] . ', collation: ' . $comparisonResults['meta']['collation'] . ')';
            if (count($comparisonResults['missing_in_current']) === 0) {
                $lines[] = '- Tables missing in current schema: none';
            } else {
                $lines[] = '- Tables missing in current schema: ' . implode(', ', $comparisonResults['missing_in_current']);
            }
            if (count($comparisonResults['missing_in_compare']) === 0) {
                $lines[] = '- Tables missing in comparison schema: none';
            } else {
                $lines[] = '- Tables missing in comparison schema: ' . implode(', ', $comparisonResults['missing_in_compare']);
            }

            if (count($comparisonResults['column_differences']) === 0) {
                $lines[] = '- Column definitions match for all shared tables.';
            } else {
                foreach ($comparisonResults['column_differences'] as $tableName => $diff) {
                    $lines[] = '### Differences in ' . $tableName;
                    if (count($diff['missing_in_current']) > 0) {
                        $lines[] = '- Columns missing in current schema: ' . implode(', ', $diff['missing_in_current']);
                    }
                    if (count($diff['missing_in_compare']) > 0) {
                        $lines[] = '- Columns missing in comparison schema: ' . implode(', ', $diff['missing_in_compare']);
                    }
                    if (count($diff['mismatched']) > 0) {
                        $lines[] = '| Column | Attribute | Current | Comparison |';
                        $lines[] = '|---|---|---|---|';
                        foreach ($diff['mismatched'] as $columnName => $attributeDiffs) {
                            foreach ($attributeDiffs as $attribute => $values) {
                                $lines[] = sprintf(
                                    '| %s | %s | %s | %s |',
                                    $columnName,
                                    $attribute,
                                    formatValueForReport($values['current']),
                                    formatValueForReport($values['compare'])
                                );
                            }
                        }
                        $lines[] = '';
                    }
                }
            }
        }
        $content = implode(PHP_EOL, $lines);
    }

    if ($outputPath !== null) {
        $dir = dirname($outputPath);
        if ($dir !== '' && $dir !== '.' && !is_dir($dir)) {
            if (!mkdir($dir, 0775, true) && !is_dir($dir)) {
                throw new RuntimeException('Unable to create directory for output: ' . $dir);
            }
        }
        file_put_contents($outputPath, $content);
        echo 'Report written to ' . $outputPath . PHP_EOL;
    } else {
        echo $content . PHP_EOL;
    }
} catch (Throwable $error) {
    fwrite(STDERR, 'Error: ' . $error->getMessage() . PHP_EOL);
    exit(1);
}

/**
 * Fetch tables, columns, constraints, and FK metadata for a schema.
 */
function fetchSchemaSnapshot(PDO $pdo, string $schemaName, ?array $tableFilter = null): array
{
    $schemaStmt = $pdo->prepare(
        'SELECT DEFAULT_CHARACTER_SET_NAME AS charset, DEFAULT_COLLATION_NAME AS collation
         FROM information_schema.SCHEMATA
         WHERE SCHEMA_NAME = :schema'
    );
    $schemaStmt->execute(['schema' => $schemaName]);
    $schemaMeta = $schemaStmt->fetch(PDO::FETCH_ASSOC);
    if ($schemaMeta === false) {
        throw new RuntimeException('Schema not found: ' . $schemaName);
    }

    $tablesStmt = $pdo->prepare(
        'SELECT TABLE_NAME, TABLE_TYPE, ENGINE, TABLE_ROWS, AUTO_INCREMENT, CREATE_TIME, UPDATE_TIME, TABLE_COLLATION
         FROM information_schema.TABLES
         WHERE TABLE_SCHEMA = :schema
         ORDER BY TABLE_NAME'
    );
    $tablesStmt->execute(['schema' => $schemaName]);
    $tables = [];
    while ($row = $tablesStmt->fetch(PDO::FETCH_ASSOC)) {
        if ($tableFilter !== null && !in_array($row['TABLE_NAME'], $tableFilter, true)) {
            continue;
        }
        $tables[] = $row;
    }

    $columnsStmt = $pdo->prepare(
        'SELECT TABLE_NAME, COLUMN_NAME, ORDINAL_POSITION, COLUMN_TYPE, DATA_TYPE, IS_NULLABLE, COLUMN_DEFAULT,
                COLUMN_KEY, EXTRA, COLUMN_COMMENT, COLLATION_NAME, CHARACTER_SET_NAME
         FROM information_schema.COLUMNS
         WHERE TABLE_SCHEMA = :schema
         ORDER BY TABLE_NAME, ORDINAL_POSITION'
    );
    $columnsStmt->execute(['schema' => $schemaName]);
    $columns = [];
    while ($row = $columnsStmt->fetch(PDO::FETCH_ASSOC)) {
        if ($tableFilter !== null && !in_array($row['TABLE_NAME'], $tableFilter, true)) {
            continue;
        }
        $columns[] = $row;
    }

    $constraintsStmt = $pdo->prepare(
        'SELECT TABLE_NAME, CONSTRAINT_NAME, CONSTRAINT_TYPE
         FROM information_schema.TABLE_CONSTRAINTS
         WHERE TABLE_SCHEMA = :schema'
    );
    $constraintsStmt->execute(['schema' => $schemaName]);
    $constraints = [];
    while ($row = $constraintsStmt->fetch(PDO::FETCH_ASSOC)) {
        if ($tableFilter !== null && !in_array($row['TABLE_NAME'], $tableFilter, true)) {
            continue;
        }
        $constraints[] = $row;
    }

    $foreignKeysStmt = $pdo->prepare(
        'SELECT TABLE_NAME, CONSTRAINT_NAME, COLUMN_NAME, REFERENCED_TABLE_NAME, REFERENCED_COLUMN_NAME, ORDINAL_POSITION
         FROM information_schema.KEY_COLUMN_USAGE
         WHERE TABLE_SCHEMA = :schema AND REFERENCED_TABLE_NAME IS NOT NULL
         ORDER BY TABLE_NAME, CONSTRAINT_NAME, ORDINAL_POSITION'
    );
    $foreignKeysStmt->execute(['schema' => $schemaName]);
    $foreignKeys = [];
    while ($row = $foreignKeysStmt->fetch(PDO::FETCH_ASSOC)) {
        if ($tableFilter !== null && !in_array($row['TABLE_NAME'], $tableFilter, true)) {
            continue;
        }
        $foreignKeys[] = $row;
    }

    $columnsByTable = [];
    foreach ($columns as $column) {
        $table = $column['TABLE_NAME'];
        if (!isset($columnsByTable[$table])) {
            $columnsByTable[$table] = [];
        }
        $columnsByTable[$table][] = $column;
    }

    $constraintsByTable = [];
    foreach ($constraints as $constraint) {
        $table = $constraint['TABLE_NAME'];
        if (!isset($constraintsByTable[$table])) {
            $constraintsByTable[$table] = [];
        }
        $constraintsByTable[$table][] = $constraint;
    }

    $foreignKeysByTable = [];
    foreach ($foreignKeys as $fk) {
        $table = $fk['TABLE_NAME'];
        if (!isset($foreignKeysByTable[$table])) {
            $foreignKeysByTable[$table] = [];
        }
        $foreignKeysByTable[$table][] = $fk;
    }

    return [
        'meta' => $schemaMeta,
        'tables' => $tables,
        'columns' => $columns,
        'constraints' => $constraints,
        'foreign_keys' => $foreignKeys,
        'columns_by_table' => $columnsByTable,
        'constraints_by_table' => $constraintsByTable,
        'foreign_keys_by_table' => $foreignKeysByTable,
    ];
}

/**
 * Normalize values for Markdown tables.
 */
function formatValueForReport($value): string
{
    if ($value === null) {
        return 'NULL';
    }
    $stringValue = (string)$value;
    return $stringValue === '' ? '(empty)' : $stringValue;
}
