<?php
declare(strict_types=1);

$ROOT = dirname(__DIR__);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

try {
    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (Throwable $connectionError) {
    fwrite(STDERR, 'Connection failed: ' . $connectionError->getMessage() . PHP_EOL);
    exit(1);
}

try {
    // para_destinatarios: ensure legacy column 'ruc' exists for backwards compatibility
    ensureColumn($pdo, 'para_destinatarios', 'ruc', "ALTER TABLE `para_destinatarios` ADD COLUMN `ruc` varchar(20) NULL AFTER `fantasia`");

    // para_producto_paletizado: keep legacy metric uv_por_capa available
    ensureColumn($pdo, 'para_producto_paletizado', 'uv_por_capa', "ALTER TABLE `para_producto_paletizado` ADD COLUMN `uv_por_capa` int unsigned NULL AFTER `capas_por_pallet`");

    // wh_deposito: add legacy metadata columns while keeping nuevos campos
    ensureColumn($pdo, 'wh_deposito', 'operativa_id', "ALTER TABLE `wh_deposito` ADD COLUMN `operativa_id` bigint unsigned NULL AFTER `id`");
    ensureColumn($pdo, 'wh_deposito', 'name', "ALTER TABLE `wh_deposito` ADD COLUMN `name` varchar(120) NULL AFTER `nombre`");
    ensureColumn($pdo, 'wh_deposito', 'pad_lado', "ALTER TABLE `wh_deposito` ADD COLUMN `pad_lado` tinyint NULL DEFAULT 2 AFTER `direccion`");
    ensureColumn($pdo, 'wh_deposito', 'pad_orient', "ALTER TABLE `wh_deposito` ADD COLUMN `pad_orient` tinyint NULL DEFAULT 1 AFTER `pad_lado`");
    ensureColumn($pdo, 'wh_deposito', 'pad_rack', "ALTER TABLE `wh_deposito` ADD COLUMN `pad_rack` tinyint NULL DEFAULT 2 AFTER `pad_orient`");
    ensureColumn($pdo, 'wh_deposito', 'pad_columna', "ALTER TABLE `wh_deposito` ADD COLUMN `pad_columna` tinyint NULL DEFAULT 2 AFTER `pad_rack`");
    ensureColumn($pdo, 'wh_deposito', 'pad_nivel', "ALTER TABLE `wh_deposito` ADD COLUMN `pad_nivel` tinyint NULL DEFAULT 2 AFTER `pad_columna`");
    ensureColumn($pdo, 'wh_deposito', 'pad_fondo', "ALTER TABLE `wh_deposito` ADD COLUMN `pad_fondo` tinyint NULL DEFAULT 2 AFTER `pad_nivel`");
    ensureColumn($pdo, 'wh_deposito', 'code_sep', "ALTER TABLE `wh_deposito` ADD COLUMN `code_sep` char(1) NULL DEFAULT '-' AFTER `pad_fondo`");
    ensureColumn($pdo, 'wh_deposito', 'is_active', "ALTER TABLE `wh_deposito` ADD COLUMN `is_active` tinyint(1) NULL DEFAULT 1 AFTER `activo`");
    ensureColumn($pdo, 'wh_deposito', 'created_by', "ALTER TABLE `wh_deposito` ADD COLUMN `created_by` bigint unsigned NULL AFTER `is_active`");
    ensureColumn($pdo, 'wh_deposito', 'updated_by', "ALTER TABLE `wh_deposito` ADD COLUMN `updated_by` bigint unsigned NULL AFTER `created_by`");

    $pdo->exec('UPDATE `wh_deposito` SET `name` = `nombre` WHERE `name` IS NULL');
    $pdo->exec('UPDATE `wh_deposito` SET `is_active` = `activo` WHERE `is_active` IS NULL');

    // para_codigo_tipos: ensure unique constraint matches legacy expectation (unique codigo)
    normalizeParaCodigoTiposIndex($pdo);

    echo "Schema synchronization applied successfully." . PHP_EOL;
} catch (Throwable $error) {
    fwrite(STDERR, 'Error: ' . $error->getMessage() . PHP_EOL);
    exit(1);
}

/**
 * Ensure a column exists, running the supplied ALTER TABLE statement if missing.
 */
function ensureColumn(PDO $pdo, string $table, string $column, string $alterSql): void
{
    $stmt = $pdo->prepare(
        'SELECT 1 FROM information_schema.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = :table AND COLUMN_NAME = :column'
    );
    $stmt->execute([
        'table' => $table,
        'column' => $column,
    ]);
    if ($stmt->fetchColumn() === false) {
        $pdo->exec($alterSql);
    }
}

/**
 * Replace the existing unique index on para_codigo_tipos so codigo is unique.
 */
function normalizeParaCodigoTiposIndex(PDO $pdo): void
{
    $indexStmt = $pdo->query('SHOW INDEX FROM `para_codigo_tipos` WHERE Key_name = "uq_para_codigo_tipos_codigo"');
    $columns = [];
    while ($row = $indexStmt->fetch(PDO::FETCH_ASSOC)) {
        $columns[(int)$row['Seq_in_index']] = $row['Column_name'];
    }
    if ($columns === []) {
        // Index missing entirely, just add it.
        $pdo->exec('ALTER TABLE `para_codigo_tipos` ADD UNIQUE INDEX `uq_para_codigo_tipos_codigo` (`codigo`)');
        return;
    }

    ksort($columns);
    $joined = implode(',', $columns);

    if ($joined !== 'codigo') {
        $pdo->exec('ALTER TABLE `para_codigo_tipos` DROP INDEX `uq_para_codigo_tipos_codigo`');
        $pdo->exec('ALTER TABLE `para_codigo_tipos` ADD UNIQUE INDEX `uq_para_codigo_tipos_codigo` (`codigo`)');
    }
}
