<?php
// scripts/debug_cumplimiento.php
declare(strict_types=1);

echo "Debugging Cumplimiento API...\n";
echo "=============================\n\n";

// Test database connection first
echo "1. Testing database connection...\n";
try {
    require_once __DIR__ . '/../config/db.php';
    $pdo = getPDO();
    echo "   ✅ Database connection successful\n";
    
    // Test basic query
    $stmt = $pdo->query("SELECT COUNT(*) as count FROM so_embarque");
    $result = $stmt->fetch();
    echo "   📊 Found " . $result['count'] . " embarques in database\n";
    
} catch (Exception $e) {
    echo "   ❌ Database error: " . $e->getMessage() . "\n";
    exit(1);
}

echo "\n2. Testing required tables...\n";
$tables = [
    'so_embarque', 'so_embarque_estado', 'so_embarque_parada', 
    'so_embarque_pre', 'so_preembarque_item', 'para_destinatarios',
    'para_clientes', 'para_moviles', 'para_choferes', 'wh_deposito'
];

foreach ($tables as $table) {
    try {
        $stmt = $pdo->query("SELECT COUNT(*) FROM $table");
        $count = $stmt->fetchColumn();
        echo "   ✅ Table $table: $count rows\n";
    } catch (Exception $e) {
        echo "   ❌ Table $table error: " . $e->getMessage() . "\n";
    }
}

echo "\n3. Testing API logic step by step...\n";

// Mock session
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
$_SESSION['usuario_id'] = 1;
$_SESSION['usuario_nombre'] = 'Test User';

// Mock GET parameters
$_GET['fecha_desde'] = date('Y-m-d', strtotime('-30 days'));
$_GET['fecha_hasta'] = date('Y-m-d');

echo "   Date range: {$_GET['fecha_desde']} to {$_GET['fecha_hasta']}\n";

try {
    // Parse filters
    $fechaDesde = trim($_GET['fecha_desde'] ?? '');
    $fechaHasta = trim($_GET['fecha_hasta'] ?? '');
    $clienteId = trim($_GET['cliente_id'] ?? '');
    $destinatarioId = trim($_GET['destinatario_id'] ?? '');
    $estado = trim($_GET['estado'] ?? '');
    $movilId = trim($_GET['movil_id'] ?? '');
    
    echo "   ✅ Filters parsed successfully\n";
    
    // Build WHERE clause
    $where = ['1=1'];
    $params = [];

    if ($fechaDesde) {
        $where[] = 'DATE(e.carga_fecha) >= ?';
        $params[] = $fechaDesde;
    }
    if ($fechaHasta) {
        $where[] = 'DATE(e.carga_fecha) <= ?';
        $params[] = $fechaHasta;
    }

    $whereClause = 'WHERE ' . implode(' AND ', $where);
    echo "   ✅ WHERE clause built: $whereClause\n";
    echo "   📝 Parameters: " . implode(', ', $params) . "\n";
    
    // Test main query
    $sql = "
        SELECT 
            e.id,
            e.codigo,
            DATE(e.carga_fecha) as fecha_embarque,
            est.code as estado_code,
            est.nombre as estado_nombre
        FROM so_embarque e
        LEFT JOIN so_embarque_estado est ON est.id = e.estado_id
        LEFT JOIN wh_deposito dep ON dep.id = e.deposito_id
        LEFT JOIN para_moviles m ON m.id = e.movil_id
        LEFT JOIN para_choferes ch ON ch.id = e.chofer_id
        {$whereClause}
        ORDER BY e.carga_fecha DESC
        LIMIT 5
    ";
    
    echo "   🔍 Testing main query (first 5 rows)...\n";
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $embarques = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo "   ✅ Main query successful, found " . count($embarques) . " embarques\n";
    
    if (count($embarques) > 0) {
        $first = $embarques[0];
        echo "   📄 Sample embarque: {$first['codigo']} ({$first['fecha_embarque']}) - {$first['estado_nombre']}\n";
    }
    
    // Test filter options queries
    echo "   🔧 Testing filter options queries...\n";
    
    $estados = $pdo->query("
        SELECT code, nombre 
        FROM so_embarque_estado 
        WHERE activo = 1 
        ORDER BY orden
    ")->fetchAll(PDO::FETCH_ASSOC);
    echo "      - Estados: " . count($estados) . " found\n";
    
    $clientes = $pdo->query("
        SELECT DISTINCT c.id, c.razon_social
        FROM para_clientes c
        WHERE c.deleted_at IS NULL
        AND EXISTS (
            SELECT 1 FROM para_destinatarios d 
            WHERE d.cliente_id = c.id AND d.deleted_at IS NULL
        )
        ORDER BY c.razon_social
        LIMIT 10
    ")->fetchAll(PDO::FETCH_ASSOC);
    echo "      - Clientes: " . count($clientes) . " found\n";
    
    echo "   ✅ All filter queries successful\n";
    
    echo "\n🎉 All tests passed! The API should work correctly.\n";
    echo "📝 The error might be related to headers or session handling in the web context.\n";
    
} catch (Exception $e) {
    echo "   ❌ API logic error: " . $e->getMessage() . "\n";
    echo "   📍 File: " . $e->getFile() . "\n";
    echo "   🔢 Line: " . $e->getLine() . "\n";
    echo "   📚 Trace:\n" . $e->getTraceAsString() . "\n";
}