<?php
// scripts/debug_sp_uv_uc.php
// Diagnóstico específico del stored procedure para conversión UV→UC
declare(strict_types=1);

require_once __DIR__ . '/../config/db.php';

function outln(string $s): void { echo $s . PHP_EOL; }

try {
    $pdo = getPDO();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec("SET NAMES utf8mb4");

    $soId = 5;
    $soCodigo = 'SO-20251023-162511-2';
    
    outln("=== DIAGNÓSTICO STORED PROCEDURE UV→UC ===");
    outln("SO ID: $soId");
    outln("SO Código: $soCodigo");
    outln("");
    
    // 1. Estado actual del pedido
    outln("1. ESTADO ACTUAL DEL PEDIDO:");
    $stmt = $pdo->prepare("
        SELECT 
            pdi.id as item_id,
            p.code as sku,
            p.denominacion,
            pdi.expected_uv,
            pdi.expected_uc, 
            pdi.prepared_uv,
            pdi.prepared_uc,
            (pdi.expected_uc - COALESCE(pdi.prepared_uc,0)) as pending_uc
        FROM so_pedido_dest_item pdi
        JOIN pro_producto p ON p.id = pdi.producto_id
        WHERE pdi.pedido_id = ?
        ORDER BY p.code
    ");
    $stmt->execute([$soId]);
    
    while ($row = $stmt->fetch()) {
        $pendUC = max(0, $row['expected_uc'] - $row['prepared_uc']);
        outln(sprintf("  %s (%s): Expected=%d UC, Prepared=%d UC, Pending=%d UC", 
            $row['sku'], $row['denominacion'], $row['expected_uc'], $row['prepared_uc'], $pendUC));
    }
    outln("");

    // 2. Stock disponible para conversión UV→UC
    outln("2. STOCK DISPONIBLE PARA CONVERSIÓN UV→UC:");
    $stmt = $pdo->prepare("
        SELECT 
            p.code as sku,
            p.denominacion,
            pdi.expected_uc,
            pdi.prepared_uc,
            (pdi.expected_uc - COALESCE(pdi.prepared_uc,0)) as pending_uc,
            COALESCE(SUM(s.qty_uv), 0) as available_uv,
            COALESCE(pp.unidades_por_uv, 0) as uc_per_uv
        FROM so_pedido_dest_item pdi
        JOIN pro_producto p ON p.id = pdi.producto_id
        LEFT JOIN para_producto_pack pp ON pp.producto_id = pdi.producto_id
        LEFT JOIN wh_stock s ON s.producto_id = pdi.producto_id 
            AND s.deposito_id = (SELECT deposito_id FROM so_pedido WHERE id = ?)
        WHERE pdi.pedido_id = ?
    GROUP BY pdi.id, p.code, p.denominacion, pdi.expected_uc, pdi.prepared_uc, pp.unidades_por_uv
        HAVING pending_uc > 0
        ORDER BY p.code
    ");
    $stmt->execute([$soId, $soId]);
    
    while ($row = $stmt->fetch()) {
        $pendUC = $row['pending_uc'];
        $availUV = $row['available_uv'];
        $ucPerUv = $row['uc_per_uv'];
        $canConvert = $ucPerUv > 0 && $availUV > 0;
        $maxConvertibleUC = $canConvert ? $availUV * $ucPerUv : 0;
        
        outln(sprintf("  %s: Pending=%d UC, Available=%d UV, Factor=%d UC/UV, Can convert=%s, Max convertible=%d UC", 
            $row['sku'], $pendUC, $availUV, $ucPerUv, 
            $canConvert ? '✅' : '❌', $maxConvertibleUC));
    }
    outln("");

    // 3. Logs más recientes de UV→UC para este pedido
    outln("3. LOGS RECIENTES DE UV→UC:");
    $stmt = $pdo->prepare("
        SELECT ts, fase, msg, take_uv, take_uc
        FROM so_preparar_auto_log 
        WHERE pedido_codigo = ? 
        AND msg LIKE '%UV→UC%'
        ORDER BY id DESC 
        LIMIT 10
    ");
    $stmt->execute([$soCodigo]);
    
    while ($row = $stmt->fetch()) {
        outln(sprintf("  [%s] %s: %s (UV=%s, UC=%s)", 
            $row['ts'], $row['fase'], $row['msg'], 
            $row['take_uv'] ?? 'null', $row['take_uc'] ?? 'null'));
    }
    outln("");

    // 4. Movimientos generados
    outln("4. MOVIMIENTOS GENERADOS:");
    $stmt = $pdo->prepare("
        SELECT 
            wm.created_at,
            p.code as sku,
            wm.delta_uv,
            wm.delta_uc,
            pos_from.codigo as from_pos,
            pos_to.codigo as to_pos
        FROM wh_move wm
        JOIN pro_producto p ON p.id = wm.producto_id
        JOIN wh_posicion pos_from ON pos_from.id = wm.from_pos_id
        JOIN wh_posicion pos_to ON pos_to.id = wm.to_pos_id
        WHERE wm.referencia = CONCAT('PRE-', ?)
        ORDER BY wm.created_at DESC
    ");
    $stmt->execute([$soId]);
    
    $movements = $stmt->fetchAll();
    if (empty($movements)) {
        outln("  No hay movimientos generados para PRE-$soId");
    } else {
        foreach ($movements as $row) {
            outln(sprintf("  [%s] %s: %d UV, %d UC (%s → %s)", 
                $row['created_at'], $row['sku'], $row['delta_uv'], $row['delta_uc'],
                $row['from_pos'], $row['to_pos']));
        }
    }
    outln("");

    // 5. Registros en so_pre_pick
    outln("5. REGISTROS EN SO_PRE_PICK:");
    $stmt = $pdo->prepare("
        SELECT 
            spp.created_at,
            p.code as sku,
            spp.uv_cajas,
            spp.uc_unidades,
            pos_from.codigo as from_pos,
            pos_to.codigo as to_pos
        FROM so_pre_pick spp
        JOIN so_pedido_dest_item pdi ON pdi.id = spp.pedido_dest_item_id
        JOIN pro_producto p ON p.id = pdi.producto_id
        JOIN wh_posicion pos_from ON pos_from.id = spp.from_pos_id
        JOIN wh_posicion pos_to ON pos_to.id = spp.to_pos_id
        WHERE spp.preembarque_id = (
            SELECT pre.id FROM so_preembarque pre WHERE pre.pedido_id = ? LIMIT 1
        )
        ORDER BY spp.created_at DESC
    ");
    $stmt->execute([$soId]);
    
    $picks = $stmt->fetchAll();
    if (empty($picks)) {
        outln("  No hay registros en so_pre_pick");
    } else {
        foreach ($picks as $row) {
            outln(sprintf("  [%s] %s: %d UV, %d UC (%s → %s)", 
                $row['created_at'], $row['sku'], $row['uv_cajas'], $row['uc_unidades'],
                $row['from_pos'], $row['to_pos']));
        }
    }
    outln("");

    // 6. Test directo del stored procedure con logging
    outln("6. EJECUTANDO SP CON UV→UC=1 Y LOGGING:");
    
    // Activar logging
    $pdo->exec("SET @so_pre_log = 1");
    $pdo->exec("SET @so_pre_direct_to_prep = 1");
    
    // Limpiar logs anteriores de este test
    $pdo->prepare("DELETE FROM so_preparar_auto_log WHERE pedido_codigo = ? AND msg LIKE '%TEST:%'")->execute([$soCodigo]);
    
    // Insertar marcador de inicio
    $pdo->prepare("INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, fase, msg) VALUES (?, ?, 'TEST', 'TEST: Inicio diagnóstico UV→UC')")->execute([$soCodigo, $soId]);
    
    try {
        $stmt = $pdo->prepare("CALL sp_so_preparar_auto(?, 'DEP1', NULL, 0)");
        $stmt->execute([$soCodigo]);
        outln("  ✅ SP ejecutado exitosamente");
    } catch (Exception $e) {
        outln("  ❌ Error ejecutando SP: " . $e->getMessage());
    }
    
    // Insertar marcador de fin
    $pdo->prepare("INSERT INTO so_preparar_auto_log (pedido_codigo, pedido_id, fase, msg) VALUES (?, ?, 'TEST', 'TEST: Fin diagnóstico UV→UC')")->execute([$soCodigo, $soId]);
    
    outln("");

    // 7. Logs generados en esta ejecución
    outln("7. LOGS DE ESTA EJECUCIÓN:");
    $stmt = $pdo->prepare("
        SELECT ts, fase, msg, take_uv, take_uc
        FROM so_preparar_auto_log 
        WHERE pedido_codigo = ? 
        AND ts >= DATE_SUB(NOW(), INTERVAL 1 MINUTE)
        ORDER BY id ASC
    ");
    $stmt->execute([$soCodigo]);
    
    $logCount = 0;
    while ($row = $stmt->fetch()) {
        $logCount++;
        if ($logCount <= 20) { // Limitar a 20 logs para no saturar
            outln(sprintf("  [%s] %s: %s", $row['ts'], $row['fase'], $row['msg']));
        }
    }
    
    if ($logCount > 20) {
        outln("  ... y " . ($logCount - 20) . " logs más");
    }
    
    outln("");
    outln("=== FIN DIAGNÓSTICO ===");

} catch (Exception $e) {
    outln("ERROR: " . $e->getMessage());
    outln("Trace: " . $e->getTraceAsString());
}