<?php
// scripts/debug_sp_uv_uc_fixed.php
// Diagnóstico específico del stored procedure para conversión UV→UC
declare(strict_types=1);

require_once __DIR__ . '/../config/db.php';

function outln(string $s): void { echo $s . PHP_EOL; }

try {
    $pdo = getPDO();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec("SET NAMES utf8mb4");

    $soId = 5;
    $soCodigo = 'SO-20251023-162511-2';
    
    outln("=== DIAGNÓSTICO STORED PROCEDURE UV→UC (Corregido) ===");
    outln("SO ID: $soId");
    outln("SO Código: $soCodigo");
    outln("");

    // 1. Estado actual del pedido (PDI)
    outln("1. ESTADO ACTUAL DEL PEDIDO (so_pedido_detalle_itm)");
    outln("=" . str_repeat("=", 60));
    
    $query = "
        SELECT 
            pdi.id, 
            p.name as producto_name,
            p.codigo as producto_codigo,
            pdi.cantidad_solicitada,
            pdi.prepared_uc,
            pdi.prepared_uv,
            pdi.factor_conversion_uc as factor_uc,
            (pdi.cantidad_solicitada - pdi.prepared_uc) as pending_uc,
            CASE WHEN pdi.factor_conversion_uc > 0 
                 THEN CEIL((pdi.cantidad_solicitada - pdi.prepared_uc) / pdi.factor_conversion_uc)
                 ELSE 0 END as needed_uv_for_pending
        FROM so_pedido_detalle_itm pdi
        JOIN para_productos p ON p.id = pdi.producto_id
        WHERE pdi.so_pedido_id = (
            SELECT id FROM so_pedido WHERE so_id = ?
        )
        ORDER BY pdi.id
    ";
    
    $stmt = $pdo->prepare($query);
    $stmt->execute([$soId]);
    $items = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    foreach($items as $item) {
        outln("  ID:{$item['id']} | {$item['producto_codigo']} | Sol:{$item['cantidad_solicitada']} | PrepUC:{$item['prepared_uc']} | PrepUV:{$item['prepared_uv']} | Factor:{$item['factor_uc']} | PendUC:{$item['pending_uc']} | NeedUV:{$item['needed_uv_for_pending']}");
    }
    outln("");

    // 2. Stock disponible por producto
    outln("2. STOCK DISPONIBLE (wh_posicion)");
    outln("=" . str_repeat("=", 60));
    
    $query = "
        SELECT 
            p.codigo as producto_codigo,
            p.name as producto_name,
            wp.pack_tipo,
            SUM(wp.stock_fisico) as stock_total
        FROM wh_posicion wp
        JOIN para_productos p ON p.id = wp.producto_id
        WHERE wp.producto_id IN (
            SELECT DISTINCT pdi.producto_id 
            FROM so_pedido_detalle_itm pdi
            WHERE pdi.so_pedido_id = (SELECT id FROM so_pedido WHERE so_id = ?)
        )
        AND wp.stock_fisico > 0
        GROUP BY p.id, wp.pack_tipo
        ORDER BY p.codigo, wp.pack_tipo
    ";
    
    $stmt = $pdo->prepare($query);
    $stmt->execute([$soId]);
    $stock = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    foreach($stock as $s) {
        outln("  {$s['producto_codigo']} | Tipo:{$s['pack_tipo']} | Stock:{$s['stock_total']}");
    }
    outln("");

    // 3. Movimientos registrados en wh_move
    outln("3. MOVIMIENTOS WH_MOVE RECIENTES");
    outln("=" . str_repeat("=", 60));
    
    $query = "
        SELECT 
            wm.id,
            wm.created_at,
            p.codigo as producto_codigo,
            wm.pack_tipo as pack_tipo_move,
            wm.cantidad,
            wm.tipo_operacion,
            wm.so_code,
            wm.comments
        FROM wh_move wm
        JOIN para_productos p ON p.id = wm.producto_id
        WHERE wm.so_code = ?
        ORDER BY wm.created_at DESC, wm.id DESC
        LIMIT 10
    ";
    
    $stmt = $pdo->prepare($query);
    $stmt->execute([$soCodigo]);
    $moves = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if(empty($moves)) {
        outln("  No hay movimientos registrados para este SO");
    } else {
        foreach($moves as $move) {
            outln("  ID:{$move['id']} | {$move['created_at']} | {$move['producto_codigo']} | Tipo:{$move['pack_tipo_move']} | Cant:{$move['cantidad']} | Op:{$move['tipo_operacion']} | Comment:{$move['comments']}");
        }
    }
    outln("");

    // 4. Análisis de conversión UV→UC por producto
    outln("4. ANÁLISIS DE CONVERSIÓN UV→UC");
    outln("=" . str_repeat("=", 60));
    
    foreach($items as $item) {
        if($item['pending_uc'] > 0 && $item['factor_uc'] > 0) {
            outln("  PRODUCTO: {$item['producto_codigo']}");
            outln("    - Pendiente UC: {$item['pending_uc']}");
            outln("    - Factor conversión: {$item['factor_uc']}");
            outln("    - UV necesarios: {$item['needed_uv_for_pending']}");
            
            // Buscar stock UV disponible
            $stockUV = array_filter($stock, function($s) use ($item) {
                return $s['producto_codigo'] === $item['producto_codigo'] && $s['pack_tipo'] === 'UV';
            });
            
            if(!empty($stockUV)) {
                $stockUVTotal = array_sum(array_column($stockUV, 'stock_total'));
                outln("    - Stock UV disponible: {$stockUVTotal}");
                
                if($stockUVTotal >= $item['needed_uv_for_pending']) {
                    outln("    ✓ CONVERSIÓN POSIBLE: Hay suficiente stock UV");
                } else {
                    outln("    ✗ CONVERSIÓN PARCIAL: Stock UV insuficiente");
                }
            } else {
                outln("    ✗ NO HAY STOCK UV");
            }
            outln("");
        }
    }

    // 5. Logs del stored procedure
    outln("5. LOGS DEL STORED PROCEDURE");
    outln("=" . str_repeat("=", 60));
    
    $query = "
        SELECT 
            created_at,
            log_level,
            message
        FROM so_preparar_auto_log
        WHERE so_code = ?
        ORDER BY created_at DESC
        LIMIT 20
    ";
    
    $stmt = $pdo->prepare($query);
    $stmt->execute([$soCodigo]);
    $logs = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if(empty($logs)) {
        outln("  No hay logs para este SO");
    } else {
        foreach($logs as $log) {
            outln("  [{$log['created_at']}] {$log['log_level']}: {$log['message']}");
        }
    }
    outln("");

    // 6. Registros en so_pre_pick
    outln("6. REGISTROS SO_PRE_PICK");
    outln("=" . str_repeat("=", 60));
    
    $query = "
        SELECT 
            spp.id,
            spp.created_at,
            p.codigo as producto_codigo,
            spp.pack_tipo,
            spp.cantidad_pick,
            spp.cantidad_prep,
            spp.lote_code,
            spp.posicion_code
        FROM so_pre_pick spp
        JOIN para_productos p ON p.id = spp.producto_id
        WHERE spp.so_code = ?
        ORDER BY spp.created_at DESC
        LIMIT 10
    ";
    
    $stmt = $pdo->prepare($query);
    $stmt->execute([$soCodigo]);
    $picks = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if(empty($picks)) {
        outln("  No hay registros de picking para este SO");
    } else {
        foreach($picks as $pick) {
            outln("  ID:{$pick['id']} | {$pick['created_at']} | {$pick['producto_codigo']} | Tipo:{$pick['pack_tipo']} | Pick:{$pick['cantidad_pick']} | Prep:{$pick['cantidad_prep']} | Lote:{$pick['lote_code']}");
        }
    }
    outln("");

    outln("=== FIN DIAGNÓSTICO ===");

} catch(Exception $e) {
    outln("ERROR: " . $e->getMessage());
    exit(1);
}
?>