<?php
/**
 * Debug script para UV→UC conversion
 * Simula la llamada al SP con logs de diagnóstico
 */
declare(strict_types=1);

$ROOT = dirname(__DIR__);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

try {
    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec('SET NAMES utf8mb4');
    
    // Set session settings
    $pdo->exec('SET SESSION transaction_isolation = "READ-COMMITTED"');
    $pdo->exec('SET SESSION autocommit = 1');
    
    // Configurar variables del SP
    $pdo->exec('SET @so_pre_log := 1');           // Activar logging
    $pdo->exec('SET @so_pre_uc_to_uv := 0');      // Desactivar UC→UV  
    $pdo->exec('SET @so_pre_uv_to_uc := 1');      // Activar UV→UC
    
    echo "=== Testing UV→UC Conversion Debug ===\n";
    echo "Parameters set:\n";
    echo "- @so_pre_log = 1 (logging enabled)\n";
    echo "- @so_pre_uc_to_uv = 0 (UC→UV disabled)\n";
    echo "- @so_pre_uv_to_uc = 1 (UV→UC enabled)\n\n";
    
    // Limpiar logs previos de este pedido
    $pdo->exec("DELETE FROM so_preparar_auto_log WHERE pedido_codigo = 'SO-16'");
    
    // Ejecutar SP
    echo "Executing SP...\n";
    $stmt = $pdo->prepare('CALL sp_so_preparar_auto(?, ?, ?, ?)');
    $stmt->execute(['SO-16', 'DEP1', null, 1]); // simulate=1
    
    // Recoger resultsets
    $sets = [];
    do {
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        if ($rows !== false && $rows) $sets[] = $rows;
    } while ($stmt->nextRowset());
    
    $stmt->closeCursor();
    
    echo "SP executed. Result sets: " . count($sets) . "\n\n";
    
    // Mostrar logs de debug
    echo "=== Debug Logs ===\n";
    $logs = $pdo->query("
        SELECT id, fase, item_id, producto_id, lote_code, take_uv, take_uc, msg, created_at
        FROM so_preparar_auto_log 
        WHERE pedido_codigo = 'SO-16' 
        ORDER BY id ASC
    ")->fetchAll(PDO::FETCH_ASSOC);
    
    if (empty($logs)) {
        echo "No debug logs found. Check if logging is working.\n";
    } else {
        foreach ($logs as $log) {
            echo sprintf(
                "[%s] %s: %s (UV:%s, UC:%s) - %s\n",
                $log['id'],
                $log['fase'] ?? 'N/A',
                $log['lote_code'] ?? 'N/A',
                $log['take_uv'] ?? '0',
                $log['take_uc'] ?? '0',
                $log['msg'] ?? ''
            );
        }
    }
    
    echo "\n=== Final Results ===\n";
    if (!empty($sets)) {
        $resumen = end($sets);
        if ($resumen) {
            echo "Final summary:\n";
            foreach ($resumen as $row) {
                print_r($row);
            }
        }
    }
    
    // Verificar si se crearon movimientos
    echo "\n=== Movement Check ===\n";
    $moves = $pdo->query("
        SELECT wm.*, pl.codigo as lote_code, pf.code as from_pos, pt.code as to_pos
        FROM wh_move wm
        LEFT JOIN para_lote pl ON pl.id = wm.lote_id
        LEFT JOIN wh_posicion pf ON pf.id = wm.from_posicion_id 
        LEFT JOIN wh_posicion pt ON pt.id = wm.to_posicion_id
        WHERE wm.pre_id IN (SELECT id FROM so_preembarque WHERE codigo = 'PRE-SO-16')
        ORDER BY wm.id DESC
        LIMIT 10
    ")->fetchAll(PDO::FETCH_ASSOC);
    
    if (empty($moves)) {
        echo "No movements found for PRE-SO-16\n";
    } else {
        echo "Recent movements:\n";
        foreach ($moves as $move) {
            echo sprintf(
                "Move %s: %s→%s, UV:%s, UC:%s, Lote:%s\n",
                $move['id'],
                $move['from_pos'] ?? 'N/A',
                $move['to_pos'] ?? 'N/A',
                $move['cantidad_uv'] ?? '0',
                $move['cantidad_uc'] ?? '0',
                $move['lote_code'] ?? 'N/A'
            );
        }
    }
    
} catch (Throwable $e) {
    echo "Error: " . $e->getMessage() . "\n";
    echo "File: " . $e->getFile() . ":" . $e->getLine() . "\n";
}