<?php
declare(strict_types=1);
// scripts/delete_all_positions.php
// Backup all rows from wh_posicion to scripts/backup_wh_posicion_YYYYmmdd_HHMMSS.json
// Then delete all rows from wh_posicion (DELETE; fallback to TRUNCATE with FK checks disabled)

$root = dirname(__DIR__);
require_once $root . '/config/db.php';

function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

try {
    $pdo = getPDO();
} catch (Throwable $e) {
    fwrite(STDERR, "DB connection error: " . $e->getMessage() . PHP_EOL);
    exit(1);
}

// get columns
try {
    $st = $pdo->prepare("SELECT COLUMN_NAME FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'wh_posicion' ORDER BY ORDINAL_POSITION");
    $st->execute();
    $cols = $st->fetchAll(PDO::FETCH_COLUMN);
} catch (Throwable $e) {
    fwrite(STDERR, "Failed to read columns: " . $e->getMessage() . PHP_EOL);
    exit(1);
}

if (empty($cols)) {
    fwrite(STDERR, "Table wh_posicion has no columns or does not exist. Aborting." . PHP_EOL);
    exit(1);
}

$colSql = implode(', ', array_map(function($c){ return "`$c`"; }, $cols));

try {
    $q = $pdo->prepare("SELECT $colSql FROM `wh_posicion` ORDER BY id");
    $q->execute();
    $rows = $q->fetchAll(PDO::FETCH_ASSOC);
} catch (Throwable $e) {
    fwrite(STDERR, "Failed to fetch rows: " . $e->getMessage() . PHP_EOL);
    exit(1);
}

$ts = date('Ymd_His');
$backupFile = __DIR__ . "/backup_wh_posicion_{$ts}.json";
try {
    file_put_contents($backupFile, json_encode($rows, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    echo "Backup written to: $backupFile\n";
    echo "Rows backed up: " . count($rows) . "\n";
} catch (Throwable $e) {
    fwrite(STDERR, "Failed to write backup: " . $e->getMessage() . PHP_EOL);
    exit(1);
}

// Now delete rows
try {
    $pdo->beginTransaction();
    $affected = $pdo->exec('DELETE FROM `wh_posicion`');
    if ($affected === false) {
        throw new RuntimeException('DELETE returned false');
    }
    $pdo->commit();
    echo "Deleted rows (DELETE): " . (int)$affected . "\n";
    echo "Done.\n";
    exit(0);
} catch (Throwable $e) {
    if ($pdo->inTransaction()) $pdo->rollBack();
    fwrite(STDERR, "DELETE failed: " . $e->getMessage() . PHP_EOL);
    // Attempt fallback: disable FK checks and TRUNCATE
    try {
        echo "Attempting fallback: TRUNCATE TABLE with foreign key checks disabled...\n";
        $pdo->exec('SET FOREIGN_KEY_CHECKS=0');
        $pdo->exec('TRUNCATE TABLE `wh_posicion`');
        $pdo->exec('SET FOREIGN_KEY_CHECKS=1');
        echo "Truncate successful.\n";
        exit(0);
    } catch (Throwable $e2) {
        fwrite(STDERR, "Fallback TRUNCATE failed: " . $e2->getMessage() . PHP_EOL);
        fwrite(STDERR, "Manual intervention required. Backup file at: $backupFile\n");
        exit(2);
    }
}
