<?php
// scripts/delete_test_positions.php
// Borra posiciones creadas por pruebas del Layout Builder (no interactivo)
declare(strict_types=1);
require_once __DIR__ . '/../config/db.php';
$pdo = getPDO();

// Verify table exists
try {
  $tst = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'wh_posicion'");
  $tst->execute();
  if ((int)$tst->fetchColumn() === 0) {
    echo "Table wh_posicion not found in DB.\n";
    exit(1);
  }
} catch (Throwable $e) {
  echo "Error checking table existence: " . $e->getMessage() . "\n";
  exit(2);
}

// Determine which columns actually exist in wh_posicion and select only those
$candidates = ['id','deposito_id','rack','columna','nivel','fondo','pos_code','pos_code_full','title','created_at','created_by'];
$existingCols = [];
try {
  $colSt = $pdo->prepare("SELECT COLUMN_NAME FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'wh_posicion'");
  $colSt->execute();
  $allCols = $colSt->fetchAll(PDO::FETCH_COLUMN);
  foreach ($candidates as $c) { if (in_array($c, $allCols)) $existingCols[] = $c; }
} catch (Throwable $e) {
  echo "Error fetching columns: " . $e->getMessage() . "\n";
  exit(4);
}

if (empty($existingCols)) {
  echo "No matching columns found on wh_posicion.\n";
  exit(0);
}

$dateFrom = '2025-10-10';
$selectCols = implode(', ', array_map(function($c){ return "`$c`"; }, $existingCols));

// Build WHERE clauses depending on which columns are present
$where = [];
$params = [':dep' => 1, ':posLike' => 'R%', ':dt' => $dateFrom . ' 00:00:00'];
$where[] = "deposito_id = :dep";
$where[] = "pos_code LIKE :posLike";
$where[] = "created_at >= :dt";

$titlePatterns = [':pattern1' => '%Generado por scripts/run_layout_builder_cli.php%', ':pattern2' => '%CLI direct run%'];
$hasCreatedBy = in_array('created_by', $existingCols);
if ($hasCreatedBy) {
  $where[] = "(created_by = :uid OR title LIKE :pattern1 OR title LIKE :pattern2)";
  $params[':uid'] = 1;
  $params += $titlePatterns;
} else {
  // created_by not present: filter only by title patterns
  $where[] = "(title LIKE :pattern1 OR title LIKE :pattern2)";
  $params += $titlePatterns;
}

$sqlSelect = "SELECT $selectCols FROM wh_posicion WHERE " . implode(' AND ', $where) . " ORDER BY id";
try {
  $st = $pdo->prepare($sqlSelect);
  $st->execute($params);
  $rows = $st->fetchAll(PDO::FETCH_ASSOC);
} catch (Throwable $e) {
  echo "Error executing select: " . $e->getMessage() . "\n";
  exit(3);
}

if (!$rows) {
    echo "No se encontraron posiciones candidatas para borrar.\n";
    exit(0);
}

echo "Posiciones candidatas (" . count($rows) . "):\n";
foreach ($rows as $r) {
    printf(" - id=%d code=%s created_at=%s title=%s created_by=%s\n", $r['id'], $r['pos_code_full'], $r['created_at'], $r['title'] ?? '', $r['created_by'] ?? '');
}

// Delete them
$ids = array_map(function($r){ return (int)$r['id']; }, $rows);
$in = implode(',', $ids);
$delSql = "DELETE FROM wh_posicion WHERE id IN ($in)";
$deleted = $pdo->exec($delSql);

echo "\nDeleted rows: " . ($deleted === false ? 0 : $deleted) . "\n";

// Verify
$check = $pdo->prepare("SELECT COUNT(*) FROM wh_posicion WHERE id IN ($in)");
$check->execute();
$remaining = (int)$check->fetchColumn();
echo "Remaining matching rows: $remaining\n";

exit(0);
