<?php
declare(strict_types=1);

/**
 * Genera un dataset de 100 salidas distribuidas en noviembre, basadas en 8 escenarios.
 */

const TOTAL_RECORDS = 100;
const START_DATE = '2025-11-01';
const END_DATE = '2025-11-30';
const OUTPUT_FILE = __DIR__ . '/../tests/data/salidas_noviembre.csv';

$scenarios = [
    'S1' => [
        'zona' => 'Capital',
        'sku_3508_pallets' => 4,
        'sku_3508_unidades' => 0,
        'sku_4409_pallets' => 0,
        'sku_4409_unidades' => 0,
        'devolucion_base' => 5,
    ],
    'S2' => [
        'zona' => 'GBA Norte',
        'sku_3508_pallets' => 0,
        'sku_3508_unidades' => 0,
        'sku_4409_pallets' => 3,
        'sku_4409_unidades' => 10,
        'devolucion_base' => 10,
    ],
    'S3' => [
        'zona' => 'GBA Sur',
        'sku_3508_pallets' => 3,
        'sku_3508_unidades' => 0,
        'sku_4409_pallets' => 2,
        'sku_4409_unidades' => 5,
        'devolucion_base' => 15,
    ],
    'S4' => [
        'zona' => 'Interior Cercano',
        'sku_3508_pallets' => 3,
        'sku_3508_unidades' => 0,
        'sku_4409_pallets' => 3,
        'sku_4409_unidades' => 0,
        'devolucion_base' => 20,
    ],
    'S5' => [
        'zona' => 'Interior Lejano',
        'sku_3508_pallets' => 8,
        'sku_3508_unidades' => 0,
        'sku_4409_pallets' => 0,
        'sku_4409_unidades' => 0,
        'devolucion_base' => 0,
    ],
    'S6' => [
        'zona' => 'Capital',
        'sku_3508_pallets' => 2,
        'sku_3508_unidades' => 20,
        'sku_4409_pallets' => 0,
        'sku_4409_unidades' => 0,
        'devolucion_base' => 10,
    ],
    'S7' => [
        'zona' => 'GBA Norte',
        'sku_3508_pallets' => 0,
        'sku_3508_unidades' => 0,
        'sku_4409_pallets' => 1,
        'sku_4409_unidades' => 30,
        'devolucion_base' => 5,
    ],
    'S8' => [
        'zona' => 'GBA Sur',
        'sku_3508_pallets' => 3,
        'sku_3508_unidades' => 0,
        'sku_4409_pallets' => 3,
        'sku_4409_unidades' => 15,
        'devolucion_base' => 0,
    ],
];

$dates = generateDateSequence(START_DATE, END_DATE, TOTAL_RECORDS);
$records = [];
$scenarioKeys = array_keys($scenarios);

$devolucionToggle = 0;
foreach ($dates as $index => $date) {
    $scenarioKey = $scenarioKeys[$index % count($scenarioKeys)];
    $scenario = $scenarios[$scenarioKey];
    $devolucion = 0;

    if ($devolucionToggle === 1 && $scenario['devolucion_base'] > 0) {
        $devolucion = random_int(max(5, $scenario['devolucion_base'] - 5), $scenario['devolucion_base']);
    }

    $devolucionToggle = ($devolucionToggle + 1) % 2;

    $records[] = [
        'fecha_programada' => $date,
        'scenario_id' => $scenarioKey,
        'cliente_id' => 1,
        'zona' => $scenario['zona'],
        'sku_3508_pallets' => $scenario['sku_3508_pallets'],
        'sku_3508_unidades' => $scenario['sku_3508_unidades'],
        'sku_4409_pallets' => $scenario['sku_4409_pallets'],
        'sku_4409_unidades' => $scenario['sku_4409_unidades'],
        'devolucion_planificada' => $devolucion,
        'observaciones' => '',
    ];
}

writeCsv(OUTPUT_FILE, $records);

echo "Dataset generated at " . OUTPUT_FILE . PHP_EOL;

function generateDateSequence(string $start, string $end, int $count): array
{
    $startDate = new DateTimeImmutable($start);
    $endDate = new DateTimeImmutable($end);
    $interval = $endDate->diff($startDate)->days;

    if ($interval <= 0) {
        throw new RuntimeException('Invalid date range.');
    }

    $dates = [];
    for ($i = 0; $i < $count; $i++) {
        $offset = (int) floor(($i / $count) * $interval);
        $dates[] = $startDate->modify('+' . $offset . ' days')->format('Y-m-d');
    }

    return $dates;
}

function writeCsv(string $filePath, array $records): void
{
    $dir = dirname($filePath);
    if (!is_dir($dir) && !mkdir($dir, 0777, true) && !is_dir($dir)) {
        throw new RuntimeException('Could not create directory: ' . $dir);
    }

    $fp = fopen($filePath, 'w');
    if ($fp === false) {
        throw new RuntimeException('Could not open file for writing: ' . $filePath);
    }

    fputcsv($fp, array_keys($records[0]));
    foreach ($records as $record) {
        fputcsv($fp, $record);
    }

    fclose($fp);
}
