<?php
declare(strict_types=1);

/**
 * CLI helper to load salidas test data from a CSV file and create outbound orders.
 *
 * Usage examples:
 *  php scripts/import_salidas_from_csv.php
 *  php scripts/import_salidas_from_csv.php --csv=tests/data/salidas_noviembre.csv --limit=10 --prepare --deposit=DEP1
 *
 * Options:
 *  --csv=PATH           CSV input path (defaults to tests/data/salidas_noviembre.csv)
 *  --cliente-id=ID      para_clientes.id to attach orders to (defaults to 1)
 *  --order-prefix=STR   Prefix for generated so_pedido.codigo (defaults to SO-AUTO)
 *  --offset=N           Skip the first N rows from the CSV (defaults to 0)
 *  --limit=N            Process at most N rows (defaults to all)
 *  --dry-run            Inspect the CSV and print what would happen without touching the DB
 *  --prepare            After creating each order runs sp_so_preparar_auto(codigo, deposito, NULL, simulateFlag)
 *  --deposit=CODE       Deposito code passed to sp_so_preparar_auto (defaults to DEP1)
 *  --simulate           When used with --prepare executes the stored procedure in simulate mode (fourth argument = 1)
 */

$ROOT = dirname(__DIR__);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

$options = getopt('', [
    'csv::',
    'cliente-id::',
    'order-prefix::',
    'offset::',
    'limit::',
    'dry-run',
    'prepare',
    'deposit::',
    'simulate',
]);

$csvPath = $options['csv'] ?? ($ROOT . '/tests/data/salidas_noviembre.csv');
$clienteId = isset($options['cliente-id']) ? (int)$options['cliente-id'] : 1;
$orderPrefix = $options['order-prefix'] ?? 'SO-AUTO';
$offset = isset($options['offset']) ? max(0, (int)$options['offset']) : 0;
$limit = isset($options['limit']) ? max(0, (int)$options['limit']) : 0;
$dryRun = array_key_exists('dry-run', $options);
$runPrepare = array_key_exists('prepare', $options);
$depositoCode = $options['deposit'] ?? 'DEP1';
$simulate = array_key_exists('simulate', $options);

if (!is_file($csvPath)) {
    fwrite(STDERR, "CSV not found: {$csvPath}\n");
    exit(1);
}

$rows = loadCsv($csvPath);
if ($offset > 0) {
    $rows = array_slice($rows, $offset);
}
if ($limit > 0) {
    $rows = array_slice($rows, 0, $limit);
}
if (!$rows) {
    fwrite(STDERR, "No rows to process after applying offset/limit.\n");
    exit(0);
}

$pdo = get_pdo();
$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
$pdo->exec('SET NAMES utf8mb4');

$clienteInfo = fetchCliente($pdo, $clienteId);
$clienteRef = build_cliente_ref_from_razon($clienteInfo['razon_social'], $clienteInfo['ruc'] ?? null);

ensureSoPedidoEstados($pdo);
$estadoId = (int)$pdo->query("SELECT id FROM so_pedido_estado WHERE code='RECIBIDO' LIMIT 1")->fetchColumn();
if ($estadoId <= 0) {
    fwrite(STDERR, "No se encontró estado RECIBIDO en so_pedido_estado.\n");
    exit(1);
}

ensureAliasSchemaWithClienteId($pdo);
$aliasHasProductoDesc = hasColumn($pdo, 'pl_producto_alias', 'producto_desc');

$destHelpers = buildDestinatarioHelpers($pdo);
$productResolver = buildProductResolver($pdo, $clienteId, $aliasHasProductoDesc);

$orderInserter = buildOrderInserter($pdo, $estadoId);
$destInserter = buildDestInserter($pdo);
$itemInserter = buildItemInserter($pdo);
$aliasUpserter = buildAliasUpserter($pdo, $clienteId, $aliasHasProductoDesc);

$summary = [
    'created' => 0,
    'skipped_existing' => 0,
    'rows_processed' => 0,
    'items_total' => 0,
    'warnings' => [],
];

$orderExistsStmt = $pdo->prepare('SELECT id FROM so_pedido WHERE codigo = ? LIMIT 1');
$prepareSpStmt = null;
if ($runPrepare) {
    try {
        $prepareSpStmt = $pdo->prepare('CALL sp_so_preparar_auto(?, ?, ?, ?)');
    } catch (Throwable $spPrepError) {
        $summary['warnings'][] = 'No se pudo preparar el statement de sp_so_preparar_auto: ' . $spPrepError->getMessage();
        $runPrepare = false;
    }
}

foreach ($rows as $index => $row) {
    $summary['rows_processed']++;
    $pedidoCodigo = sprintf('%s-%04d', $orderPrefix, $offset + $index + 1);
    $items = buildItemsFromRow($row);
    if (!$items) {
        $summary['warnings'][] = "Row {$summary['rows_processed']} skipped: no SKU quantities.";
        continue;
    }

    $observacion = buildObservacion($row);
    $destPayload = buildDestinatarioPayload($row, $pedidoCodigo);
    $docPayload = buildDocumentoPayload($row, $pedidoCodigo);
    $pedidoFecha = $row['fecha_programada'] ?? date('Y-m-d');

    if ($dryRun) {
        echo "[DRY-RUN] Would create {$pedidoCodigo} for cliente {$clienteRef} on {$pedidoFecha} with " . count($items) . " item(s).\n";
        continue;
    }

    $orderExistsStmt->execute([$pedidoCodigo]);
    if ($orderExistsStmt->fetchColumn()) {
        $summary['skipped_existing']++;
        continue;
    }

    try {
        $pdo->beginTransaction();

        $pedidoId = $orderInserter($pedidoCodigo, $clienteRef, $clienteId, $pedidoFecha, $observacion);
        $destinatarioId = resolveDestinatario($pdo, $destHelpers, $clienteId, $destPayload);
        $pedidoDestId = $destInserter($pedidoId, $destinatarioId, $docPayload['tipo'], $docPayload['numero'], $docPayload['nota']);

        foreach ($items as $item) {
            $productoId = $productResolver($item['sku']);
            $aliasUpserter($item['sku'], $productoId);
            $itemInserter($pedidoDestId, $productoId, $item['uv'], $item['uc']);
            $summary['items_total'] += ($item['uv'] + $item['uc']);
        }

        $pdo->commit();
        $summary['created']++;

        if ($runPrepare && $prepareSpStmt !== null) {
            try {
                $prepareSpStmt->execute([$pedidoCodigo, $depositoCode, null, $simulate ? 1 : 0]);
            } catch (Throwable $spError) {
                $summary['warnings'][] = "sp_so_preparar_auto failed for {$pedidoCodigo}: " . $spError->getMessage();
            }
            $prepareSpStmt->closeCursor();
        }
    } catch (Throwable $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        $summary['warnings'][] = "Order {$pedidoCodigo} failed: " . $e->getMessage();
    }
}

printSummary($summary, $dryRun);

function loadCsv(string $path): array
{
    $fp = fopen($path, 'r');
    if ($fp === false) {
        throw new RuntimeException('Could not open CSV: ' . $path);
    }

    $headers = fgetcsv($fp);
    if ($headers === false) {
        throw new RuntimeException('CSV file is empty: ' . $path);
    }

    $rows = [];
    while (($row = fgetcsv($fp)) !== false) {
        if (count($row) === 1 && $row[0] === null) {
            continue;
        }
        $assoc = [];
        foreach ($headers as $idx => $key) {
            $assoc[$key] = $row[$idx] ?? null;
        }
        $rows[] = $assoc;
    }

    fclose($fp);
    return $rows;
}

function fetchCliente(PDO $pdo, int $clienteId): array
{
    $stmt = $pdo->prepare('SELECT razon_social, ruc FROM para_clientes WHERE id = ? LIMIT 1');
    $stmt->execute([$clienteId]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$row) {
        throw new RuntimeException('Cliente not found: ' . $clienteId);
    }
    return $row;
}

function buildItemsFromRow(array $row): array
{
    $items = [];
    $skuMap = [
        'sku_3508_pallets' => ['sku' => '3508', 'target' => 'uv'],
        'sku_3508_unidades' => ['sku' => '3508', 'target' => 'uc'],
        'sku_4409_pallets' => ['sku' => '4409', 'target' => 'uv'],
        'sku_4409_unidades' => ['sku' => '4409', 'target' => 'uc'],
    ];

    foreach ($skuMap as $key => $cfg) {
        $val = isset($row[$key]) ? (int)$row[$key] : 0;
        if ($val <= 0) {
            continue;
        }
        if (!isset($items[$cfg['sku']])) {
            $items[$cfg['sku']] = ['sku' => $cfg['sku'], 'uv' => 0, 'uc' => 0];
        }
        $items[$cfg['sku']][$cfg['target']] += $val;
    }

    return array_values(array_filter($items, static fn(array $item): bool => ($item['uv'] > 0 || $item['uc'] > 0)));
}

function buildObservacion(array $row): ?string
{
    $parts = [];
    $devol = isset($row['devolucion_planificada']) ? (int)$row['devolucion_planificada'] : 0;
    if ($devol > 0) {
        $parts[] = 'Dev planificada ' . $devol;
    }
    $obs = isset($row['observaciones']) ? trim((string)$row['observaciones']) : '';
    if ($obs !== '') {
        $parts[] = $obs;
    }
    if (!$parts) {
        return null;
    }
    $text = implode(' | ', $parts);
    return trunc_str($text, 200);
}

function buildDestinatarioPayload(array $row, string $pedidoCodigo): array
{
    $zona = trim((string)($row['zona'] ?? 'GEN'));
    $scenario = trim((string)($row['scenario_id'] ?? 'S'));
    $cod = strtoupper(preg_replace('/[^A-Z0-9]+/', '', $zona));
    if ($cod === '') {
        $cod = 'ZONA';
    }
    $cod = trunc_str('AUTO-' . $cod, 20);
    $nombre = trunc_str('Destino ' . $zona, 80);
    $razon = trunc_str('Dest ' . $zona, 20);
    $direccion = 'Circuito ' . ($zona !== '' ? $zona : 'Sin zona');
    $telefono = '1100' . str_pad(substr($scenario, -1), 3, '0', STR_PAD_LEFT) . '000';

    return [
        'cod' => $cod,
        'nombre' => $nombre,
        'razon' => $razon,
        'direccion' => $direccion,
        'telefono' => $telefono,
        'referencia' => $pedidoCodigo,
    ];
}

function buildDocumentoPayload(array $row, string $pedidoCodigo): array
{
    $tipo = 'FACTURA';
    $numero = $pedidoCodigo . '-' . ($row['scenario_id'] ?? 'S');
    $nota = null;
    return [
        'tipo' => trunc_str($tipo, 20),
        'numero' => trunc_str($numero, 64),
        'nota' => $nota,
    ];
}

function buildDestinatarioHelpers(PDO $pdo): array
{
    $helpers = [];
    $helpers['hasRazon'] = hasColumn($pdo, 'para_destinatarios', 'razon_social');
    $helpers['hasNombre'] = hasColumn($pdo, 'para_destinatarios', 'nombre');
    $helpers['hasFantasia'] = hasColumn($pdo, 'para_destinatarios', 'fantasia');
    $helpers['hasTelefono'] = hasColumn($pdo, 'para_destinatarios', 'telefono');
    $helpers['hasDireccion'] = hasColumn($pdo, 'para_destinatarios', 'direccion');
    $helpers['hasCiudad'] = hasColumn($pdo, 'para_destinatarios', 'ciudad_id');

    $helpers['findByCod'] = $pdo->prepare('SELECT id FROM para_destinatarios WHERE cliente_id = ? AND cod = ? LIMIT 1');
    if ($helpers['hasRazon']) {
        $helpers['findByName'] = $pdo->prepare('SELECT id FROM para_destinatarios WHERE cliente_id = ? AND (razon_social = ? OR fantasia = ?) LIMIT 1');
    } else {
        $helpers['findByName'] = $pdo->prepare('SELECT id FROM para_destinatarios WHERE cliente_id = ? AND (nombre = ? OR fantasia = ?) LIMIT 1');
    }

    if ($helpers['hasRazon'] && $helpers['hasNombre'] && $helpers['hasTelefono'] && $helpers['hasDireccion'] && $helpers['hasCiudad'] && $helpers['hasFantasia']) {
        $helpers['insert'] = $pdo->prepare('INSERT INTO para_destinatarios (cliente_id, cod, nombre, fantasia, razon_social, telefono, direccion, ciudad_id) VALUES (?,?,?,?,?,?,?,?)');
    } elseif ($helpers['hasRazon'] && $helpers['hasNombre']) {
        $helpers['insert'] = $pdo->prepare('INSERT INTO para_destinatarios (cliente_id, cod, nombre, razon_social) VALUES (?,?,?,?)');
    } elseif ($helpers['hasRazon']) {
        $helpers['insert'] = $pdo->prepare('INSERT INTO para_destinatarios (cliente_id, cod, razon_social) VALUES (?,?,?)');
    } else {
        $helpers['insert'] = $pdo->prepare('INSERT INTO para_destinatarios (cliente_id, cod, nombre) VALUES (?,?,?)');
    }

    return $helpers;
}

function resolveDestinatario(PDO $pdo, array $helpers, int $clienteId, array $payload): int
{
    $cod = $payload['cod'];
    $nombre = $payload['nombre'];
    $razon = $payload['razon'] ?? $nombre;
    $direccion = $payload['direccion'];
    $telefono = $payload['telefono'];
    $telefonoDigits = preg_replace('/[^0-9]/', '', $telefono) ?: '0';

    $helpers['findByCod']->execute([$clienteId, $cod]);
    $existing = (int)$helpers['findByCod']->fetchColumn();
    if ($existing > 0) {
        return $existing;
    }

    $helpers['findByName']->execute([$clienteId, $nombre, $nombre]);
    $existing = (int)$helpers['findByName']->fetchColumn();
    if ($existing > 0) {
        return $existing;
    }

    $insert = $helpers['insert'];
    if ($helpers['hasRazon'] && $helpers['hasNombre'] && $helpers['hasTelefono'] && $helpers['hasDireccion'] && $helpers['hasCiudad'] && $helpers['hasFantasia']) {
        $insert->execute([$clienteId, $cod, $nombre, $nombre, $razon, $telefonoDigits, $direccion, 0]);
    } elseif ($helpers['hasRazon'] && $helpers['hasNombre']) {
        $insert->execute([$clienteId, $cod, $nombre, $razon]);
    } elseif ($helpers['hasRazon']) {
        $insert->execute([$clienteId, $cod, $razon]);
    } else {
        $insert->execute([$clienteId, $cod, $nombre]);
    }

    return (int)$pdo->lastInsertId();
}

function buildOrderInserter(PDO $pdo, int $estadoId): callable
{
    $stmt = $pdo->prepare('INSERT INTO so_pedido (codigo, cliente_ref, cliente_id, fecha_pedido, estado_id, observacion) VALUES (?,?,?,?,?,?)');

    return static function (string $codigo, string $clienteRef, int $clienteId, string $fechaPedido, ?string $observacion) use ($stmt, $estadoId, $pdo): int {
        $stmt->execute([$codigo, $clienteRef, $clienteId, $fechaPedido, $estadoId, $observacion]);
        return (int)$pdo->lastInsertId();
    };
}

function buildDestInserter(PDO $pdo): callable
{
    $stmt = $pdo->prepare('INSERT INTO so_pedido_dest (pedido_id, destinatario_id, doc_tipo, doc_numero, nota) VALUES (?,?,?,?,?)');

    return static function (int $pedidoId, int $destinatarioId, ?string $docTipo, ?string $docNumero, ?string $nota) use ($stmt, $pdo): int {
        $stmt->execute([$pedidoId, $destinatarioId, $docTipo, $docNumero, $nota]);
        return (int)$pdo->lastInsertId();
    };
}

function buildItemInserter(PDO $pdo): callable
{
    $stmt = $pdo->prepare('INSERT INTO so_pedido_dest_item (pedido_dest_id, producto_id, lote_codigo, expected_uv, expected_uc) VALUES (?,?,?,?,?)');

    return static function (int $pedidoDestId, int $productoId, int $uv, int $uc) use ($stmt): void {
        $stmt->execute([$pedidoDestId, $productoId, null, $uv, $uc]);
    };
}

function buildAliasUpserter(PDO $pdo, int $clienteId, bool $aliasHasProductoDesc): callable
{
    if ($aliasHasProductoDesc) {
        $stmt = $pdo->prepare('INSERT INTO pl_producto_alias (cliente_id, sku_cliente, producto_id, producto_desc) VALUES (?,?,?,?) ON DUPLICATE KEY UPDATE producto_id = VALUES(producto_id), producto_desc = VALUES(producto_desc)');
    } else {
        $stmt = $pdo->prepare('INSERT INTO pl_producto_alias (cliente_id, sku_cliente, producto_id) VALUES (?,?,?) ON DUPLICATE KEY UPDATE producto_id = VALUES(producto_id)');
    }

    return static function (string $sku, int $productoId) use ($stmt, $clienteId, $aliasHasProductoDesc): void {
        if ($aliasHasProductoDesc) {
            $stmt->execute([$clienteId, $sku, $productoId, null]);
        } else {
            $stmt->execute([$clienteId, $sku, $productoId]);
        }
    };
}

function buildProductResolver(PDO $pdo, int $clienteId, bool $aliasHasProductoDesc): callable
{
    $findAlias = $pdo->prepare('SELECT producto_id FROM pl_producto_alias WHERE cliente_id = ? AND sku_cliente = ? LIMIT 1');
    $findProd = $pdo->prepare('SELECT id FROM para_productos WHERE sku = ? LIMIT 1');
    $insertProd = $pdo->prepare('INSERT INTO para_productos (sku, denominacion) VALUES (?, ?)');

    return static function (string $sku) use ($pdo, $clienteId, $findAlias, $findProd, $insertProd): int {
        $findAlias->execute([$clienteId, $sku]);
        $prodId = (int)$findAlias->fetchColumn();
        if ($prodId > 0) {
            return $prodId;
        }

        $findProd->execute([$sku]);
        $prodId = (int)$findProd->fetchColumn();
        if ($prodId > 0) {
            return $prodId;
        }

        $den = 'SKU ' . $sku;
        $insertProd->execute([$sku, $den]);
        return (int)$pdo->lastInsertId();
    };
}

function ensureAliasSchemaWithClienteId(PDO $pdo): void
{
    if (!hasColumn($pdo, 'pl_producto_alias', 'cliente_id')) {
        try {
            $pdo->exec('ALTER TABLE pl_producto_alias ADD COLUMN cliente_id BIGINT UNSIGNED NULL AFTER id');
        } catch (Throwable $e) {
            // no-op if another process handled it
        }
    }

    try {
        $stmt = $pdo->query("SELECT COUNT(1) FROM INFORMATION_SCHEMA.STATISTICS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'pl_producto_alias' AND INDEX_NAME = 'uk_alias_cli_id'");
        $exists = (int)$stmt->fetchColumn() > 0;
        if (!$exists) {
            $pdo->exec('ALTER TABLE pl_producto_alias ADD UNIQUE KEY uk_alias_cli_id (cliente_id, sku_cliente)');
        }
    } catch (Throwable $e) {
        // ignore metadata failures
    }
}

function ensureSoPedidoEstados(PDO $pdo): void
{
    static $done = false;
    if ($done) {
        return;
    }

    $defaults = [
        ['RECIBIDO', 'Recibido', 1],
        ['EN_PREPARACION', 'En preparación', 2],
        ['PARCIAL', 'Parcialmente preparado', 3],
        ['PREPARADO', 'Preparado', 4],
        ['EN_EMBARQUE', 'En embarque', 5],
        ['DESPACHADO', 'Despachado', 6],
        ['CERRADO', 'Cerrado', 7],
        ['CANCELADO', 'Cancelado', 9],
    ];

    $stmt = $pdo->prepare('INSERT INTO so_pedido_estado (code, nombre, orden) VALUES (?,?,?) ON DUPLICATE KEY UPDATE nombre = VALUES(nombre), orden = VALUES(orden)');
    foreach ($defaults as $row) {
        try {
            $stmt->execute($row);
        } catch (Throwable $e) {
            // ignore
        }
    }

    $done = true;
}

function hasColumn(PDO $pdo, string $table, string $col): bool
{
    static $cache = [];
    $key = $table . '|' . $col;
    if (array_key_exists($key, $cache)) {
        return $cache[$key];
    }
    $stmt = $pdo->prepare('SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?');
    $stmt->execute([$table, $col]);
    $cache[$key] = ((int)$stmt->fetchColumn() > 0);
    return $cache[$key];
}

function trunc_str(?string $s, int $max): string
{
    $s = (string)$s;
    if ($max <= 0) {
        return '';
    }
    if (function_exists('mb_substr')) {
        return mb_substr($s, 0, $max);
    }
    return substr($s, 0, $max);
}

function build_cliente_ref_from_razon(string $razon, ?string $ruc = null): string
{
    $s = $razon;
    if (function_exists('iconv')) {
        $tmp = @iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $s);
        if ($tmp !== false) {
            $s = $tmp;
        }
    }
    $s = strtoupper($s);
    $s = preg_replace('/[^A-Z0-9 ]+/', ' ', $s);
    $tokens = preg_split('/\s+/', trim($s)) ?: [];
    $stop = ['SA','SAS','SRL','SAC','LTDA','LTD','CORP','CORPORATION','COMPANIA','CIA','DE','LA','EL','LOS','LAS','Y'];
    $alias = '';
    foreach ($tokens as $t) {
        if ($t === '' || in_array($t, $stop, true)) {
            continue;
        }
        $alias = $t;
        break;
    }
    if ($alias === '') {
        $ruc = (string)$ruc;
        $ruc = preg_replace('/[^0-9A-Z]+/', '', strtoupper($ruc ?? 'GEN'));
        $alias = $ruc !== '' ? $ruc : 'GEN';
    }
    return 'CLI-' . $alias;
}

function printSummary(array $summary, bool $dryRun): void
{
    echo "\n";
    if ($dryRun) {
        echo "Dry-run complete.\n";
    } else {
        echo "Import complete.\n";
    }
    echo 'Rows processed: ' . $summary['rows_processed'] . "\n";
    if (!$dryRun) {
        echo 'Orders created: ' . $summary['created'] . "\n";
        echo 'Orders skipped (existing code): ' . $summary['skipped_existing'] . "\n";
        echo 'Total items requested (uv + uc): ' . $summary['items_total'] . "\n";
    }
    if (!empty($summary['warnings'])) {
        echo "Warnings:\n";
        foreach ($summary['warnings'] as $warn) {
            echo ' - ' . $warn . "\n";
        }
    }
}
