<?php
declare(strict_types=1);

/**
 * Migración inventario 709 -> SOL (pallets + items)
 * Autor: Raul + ChatGPT
 *
 * Requisitos:
 * - MySQL 8.x
 * - PDO habilitado
 * - Acceso lectura (DB 709) y escritura (DB sol)
 *
 * LÓGICA:
 * - Lee 709.cstock filtrado por $COD_INVENTARIO
 * - Une a 709.para_productos para obtener equivalencias (cajaspallets, unicajas, cod)
 * - Resuelve producto destino por SKU (sol.para_productos.sku = 709.para_productos.cod)
 * - Crea N pallets completos (pallets * cajaspallets por pallet)
 * - Crea pallet “sueltos” si hay cajaspick/unidadespick
 * - Inserta wh_pallet y wh_pallet_item
 * - Evita duplicados usando códigos de pallet únicos
 */

ini_set('display_errors', '1');
error_reporting(E_ALL);

//////////////////////////////
// CONFIGURACIÓN
//////////////////////////////
$COD_INVENTARIO = '09-2025'; // <-- cambia si corresponde

// Cargar .env y helper de conexión principal (SOL)
require_once __DIR__ . '/../config/db.php';

// Conexión origen (709) tomada desde variables de entorno
// Espera en .env:
//   DB709_HOST, DB709_NAME, DB709_USER, DB709_PASS, (opcionales) DB709_CHARSET, DB709_COLLATION, DB709_PERSISTENT, DB709_TIMEOUT
// Fallbacks seguros si no están definidos
$SRC_DSN  = sprintf(
    'mysql:host=%s;dbname=%s;charset=%s',
    $_ENV['DB709_HOST']    ?? 'localhost',
    $_ENV['DB709_NAME']    ?? '709',
    $_ENV['DB709_CHARSET'] ?? 'utf8mb4'
);
$SRC_USER = $_ENV['DB709_USER'] ?? ($_ENV['DB_USER'] ?? 'root');
$SRC_PASS = $_ENV['DB709_PASS'] ?? ($_ENV['DB_PASS'] ?? '');

// Parámetros de inserción en SOL
$DEPOSITO_ID            = 1;        // requerido por wh_pallet
$ESTADO_ID              = 1;        // requerido por wh_pallet (ajusta a tu catálogo)
$POSICION_ID            = null;     // deja null por ahora o coloca una posición por defecto
$DEFAULT_CLIENTE_ID     = null;     // fallback si no logramos inferir cliente_id del producto destino
$OBS_MIGRACION_PREFIX   = "Migración 709 $COD_INVENTARIO";

// Modo simulación (no escribe en DB destino)
$DRY_RUN = false;

// Prefijo/plantilla de códigos de pallet
// Ej: PAL-INV-09-2025-<SKU>-0001  /  PAL-INV-09-2025-<SKU>-SUELTOS
function makePalletCode(string $sku, ?int $n = null, string $codInventario = '09-2025'): string {
    if ($n === null) {
        return sprintf("PAL-INV-%s-%s-SUELTOS", preg_replace('/\s+/', '', $codInventario), $sku);
    }
    return sprintf("PAL-INV-%s-%s-%04d", preg_replace('/\s+/', '', $codInventario), $sku, $n);
}

/**
 * HOOK: Resolver/crear lote y devolver lote_id.
 * IMPORTANTE: AJUSTA esta función a tu tabla real de lotes en SOL.
 * - Si ya tienes tabla/servicio de lotes (p.ej. para_lote o wh_lote), resuelve por (sku + $COD_INVENTARIO) o
 *   crea 1 registro con nombre "INV-09-2025" y reúsalo para todos los items de esta migración.
 *
 * Debe devolver un int válido.
 */
function getOrCreateLoteId(PDO $pdoSol, string $sku, string $codInventario): int {
    // TODO: Implementar según tu esquema real. Aquí dejo un mock sobre tabla hipotética `para_lote`.
    // Si NO tienes esa tabla, crea una mínima o cambia la consulta/insert a la tabla correcta.

    // 1) Intentar fetch de un lote único por inventario
    $loteNombre = "INV-$codInventario";
    $sqlSel = "SELECT id FROM para_lote WHERE nombre = :n LIMIT 1";
    try {
        $stmt = $pdoSol->prepare($sqlSel);
        $stmt->execute([':n' => $loteNombre]);
        $id = $stmt->fetchColumn();
        if ($id) return (int)$id;
    } catch (\PDOException $e) {
        // Si no existe la tabla, lanza una excepción clara:
        throw new \RuntimeException("Ajusta getOrCreateLoteId(): no existe tabla para_lote en SOL. Define tu tabla real de lotes.", 0, $e);
    }

    // 2) Crear si no existe
    $sqlIns = "INSERT INTO para_lote (nombre, created_at) VALUES (:n, NOW())";
    $pdoSol->prepare($sqlIns)->execute([':n' => $loteNombre]);
    return (int)$pdoSol->lastInsertId();
}

//////////////////////////////
// CONEXIONES
//////////////////////////////
// 1) Origen 709 con opciones similares a config/db.php
$srcOptions = [
    PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    PDO::ATTR_EMULATE_PREPARES   => false,
];
$srcTimeout = isset($_ENV['DB709_TIMEOUT']) ? (int)$_ENV['DB709_TIMEOUT'] : (isset($_ENV['DB_TIMEOUT']) ? (int)$_ENV['DB_TIMEOUT'] : 5);
if ($srcTimeout > 0) { $srcOptions[PDO::ATTR_TIMEOUT] = $srcTimeout; }
if (!empty($_ENV['DB709_PERSISTENT'])) { $srcOptions[PDO::ATTR_PERSISTENT] = true; }

$pdo709 = new PDO($SRC_DSN, $SRC_USER, $SRC_PASS, $srcOptions);

// Ajustar collation de sesión si se definió
if (!empty($_ENV['DB709_COLLATION'])) {
    $srcCharset = $_ENV['DB709_CHARSET'] ?? 'utf8mb4';
    $srcCollate = $_ENV['DB709_COLLATION'];
    try {
        $pdo709->exec(sprintf('SET NAMES %s COLLATE %s', $srcCharset, $srcCollate));
        $pdo709->exec("SET collation_connection = '" . str_replace("'", "''", $srcCollate) . "'");
    } catch (Throwable $e) {
        error_log('DB709 collation session setup failed: ' . $e->getMessage());
    }
}

// 2) Destino SOL usando el helper centralizado (lee DB_HOST/DB_NAME/DB_USER/DB_PASS del .env)
$pdoSol = getPDO();

//////////////////////////////
// INPUT: cstock + para_productos (709)
//////////////////////////////
$sql = <<<SQL
SELECT
  cs.id            AS cstock_id,
  cs.codigo        AS producto_id_709,   -- suele mapear a 709.para_productos.id
  cs.pallets,
  cs.cajaspick,
  cs.unidadespick,
  cs.unidades,
  p.cod            AS sku_709,           -- para match con sol.para_productos.sku
  p.denominacion,
  p.cliente_id     AS cliente_id_709,
  COALESCE(NULLIF(p.cajaspallets, 0), 0) AS cajaspallets,
  COALESCE(NULLIF(p.unicajas, 0), 0)     AS unicajas
FROM cstock cs
JOIN para_productos p ON p.id = cs.codigo
WHERE cs.codInventario = :inv
SQL;

$rows = $pdo709->prepare($sql);
$rows->execute([':inv' => $COD_INVENTARIO]);
$data = $rows->fetchAll();

if (!$data) {
    die("No hay registros en 709.cstock para codInventario = {$COD_INVENTARIO}\n");
}

//////////////////////////////
// PREP DESTINO: resolver productos SOL por SKU
//////////////////////////////
$getProdSol = $pdoSol->prepare("
  SELECT id, cliente_id
  FROM para_productos
  WHERE sku = :sku
  LIMIT 1
");

//////////////////////////////
// PREP DESTINO: insertar pallet + item
//////////////////////////////
$insPallet = $pdoSol->prepare("
  INSERT INTO wh_pallet
    (codigo, deposito_id, cliente_id, posicion_id, estado_id, pickeado, reservado, observacion, created_at, updated_at)
  VALUES
    (:codigo, :deposito_id, :cliente_id, :posicion_id, :estado_id, 0, 0, :obs, NOW(), NOW())
");

$selPalletId = $pdoSol->prepare("SELECT id FROM wh_pallet WHERE codigo = :codigo LIMIT 1");

$insItem = $pdoSol->prepare("
  INSERT INTO wh_pallet_item
    (pallet_id, producto_id, lote_id, uv_cajas, uc_unidades, created_at, updated_at)
  VALUES
    (:pallet_id, :producto_id, :lote_id, :uv_cajas, :uc_unidades, NOW(), NOW())
");

//////////////////////////////
// TRANSACCIÓN
//////////////////////////////
$pdoSol->beginTransaction();

try {
    // Resolver un lote_id global por inventario (simple y consistente)
    $loteIdGlobal = getOrCreateLoteId($pdoSol, 'ALL', $COD_INVENTARIO);

    $report = [
        'productos_ok'      => 0,
        'pallets_creados'   => 0,
        'pallets_sueltos'   => 0,
        'items_creados'     => 0,
        'productos_sin_match' => [],
        'errores'           => []
    ];

    foreach ($data as $r) {
        $sku709       = trim((string)$r['sku_709']);
        $pallets      = (int)$r['pallets'];
        $cajaPorPal   = (int)$r['cajaspallets'];
        $cajasSuelto  = max(0, (int)$r['cajaspick']);
        $unidSuelto   = max(0, (int)$r['unidadespick']);

        if ($sku709 === '' || $sku709 === null) {
            $report['productos_sin_match'][] = ['motivo' => 'sku_709 vacío', 'row' => $r];
            continue;
        }

        // Resolver producto destino
        $getProdSol->execute([':sku' => $sku709]);
        $prodSol = $getProdSol->fetch();

        if (!$prodSol) {
            $report['productos_sin_match'][] = ['motivo' => 'sku no existe en SOL', 'sku' => $sku709, 'row' => $r];
            continue;
        }

        $productoIdSol = (int)$prodSol['id'];
        $clienteIdSol  = $prodSol['cliente_id'] !== null ? (int)$prodSol['cliente_id'] : ($DEFAULT_CLIENTE_ID ?? null);

        if ($clienteIdSol === null) {
            $report['productos_sin_match'][] = ['motivo' => 'sin cliente_id en SOL y sin DEFAULT_CLIENTE_ID', 'sku' => $sku709, 'row' => $r];
            continue;
        }

        $report['productos_ok']++;

        // Pallets completos
        for ($i = 1; $i <= $pallets; $i++) {
            $codigoPallet = makePalletCode($sku709, $i, $COD_INVENTARIO);

            // Idempotencia: si ya existe el pallet con ese código, no lo insertes de nuevo
            $selPalletId->execute([':codigo' => $codigoPallet]);
            $pid = $selPalletId->fetchColumn();

            if (!$pid && !$DRY_RUN) {
                $insPallet->execute([
                    ':codigo'      => $codigoPallet,
                    ':deposito_id' => $DEPOSITO_ID,
                    ':cliente_id'  => $clienteIdSol,
                    ':posicion_id' => $POSICION_ID,
                    ':estado_id'   => $ESTADO_ID,
                    ':obs'         => "{$OBS_MIGRACION_PREFIX} | SKU {$sku709} | Pallet completo"
                ]);
                $pid = (int)$pdoSol->lastInsertId();
                $report['pallets_creados']++;
            }

            if ($pid && !$DRY_RUN) {
                $uv = max(0, $cajaPorPal);
                $insItem->execute([
                    ':pallet_id'   => $pid,
                    ':producto_id' => $productoIdSol,
                    ':lote_id'     => $loteIdGlobal,
                    ':uv_cajas'    => $uv,
                    ':uc_unidades' => 0
                ]);
                $report['items_creados']++;
            }
        }

        // Pallet “sueltos” (si corresponde)
        if ($cajasSuelto > 0 || $unidSuelto > 0) {
            $codigoPallet = makePalletCode($sku709, null, $COD_INVENTARIO);

            $selPalletId->execute([':codigo' => $codigoPallet]);
            $pid = $selPalletId->fetchColumn();

            if (!$pid && !$DRY_RUN) {
                $insPallet->execute([
                    ':codigo'      => $codigoPallet,
                    ':deposito_id' => $DEPOSITO_ID,
                    ':cliente_id'  => $clienteIdSol,
                    ':posicion_id' => $POSICION_ID,
                    ':estado_id'   => $ESTADO_ID,
                    ':obs'         => "{$OBS_MIGRACION_PREFIX} | SKU {$sku709} | Pallet sueltos"
                ]);
                $pid = (int)$pdoSol->lastInsertId();
                $report['pallets_sueltos']++;
            }

            if ($pid && !$DRY_RUN) {
                $insItem->execute([
                    ':pallet_id'   => $pid,
                    ':producto_id' => $productoIdSol,
                    ':lote_id'     => $loteIdGlobal,
                    ':uv_cajas'    => $cajasSuelto,
                    ':uc_unidades' => $unidSuelto
                ]);
                $report['items_creados']++;
            }
        }
    }

    if ($DRY_RUN) {
        $pdoSol->rollBack();
        echo "=== DRY RUN ===\n";
        print_r($report);
        exit;
    }

    $pdoSol->commit();
    echo "Migración completada.\n";
    print_r($report);

} catch (\Throwable $e) {
    $pdoSol->rollBack();
    echo "ERROR en migración: " . $e->getMessage() . "\n";
    exit(1);
}
