<?php
// scripts/normalize_collation.php
// CLI: php scripts/normalize_collation.php [--collation=utf8mb4_0900_ai_ci] [--dry-run]
// Normaliza TODAS las tablas del schema actual a un charset/collation objetivo.

declare(strict_types=1);

require_once __DIR__ . '/../config/db.php';

function println(string $s = ''): void { echo $s . PHP_EOL; }

try {
    $pdo = getPDO();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec('SET NAMES utf8mb4');

    // Parse args
    $args = $argv ?? [];
    $targetCollation = 'utf8mb4_0900_ai_ci';
    $dryRun = false;
    foreach ($args as $a) {
        if (strpos($a, '--collation=') === 0) {
            $targetCollation = substr($a, strlen('--collation='));
        } elseif ($a === '--dry-run') {
            $dryRun = true;
        }
    }

    $targetCharset = 'utf8mb4';
    if (stripos($targetCollation, 'utf8mb4') !== 0) {
        // Infer charset from collation name simple heuristic
        $targetCharset = explode('_', $targetCollation)[0] ?: 'utf8mb4';
    }

    // Detect current database
    $db = $pdo->query('SELECT DATABASE()')->fetchColumn();
    if (!$db) {
        throw new RuntimeException('No database selected in DSN');
    }

    println("Normalizando collation para DB '$db' → $targetCharset / $targetCollation" . ($dryRun ? ' (dry-run)' : ''));

    // Show current collations
    $vars = $pdo->query("SHOW VARIABLES LIKE 'collation_%'")->fetchAll(PDO::FETCH_KEY_PAIR);
    println('Collation variables (antes):');
    foreach ($vars as $k => $v) println("  $k = $v");

    // Alter database collation
    $sqlAlterDb = sprintf('ALTER DATABASE `%s` CHARACTER SET %s COLLATE %s', $db, $targetCharset, $targetCollation);
    println('> ' . $sqlAlterDb);
    if (!$dryRun) {
        $pdo->exec($sqlAlterDb);
    }

    // List tables
    $st = $pdo->prepare('SELECT TABLE_NAME FROM information_schema.TABLES WHERE TABLE_SCHEMA = ? AND TABLE_TYPE = "BASE TABLE" ORDER BY TABLE_NAME');
    $st->execute([$db]);
    $tables = $st->fetchAll(PDO::FETCH_COLUMN) ?: [];
    if (!$tables) {
        println('No se encontraron tablas.');
        exit(0);
    }

    $ok = 0; $fail = 0;
    foreach ($tables as $t) {
        $sql = sprintf('ALTER TABLE `%s` CONVERT TO CHARACTER SET %s COLLATE %s', $t, $targetCharset, $targetCollation);
        println('> ' . $sql);
        if ($dryRun) continue;
        try {
            $pdo->exec($sql);
            $ok++;
        } catch (Throwable $e) {
            $fail++;
            println('  ! Error: ' . $e->getMessage());
        }
    }

    // Summary
    println(sprintf('Hecho. OK=%d, FAIL=%d%s', $ok, $fail, $dryRun ? ' (dry-run)' : ''));

    // Report any remaining columns with mismatched collation
    $st2 = $pdo->prepare('SELECT TABLE_NAME, COLUMN_NAME, COLLATION_NAME FROM information_schema.COLUMNS WHERE TABLE_SCHEMA=? AND COLLATION_NAME IS NOT NULL AND COLLATION_NAME<>? ORDER BY TABLE_NAME, ORDINAL_POSITION');
    $st2->execute([$db, $targetCollation]);
    $left = $st2->fetchAll(PDO::FETCH_ASSOC) ?: [];
    if ($left) {
        println('Columnas que aún difieren de la collation objetivo:');
        foreach ($left as $row) {
            println(sprintf('  %s.%s -> %s', $row['TABLE_NAME'], $row['COLUMN_NAME'], $row['COLLATION_NAME']));
        }
        println('Nota: Si permanecen, podrían tener collation explícita o ser vistas/materializadas. Revise manualmente.');
    } else {
        println('Todas las columnas de texto están en la collation objetivo.');
    }

    exit(0);
} catch (Throwable $e) {
    fwrite(STDERR, 'Error: ' . $e->getMessage() . PHP_EOL);
    exit(1);
}
