<?php
/**
 * Genera registros simulados para so_embarque_rendiciones tomando como base
 * los pedidos asociados a cada embarque. Asigna montos aproximados según la
 * cantidad enviada por SKU y fija la condición a FCO.
 *
 * Uso típico:
 *   php scripts/populate_embarque_rendiciones.php --embarque-like=EMB-202511-%
 *
 * Opciones:
 *   --embarque-like=PAT    Filtra códigos de embarque (default EMB-202511-%).
 *   --pedido-like=PAT      Filtra pedidos asociados (default SO-AUTO-%).
 *   --dry-run              Muestra la simulación sin escribir en la base.
 *   --rewrite              Borra rendiciones existentes del embarque antes de insertar.
 *   --limit=N              Procesa a lo sumo N embarques.
 *   --seed=N               Semilla para el generador aleatorio (repetible).
 */

declare(strict_types=1);


$ROOT = dirname(__DIR__);

require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';
require_once $ROOT . '/app/Support/InventarioReport.php';
require_once $ROOT . '/app/Support/EmbarqueDocs.php';

$options = getopt('', [
    'embarque-like::',
    'pedido-like::',
    'dry-run',
    'rewrite',
    'limit::',
    'seed::',
]);

$embarqueLike = $options['embarque-like'] ?? 'EMB-202511-%';
$pedidoLike = $options['pedido-like'] ?? 'SO-AUTO-%';
$dryRun = array_key_exists('dry-run', $options);
$rewrite = array_key_exists('rewrite', $options);
$limit = isset($options['limit']) ? max(0, (int)$options['limit']) : 0;
$seed = isset($options['seed']) ? (int)$options['seed'] : null;

if ($seed !== null) {
    mt_srand($seed);
}

$PRICE_PER_UNIT = [
    '3508' => 18500.0,
    '4409' => 23500.0,
    '__DEFAULT__' => 16500.0,
];

try {
    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec('SET NAMES utf8mb4');

    $embarques = fetchEmbarquesConPedidos($pdo, $embarqueLike, $pedidoLike);
    if (!$embarques) {
        echo "No se encontraron embarques para {$embarqueLike}.\n";
        exit(0);
    }
    if ($limit > 0 && count($embarques) > $limit) {
        $embarques = array_slice($embarques, 0, $limit);
    }

    $existingStmt = $pdo->prepare('SELECT COUNT(*) FROM so_embarque_rendiciones WHERE embarque_id = ?');
    $deleteStmt = $pdo->prepare('DELETE FROM so_embarque_rendiciones WHERE embarque_id = ?');
    $insertStmt = $pdo->prepare(
        'INSERT INTO so_embarque_rendiciones (embarque_id, factura, condicion, fecha_factura, monto, fecha_rendicion)
         VALUES (:emb, :factura, :condicion, :fecha_factura, :monto, :fecha_rendicion)'
    );

    $summary = [
        'processed' => 0,
        'skipped_existing' => 0,
        'rows_created' => 0,
        'rows_preview' => 0,
        'rewritten' => 0,
    ];

    foreach ($embarques as $emb) {
        $embarqueId = (int)$emb['id'];
        $embarqueCodigo = (string)$emb['codigo'];
        $existing = fetchCount($existingStmt, $embarqueId);
        if ($existing > 0 && !$rewrite) {
            echo "Omitido {$embarqueCodigo}: ya tiene rendiciones (use --rewrite para reemplazar).\n";
            $summary['skipped_existing']++;
            continue;
        }

        $context = embarque_collect_destinatarios_docs($pdo, $embarqueId);
        $pedidoDocs = $context['pedido_docs'] ?? [];

        $facturaRows = [];
        $pedidoCount = 0;
        foreach ($emb['pedidos'] as $pedido) {
            $pedidoCount++;
            $factura = resolveFactura($pdo, $pedido, $pedidoDocs);
            $items = fetchPedidoItems($pdo, $pedido['id']);
            if (!$items) {
                continue;
            }
            $productInfo = inventario_fetch_product_info($pdo, array_column($items, 'producto_id'));
            $monto = calculateMonto($items, $productInfo, $PRICE_PER_UNIT);
            if ($monto <= 0) {
                $monto = mt_rand(75000, 140000);
            }
            $facturaRows[] = [
                'factura' => $factura,
                'monto' => $monto,
            ];
        }

        if (!$facturaRows) {
            echo "Embarque {$embarqueCodigo}: sin facturas detectadas para generar rendiciones.\n";
            continue;
        }

        $vueltaDate = resolveVueltaDate($pdo, $embarqueId, $emb);
        $fechaFactura = $vueltaDate->format('Y-m-d');
        $rowsToInsert = aggregateFacturas($facturaRows, $fechaFactura, $vueltaDate);

        if (empty($rowsToInsert)) {
            echo "Embarque {$embarqueCodigo}: no se generaron montos válidos.\n";
            continue;
        }

        if ($dryRun) {
            echo "[DRY-RUN] Embarque {$embarqueCodigo}: " . count($rowsToInsert) . " rendición(es) generada(s).\n";
            foreach ($rowsToInsert as $row) {
                echo sprintf(
                    "  - Factura %s | Monto %.2f | Fecha %s %s\n",
                    $row['factura'],
                    $row['monto'],
                    $row['fecha_factura'],
                    $row['fecha_rendicion']
                );
            }
            $summary['rows_preview'] += count($rowsToInsert);
            continue;
        }

        $pdo->beginTransaction();
        try {
            if ($existing > 0) {
                $deleteStmt->execute([$embarqueId]);
                $summary['rewritten']++;
            }
            foreach ($rowsToInsert as $row) {
                $insertStmt->execute([
                    ':emb' => $embarqueId,
                    ':factura' => $row['factura'],
                    ':condicion' => 'FCO',
                    ':fecha_factura' => $row['fecha_factura'],
                    ':monto' => $row['monto'],
                    ':fecha_rendicion' => $row['fecha_rendicion'],
                ]);
            }
            $pdo->commit();
            $summary['processed']++;
            $summary['rows_created'] += count($rowsToInsert);
            echo "Embarque {$embarqueCodigo}: registradas " . count($rowsToInsert) . " rendiciones.\n";
        } catch (Throwable $tx) {
            if ($pdo->inTransaction()) {
                $pdo->rollBack();
            }
            throw $tx;
        }
    }

    if ($dryRun) {
        echo "\nResumen (dry-run): embarques evaluados " . count($embarques) . ", rendiciones simuladas {$summary['rows_preview']}.\n";
    } else {
        echo "\nResumen: procesados {$summary['processed']}, omitidos {$summary['skipped_existing']}, filas nuevas {$summary['rows_created']}";
        if ($summary['rewritten'] > 0) {
            echo ", embarques reescritos {$summary['rewritten']}";
        }
        echo ".\n";
    }

    exit(0);
} catch (Throwable $e) {
    fwrite(STDERR, 'Error: ' . $e->getMessage() . "\n");
    exit(1);
}

function fetchEmbarquesConPedidos(PDO $pdo, string $embarqueLike, string $pedidoLike): array
{
    $sql = 'SELECT e.id, e.codigo, e.salida_at, e.updated_at, e.creado_at
              FROM so_embarque e
              WHERE e.codigo LIKE :emb
              ORDER BY e.codigo';
    $stmt = $pdo->prepare($sql);
    $stmt->execute([':emb' => $embarqueLike]);
    $embarques = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
    if (!$embarques) {
        return [];
    }

    $sqlPedidos = 'SELECT e.id AS embarque_id, ped.id AS pedido_id, ped.codigo, ped.observacion
                     FROM so_embarque e
                     JOIN so_embarque_pre ep ON ep.embarque_id = e.id
                     JOIN so_preembarque pre ON pre.id = ep.preembarque_id
                     JOIN so_pedido ped ON ped.id = pre.pedido_id
                    WHERE e.codigo LIKE :emb AND ped.codigo LIKE :ped
                    ORDER BY e.codigo, ped.codigo';
    $stPedidos = $pdo->prepare($sqlPedidos);
    $stPedidos->execute([
        ':emb' => $embarqueLike,
        ':ped' => $pedidoLike,
    ]);
    $rows = $stPedidos->fetchAll(PDO::FETCH_ASSOC) ?: [];

    $map = [];
    foreach ($embarques as $emb) {
        $map[$emb['id']] = [
            'id' => (int)$emb['id'],
            'codigo' => (string)$emb['codigo'],
            'salida_at' => $emb['salida_at'] ?? null,
            'updated_at' => $emb['updated_at'] ?? null,
            'creado_at' => $emb['creado_at'] ?? null,
            'pedidos' => [],
        ];
    }
    foreach ($rows as $row) {
        $embId = (int)$row['embarque_id'];
        if (!isset($map[$embId])) {
            continue;
        }
        $map[$embId]['pedidos'][] = [
            'id' => (int)$row['pedido_id'],
            'codigo' => (string)$row['codigo'],
            'observacion' => $row['observacion'],
        ];
    }

    return array_values($map);
}

function fetchCount(PDOStatement $stmt, int $embarqueId): int
{
    $stmt->execute([$embarqueId]);
    return (int)$stmt->fetchColumn();
}

function resolveFactura(PDO $pdo, array $pedido, array $pedidoDocs): string
{
    $pedidoId = (int)$pedido['id'];
    $factura = fetchPedidoFactura($pdo, $pedidoId);
    if ($factura !== null && $factura !== '') {
        return $factura;
    }
    $docs = $pedidoDocs[$pedidoId] ?? [];
    foreach ($docs as $doc) {
        $candidate = $doc['factura'] ?? $doc['doc_numero'] ?? $doc['label'] ?? '';
        $candidate = trim((string)$candidate);
        if ($candidate !== '') {
            return $candidate;
        }
    }
    return 'FACT-' . preg_replace('/[^A-Z0-9]+/', '', strtoupper($pedido['codigo']));
}

function fetchPedidoItems(PDO $pdo, int $pedidoId): array
{
    $sql = 'SELECT di.producto_id, di.expected_uv, di.expected_uc, pr.sku
              FROM so_pedido_dest_item di
              JOIN so_pedido_dest pd ON pd.id = di.pedido_dest_id
              JOIN para_productos pr ON pr.id = di.producto_id
             WHERE pd.pedido_id = ?
             ORDER BY di.id';
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$pedidoId]);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
    foreach ($rows as &$row) {
        $row['producto_id'] = (int)($row['producto_id'] ?? 0);
        $row['expected_uv'] = (int)($row['expected_uv'] ?? 0);
        $row['expected_uc'] = (int)($row['expected_uc'] ?? 0);
        $row['sku'] = trim((string)($row['sku'] ?? ''));
    }
    unset($row);
    return $rows;
}

function fetchPedidoFactura(PDO $pdo, int $pedidoId): ?string
{
    static $stmt = null;
    if ($stmt === null) {
        $stmt = $pdo->prepare('SELECT doc_numero FROM so_pedido_dest WHERE pedido_id = ? AND doc_numero IS NOT NULL AND doc_numero <> "" ORDER BY id LIMIT 1');
    }
    $stmt->execute([$pedidoId]);
    $val = $stmt->fetchColumn();
    return $val !== false ? (string)$val : null;
}

function convertUvToUnits(int $uv, ?float $unitsPerCase, ?float $casesPerPallet): float
{
    if ($uv <= 0) {
        return 0.0;
    }
    if ($casesPerPallet !== null && $casesPerPallet > 0 && $unitsPerCase !== null && $unitsPerCase > 0) {
        return $uv * $casesPerPallet * $unitsPerCase;
    }
    if ($unitsPerCase !== null && $unitsPerCase > 0) {
        return $uv * $unitsPerCase;
    }
    if ($casesPerPallet !== null && $casesPerPallet > 0) {
        return $uv * $casesPerPallet;
    }
    return (float)$uv;
}

function resolveUnitPrice(string $sku, array $priceMap): float
{
    $key = strtoupper(trim($sku));
    if ($key !== '' && isset($priceMap[$key])) {
        return $priceMap[$key];
    }
    return $priceMap['__DEFAULT__'];
}

function calculateMonto(array $items, array $productInfo, array $priceMap): float
{
    $total = 0.0;
    foreach ($items as $item) {
        $pid = $item['producto_id'];
        $info = $productInfo[$pid] ?? null;
        $unitsPerCase = $info['unidades_por_caja'] ?? null;
        $casesPerPallet = $info['cajas_por_pallet'] ?? null;
        $units = convertUvToUnits((int)$item['expected_uv'], $unitsPerCase, $casesPerPallet) + (int)$item['expected_uc'];
        if ($units <= 0) {
            continue;
        }
        $price = resolveUnitPrice($item['sku'] ?? '', $priceMap);
        $total += $units * $price;
    }
    if ($total <= 0) {
        return 0.0;
    }
    $factor = mt_rand(94, 108) / 100; // +/- ~6%
    $total *= $factor;
    return round($total, 2);
}

function resolveVueltaDate(PDO $pdo, int $embarqueId, array $emb): DateTimeImmutable
{
    try {
        $stmt = $pdo->prepare('SELECT MAX(hr_salida) FROM so_embarque_seguimiento_dest WHERE embarque_id = ?');
        $stmt->execute([$embarqueId]);
        $val = $stmt->fetchColumn();
        if ($val) {
            $dt = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', (string)$val);
            if ($dt instanceof DateTimeImmutable) {
                return $dt;
            }
            $parsed = strtotime((string)$val);
            if ($parsed !== false) {
                return (new DateTimeImmutable())->setTimestamp($parsed);
            }
        }
    } catch (Throwable $ignored) {
        // Tabla puede no existir aún, continuar con fallback
    }

    $candidates = [
        $emb['salida_at'] ?? null,
        $emb['updated_at'] ?? null,
        $emb['creado_at'] ?? null,
    ];
    foreach ($candidates as $raw) {
        $dt = parseDateTime($raw);
        if ($dt !== null) {
            return $dt;
        }
    }
    return new DateTimeImmutable('now');
}

function parseDateTime(?string $raw): ?DateTimeImmutable
{
    $raw = $raw !== null ? trim($raw) : '';
    if ($raw === '') {
        return null;
    }
    $patterns = ['Y-m-d H:i:s', 'Y-m-d H:i', DateTimeInterface::ATOM];
    foreach ($patterns as $pattern) {
        $dt = DateTimeImmutable::createFromFormat($pattern, $raw);
        if ($dt instanceof DateTimeImmutable) {
            return $dt;
        }
    }
    $timestamp = strtotime($raw);
    if ($timestamp === false) {
        return null;
    }
    return (new DateTimeImmutable())->setTimestamp($timestamp);
}

function aggregateFacturas(array $rows, string $fechaFactura, DateTimeImmutable $vueltaDate): array
{
    $aggregate = [];
    foreach ($rows as $row) {
        $factura = trim((string)$row['factura']);
        if ($factura === '') {
            continue;
        }
        if (!isset($aggregate[$factura])) {
            $aggregate[$factura] = 0.0;
        }
        $aggregate[$factura] += (float)$row['monto'];
    }
    $result = [];
    $offsetMinutes = 0;
    foreach ($aggregate as $factura => $monto) {
        if ($monto <= 0) {
            $monto = mt_rand(60000, 95000);
        }
        $offsetMinutes += mt_rand(10, 45);
        $fechaRendicion = $vueltaDate->add(new DateInterval('PT' . $offsetMinutes . 'M'));
        $result[] = [
            'factura' => $factura,
            'fecha_factura' => $fechaFactura,
            'fecha_rendicion' => $fechaRendicion->format('Y-m-d H:i:s'),
            'monto' => round($monto, 2),
        ];
    }
    return $result;
}
