<?php
/**
 * Genera registros simulados de vuelta (so_retorno) para embarques filtrados.
 * Usa la última hora de salida del seguimiento como base y aplica un offset
 * aleatorio para simular la hora de regreso al depósito.
 *
 * Uso típico:
 *   php scripts/populate_embarque_vuelta.php --embarque-like=EMB-202511-%
 *
 * Opciones:
 *   --embarque-like=PAT    Filtra códigos de embarque (default EMB-202511-%).
 *   --dry-run              Muestra las inserciones sin escribir en la base.
 *   --rewrite              Borra retornos existentes del embarque antes de insertar.
 *   --limit=N              Procesa a lo sumo N embarques.
 *   --seed=N               Semilla para el generador aleatorio (repetible).
 */

declare(strict_types=1);

$ROOT = dirname(__DIR__);

require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

$options = getopt('', [
    'embarque-like::',
    'dry-run',
    'rewrite',
    'limit::',
    'seed::',
]);

$embarqueLike = $options['embarque-like'] ?? 'EMB-202511-%';
$dryRun = array_key_exists('dry-run', $options);
$rewrite = array_key_exists('rewrite', $options);
$limit = isset($options['limit']) ? max(0, (int)$options['limit']) : 0;
$seed = isset($options['seed']) ? (int)$options['seed'] : null;

if ($seed !== null) {
    mt_srand($seed);
}

try {
    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec('SET NAMES utf8mb4');

    $embarques = fetchEmbarques($pdo, $embarqueLike, $limit);
    if (!$embarques) {
        echo "No se encontraron embarques para {$embarqueLike}.\n";
        exit(0);
    }

    $retCountStmt = $pdo->prepare('SELECT COUNT(*) FROM so_retorno WHERE embarque_id = ?');
    $deleteStmt = $pdo->prepare('DELETE FROM so_retorno WHERE embarque_id = ?');
    $insertStmt = $pdo->prepare(
        'INSERT INTO so_retorno (embarque_id, llegada_at, observacion, created_at)
         VALUES (:emb, :llegada, :obs, :created)'
    );

    $summary = [
        'processed' => 0,
        'skipped_existing' => 0,
        'rows_created' => 0,
        'rewritten' => 0,
    ];

    foreach ($embarques as $emb) {
        $embId = (int)$emb['id'];
        $codigo = (string)$emb['codigo'];
        $existing = fetchCount($retCountStmt, $embId);
        if ($existing > 0 && !$rewrite) {
            echo "Omitido {$codigo}: ya tiene registro de vuelta (use --rewrite para reemplazar).\n";
            $summary['skipped_existing']++;
            continue;
        }

        $llegada = resolveLlegadaVuelta($pdo, $emb);
        $obs = buildObservacion($llegada);

        if ($dryRun) {
            echo "[DRY-RUN] {$codigo}: llegada {$llegada->format('Y-m-d H:i:s')} | obs {$obs}\n";
            $summary['rows_created']++;
            continue;
        }

        $pdo->beginTransaction();
        try {
            if ($existing > 0) {
                $deleteStmt->execute([$embId]);
                $summary['rewritten']++;
            }
            $insertStmt->execute([
                ':emb' => $embId,
                ':llegada' => $llegada->format('Y-m-d H:i:s'),
                ':obs' => $obs,
                ':created' => $llegada->format('Y-m-d H:i:s'),
            ]);
            $pdo->commit();
            $summary['processed']++;
            $summary['rows_created']++;
            echo "{$codigo}: registrada hora de vuelta {$llegada->format('Y-m-d H:i:s')}.\n";
        } catch (Throwable $tx) {
            if ($pdo->inTransaction()) {
                $pdo->rollBack();
            }
            throw $tx;
        }
    }

    if ($dryRun) {
        echo "\nResumen (dry-run): embarques evaluados " . count($embarques) . ", retornos simulados {$summary['rows_created']}.\n";
    } else {
        echo "\nResumen: procesados {$summary['processed']}, omitidos {$summary['skipped_existing']}, filas nuevas {$summary['rows_created']}";
        if ($summary['rewritten'] > 0) {
            echo ", embarques reescritos {$summary['rewritten']}";
        }
        echo ".\n";
    }

    exit(0);
} catch (Throwable $e) {
    fwrite(STDERR, 'Error: ' . $e->getMessage() . "\n");
    exit(1);
}

function fetchEmbarques(PDO $pdo, string $embarqueLike, int $limit): array
{
    $sql = 'SELECT e.id, e.codigo, e.salida_at, e.updated_at, e.creado_at
              FROM so_embarque e
             WHERE e.codigo LIKE :emb
             ORDER BY e.codigo';
    if ($limit > 0) {
        $sql .= ' LIMIT ' . (int)$limit;
    }
    $stmt = $pdo->prepare($sql);
    $stmt->execute([':emb' => $embarqueLike]);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
    return $rows;
}

function fetchCount(PDOStatement $stmt, int $embarqueId): int
{
    $stmt->execute([$embarqueId]);
    return (int)$stmt->fetchColumn();
}

function resolveLlegadaVuelta(PDO $pdo, array $emb): DateTimeImmutable
{
    $embId = (int)$emb['id'];
    $base = resolveVueltaDate($pdo, $embId, $emb);
    $offset = mt_rand(30, 90); // minutos adicionales para retornar al depósito
    $llegada = $base->add(new DateInterval('PT' . $offset . 'M'));

    if (!empty($emb['salida_at'])) {
        $salida = parseDateTime((string)$emb['salida_at']);
        if ($salida instanceof DateTimeImmutable && $llegada <= $salida) {
            $fallbackOffset = mt_rand(120, 240);
            $llegada = $salida->add(new DateInterval('PT' . $fallbackOffset . 'M'));
        }
    }

    return $llegada;
}

function resolveVueltaDate(PDO $pdo, int $embarqueId, array $emb): DateTimeImmutable
{
    try {
        $stmt = $pdo->prepare('SELECT MAX(hr_salida) FROM so_embarque_seguimiento_dest WHERE embarque_id = ?');
        $stmt->execute([$embarqueId]);
        $val = $stmt->fetchColumn();
        if ($val) {
            $dt = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', (string)$val);
            if ($dt instanceof DateTimeImmutable) {
                return $dt;
            }
            $parsed = strtotime((string)$val);
            if ($parsed !== false) {
                return (new DateTimeImmutable())->setTimestamp($parsed);
            }
        }
    } catch (Throwable $ignored) {
        // Tabla puede no existir, continuar con fallback
    }

    $candidates = [
        $emb['salida_at'] ?? null,
        $emb['updated_at'] ?? null,
        $emb['creado_at'] ?? null,
    ];
    foreach ($candidates as $raw) {
        $dt = parseDateTime($raw);
        if ($dt !== null) {
            return $dt;
        }
    }

    return new DateTimeImmutable('now');
}

function parseDateTime($raw): ?DateTimeImmutable
{
    $raw = $raw !== null ? trim((string)$raw) : '';
    if ($raw === '') {
        return null;
    }
    $patterns = ['Y-m-d H:i:s', 'Y-m-d H:i', DateTimeInterface::ATOM];
    foreach ($patterns as $pattern) {
        $dt = DateTimeImmutable::createFromFormat($pattern, $raw);
        if ($dt instanceof DateTimeImmutable) {
            return $dt;
        }
    }
    $timestamp = strtotime($raw);
    if ($timestamp === false) {
        return null;
    }
    return (new DateTimeImmutable())->setTimestamp($timestamp);
}

function buildObservacion(DateTimeImmutable $llegada): string
{
    return 'Sim auto - llegada ' . $llegada->format('Y-m-d H:i');
}
