<?php
declare(strict_types=1);

use DateTimeImmutable;
use DateTimeInterface;

/**
 * Rellena so_embarque_seguimiento_dest con datos simulados para cada
 * destinatario de los embarques seleccionados.
 *
 * Uso:
 *   php scripts/populate_seguimiento_dest.php --like=EMB-202511-%
 *   php scripts/populate_seguimiento_dest.php --like=EMB-202511-% --dry-run --seed=123
 */

$ROOT = dirname(__DIR__);

$bufferActive = false;
if (PHP_SAPI === 'cli' && !ob_get_level()) {
    ob_start();
    $bufferActive = true;
}

require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';
require_once $ROOT . '/app/Support/ApiHelpers.php';

$options = getopt('', [
    'like::',
    'dry-run',
    'seed::',
    'limit::',
    'clear',
]);

$likePattern = $options['like'] ?? 'EMB-202511-%';
$dryRun = array_key_exists('dry-run', $options);
$seed = isset($options['seed']) ? (int)$options['seed'] : null;
$limit = isset($options['limit']) ? max(0, (int)$options['limit']) : 0;
$shouldClear = array_key_exists('clear', $options);

if ($seed !== null) {
    mt_srand($seed);
}

try {
    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec('SET NAMES utf8mb4');

    $embarques = fetchEmbarques($pdo, $likePattern);
    if (!$embarques) {
        echo "No se encontraron embarques para {$likePattern}.\n";
        exit(0);
    }

    if ($limit > 0 && count($embarques) > $limit) {
        $embarques = array_slice($embarques, 0, $limit);
    }

    if ($shouldClear) {
        clearSeguimiento($pdo, array_column($embarques, 'id'));
    }

    $upsertStmt = buildUpsertStatement($pdo);
    $processed = 0;
    $rowsAffected = 0;

    foreach ($embarques as $emb) {
        $paradas = fetchParadas($pdo, (int)$emb['id']);
        if (!$paradas) {
            echo "Embarque {$emb['codigo']}: sin paradas asociadas.\n";
            continue;
        }

        $kmActual = mt_rand(8000, 14000);
        $baseTime = resolveBaseTime($emb);
        $currentTime = $baseTime;

        foreach ($paradas as $index => $parada) {
            $travelMinutes = mt_rand(15, 45);
            $arrival = $currentTime->modify('+' . $travelMinutes . ' minutes');

            $totalMinutes = mt_rand(60, 120);
            $waitMinutes = mt_rand(5, 20);
            $serviceMinutes = mt_rand(20, 40);
            $controlMinutes = $totalMinutes - $waitMinutes - $serviceMinutes;
            if ($controlMinutes < 5) {
                $controlMinutes = 5;
            }
            $serviceMinutes = max(15, $serviceMinutes);
            $adjustedTotal = $waitMinutes + $serviceMinutes + $controlMinutes;
            if ($adjustedTotal < 60) {
                $controlMinutes += (60 - $adjustedTotal);
            } elseif ($adjustedTotal > 120) {
                $overflow = $adjustedTotal - 120;
                $reduce = min($overflow, $controlMinutes - 5);
                $controlMinutes -= $reduce;
                $overflow -= $reduce;
                if ($overflow > 0) {
                    $serviceMinutes = max(15, $serviceMinutes - $overflow);
                }
            }

            $start = $arrival->modify('+' . $waitMinutes . ' minutes');
            $end = $start->modify('+' . $serviceMinutes . ' minutes');
            $departure = $end->modify('+' . $controlMinutes . ' minutes');
            $totalUsed = (int)round(($departure->getTimestamp() - $arrival->getTimestamp()) / 60);

            $distance = mt_rand(5, 35);
            $kmInicial = $kmActual;
            $kmLlegada = $kmInicial + $distance;
            $kmActual = $kmLlegada;
            $currentTime = $departure;

            $payload = [
                'inicio_carga' => formatDateTime($start),
                'fin_carga' => formatDateTime($end),
                'km_inicial' => $kmInicial,
                'km_llegada' => $kmLlegada,
                'hr_llegada' => formatDateTime($arrival),
                'hr_inicio' => formatDateTime($start),
                'espera_descargar' => formatDuration($waitMinutes),
                'hr_termino' => formatDateTime($end),
                'tiempo_descarga' => formatDuration($serviceMinutes),
                'hr_salida' => formatDateTime($departure),
                'tiempo_control' => formatDuration($controlMinutes),
                'tiempo_total' => formatDuration($totalUsed),
                'tipo_carga' => 'Simulada',
                'salida_camara' => 'N/A',
            ];

            if ($dryRun) {
                echo sprintf(
                    "[DRY-RUN] Embarque %s parada %d (dest %d): llegada %s, salida %s, total %s, km %d-%d\n",
                    $emb['codigo'],
                    (int)$parada['orden'],
                    (int)$parada['destinatario_id'],
                    $payload['hr_llegada'],
                    $payload['hr_salida'],
                    $payload['tiempo_total'],
                    $kmInicial,
                    $kmLlegada
                );
                continue;
            }

            $params = array_merge([
                (int)$emb['id'],
                (int)$parada['destinatario_id'],
            ], array_values($payload));
            $upsertStmt->execute($params);
            $rowsAffected += $upsertStmt->rowCount() > 0 ? 1 : 0;
        }

        $processed++;
    }

    $scope = $limit > 0 ? "{$processed} de {$limit}" : (string)$processed;

    if ($dryRun) {
        echo "\nDry-run completado para {$scope} embarques.\n";
    } else {
        echo "\nSeguimiento actualizado para {$scope} embarques. Filas afectadas: {$rowsAffected}.\n";
    }

    if ($bufferActive) {
        ob_end_flush();
    }

    exit(0);
} catch (Throwable $e) {
    if ($bufferActive && ob_get_level()) {
        ob_end_flush();
    }
    fwrite(STDERR, 'Error: ' . $e->getMessage() . "\n");
    exit(1);
}

function fetchEmbarques(PDO $pdo, string $likePattern): array
{
    $sql = "SELECT e.id, e.codigo, e.salida_at, e.carga_fin_at, e.carga_inicio_at, e.llegada_at, e.creado_at,
                   MIN(p.fecha_pedido) AS fecha_pedido
            FROM so_embarque e
            JOIN so_embarque_pre ep ON ep.embarque_id = e.id
            JOIN so_preembarque pre ON pre.id = ep.preembarque_id
            JOIN so_pedido p ON p.id = pre.pedido_id
            WHERE e.codigo LIKE :like
            GROUP BY e.id, e.codigo, e.salida_at, e.carga_fin_at, e.carga_inicio_at, e.llegada_at, e.creado_at
            ORDER BY e.codigo";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([':like' => $likePattern]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
}

function fetchParadas(PDO $pdo, int $embarqueId): array
{
    $stmt = $pdo->prepare('SELECT id, destinatario_id, orden FROM so_embarque_parada WHERE embarque_id = ? ORDER BY orden');
    $stmt->execute([$embarqueId]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
}

function clearSeguimiento(PDO $pdo, array $embarqueIds): void
{
    $embarqueIds = array_values(array_unique(array_map('intval', $embarqueIds)));
    if (!$embarqueIds) {
        return;
    }

    $chunkSize = 200;
    $stmt = null;
    foreach (array_chunk($embarqueIds, $chunkSize) as $chunk) {
        $placeholders = implode(',', array_fill(0, count($chunk), '?'));
        $sql = 'DELETE FROM so_embarque_seguimiento_dest WHERE embarque_id IN (' . $placeholders . ')';
        $stmt = $pdo->prepare($sql);
        foreach ($chunk as $index => $id) {
            $stmt->bindValue($index + 1, $id, PDO::PARAM_INT);
        }
        $stmt->execute();
    }
}

function resolveBaseTime(array $emb): DateTimeImmutable
{
    $candidates = [
        $emb['salida_at'] ?? null,
        $emb['carga_fin_at'] ?? null,
        $emb['carga_inicio_at'] ?? null,
        $emb['llegada_at'] ?? null,
        $emb['creado_at'] ?? null,
    ];
    foreach ($candidates as $raw) {
        $dt = parseDateTime($raw);
        if ($dt !== null) {
            return $dt;
        }
    }
    $fecha = $emb['fecha_pedido'] ?? date('Y-m-d');
    $dt = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', $fecha . ' 07:00:00');
    return $dt ?: new DateTimeImmutable('now');
}

function parseDateTime(?string $raw): ?DateTimeImmutable
{
    $raw = $raw !== null ? trim($raw) : '';
    if ($raw === '') {
        return null;
    }
    $patterns = ['Y-m-d H:i:s', 'Y-m-d H:i', DateTimeInterface::ATOM];
    foreach ($patterns as $pattern) {
        $dt = DateTimeImmutable::createFromFormat($pattern, $raw);
        if ($dt instanceof DateTimeImmutable) {
            return $dt;
        }
    }
    $timestamp = strtotime($raw);
    if ($timestamp === false) {
        return null;
    }
    return (new DateTimeImmutable())->setTimestamp($timestamp);
}

function formatDateTime(DateTimeImmutable $dt): string
{
    return $dt->format('Y-m-d H:i');
}

function formatDuration(int $minutes): string
{
    $minutes = max(0, $minutes);
    $hours = intdiv($minutes, 60);
    $mins = $minutes % 60;
    return sprintf('%02d:%02d', $hours, $mins);
}

function buildUpsertStatement(PDO $pdo): PDOStatement
{
    $columns = [
        'inicio_carga', 'fin_carga', 'km_inicial', 'km_llegada',
        'hr_llegada', 'hr_inicio', 'espera_descargar', 'hr_termino',
        'tiempo_descarga', 'hr_salida', 'tiempo_control', 'tiempo_total',
        'tipo_carga', 'salida_camara',
    ];
    $placeholders = implode(',', array_fill(0, count($columns), '?'));
    $updates = implode(',', array_map(static function (string $col): string {
        return $col . '=VALUES(' . $col . ')';
    }, $columns));

    $sql = 'INSERT INTO so_embarque_seguimiento_dest '
         . '(embarque_id, destinatario_id, ' . implode(',', $columns) . ') '
         . 'VALUES (?, ?, ' . $placeholders . ') '
         . 'ON DUPLICATE KEY UPDATE ' . $updates;

    return $pdo->prepare($sql);
}
