<?php
declare(strict_types=1);

/**
 * Avanza embarques a estado ENTREGADO llenando tiempos básicos.
 *
 * Uso:
 *   php scripts/progress_embarques.php --like=EMB-202511-%
 *   php scripts/progress_embarques.php --like=EMB-202511-00% --dry-run
 */

$ROOT = dirname(__DIR__);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

$options = getopt('', [
    'like::',
    'dry-run',
]);

$likePattern = $options['like'] ?? 'EMB-202511-%';
$dryRun = array_key_exists('dry-run', $options);

$pdo = get_pdo();
$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
$pdo->exec('SET NAMES utf8mb4');

$estadoEntregado = fetchId($pdo, 'SELECT id FROM so_embarque_estado WHERE code=? LIMIT 1', ['ENTREGADO']);
if ($estadoEntregado <= 0) {
    fwrite(STDERR, "No se encontró el estado ENTREGADO en so_embarque_estado.\n");
    exit(1);
}

$rows = fetchEmbarques($pdo, $likePattern);
if (!$rows) {
    echo "No hay embarques que coincidan con {$likePattern}.\n";
    exit(0);
}

$paradaUpdater = new ParadaUpdater($pdo);
$progress = [];

foreach ($rows as $row) {
    $timeline = buildTimeline($row['fecha_pedido'], (int)$row['ordinal']);
    $progress[] = [
        'codigo' => $row['codigo'],
        'fecha' => $row['fecha_pedido'],
        'salida' => $timeline['salida_at'],
    ];

    if ($dryRun) {
        continue;
    }

    $upd = $pdo->prepare('UPDATE so_embarque
        SET estado_id = :estado,
            llegada_at = :llegada,
            carga_fecha = :carga_fecha,
            carga_inicio_at = :carga_inicio,
            carga_fin_at = :carga_fin,
            salida_at = :salida,
            observacion = :obs
        WHERE id = :id');
    $upd->execute([
        ':estado' => $estadoEntregado,
        ':llegada' => $timeline['llegada_at'],
        ':carga_fecha' => $timeline['carga_fecha'],
        ':carga_inicio' => $timeline['carga_inicio_at'],
        ':carga_fin' => $timeline['carga_fin_at'],
        ':salida' => $timeline['salida_at'],
        ':obs' => 'Auto entregado por script',
        ':id' => (int)$row['id'],
    ]);

    $paradaUpdater->updateByEmbarque((int)$row['id'], $timeline['parada_base']);
}

foreach ($progress as $info) {
    echo sprintf(
        "Embarque %s (%s) -> salida %s\n",
        $info['codigo'],
        $info['fecha'],
        $info['salida']
    );
}

if ($dryRun) {
    echo "Dry-run: sin cambios aplicados.\n";
}

final class ParadaUpdater
{
    private PDO $pdo;

    public function __construct(PDO $pdo)
    {
        $this->pdo = $pdo;
    }

    public function updateByEmbarque(int $embarqueId, DateTimeImmutable $base): void
    {
        $stmt = $this->pdo->prepare('SELECT id, orden FROM so_embarque_parada WHERE embarque_id = ? ORDER BY orden');
        $stmt->execute([$embarqueId]);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        if (!$rows) {
            return;
        }

        $upd = $this->pdo->prepare('UPDATE so_embarque_parada
            SET hora_llegada = :llegada,
                hora_inicio_descarga = :inicio,
                hora_fin_descarga = :fin,
                hora_salida = :salida
            WHERE id = :id');

        foreach ($rows as $index => $row) {
            $offset = $index * 45;
            $llegada = $base->modify('+' . ($offset) . ' minutes');
            $inicio = $llegada->modify('+5 minutes');
            $fin = $inicio->modify('+20 minutes');
            $salida = $fin->modify('+5 minutes');

            $upd->execute([
                ':llegada' => $llegada->format('Y-m-d H:i:s'),
                ':inicio' => $inicio->format('Y-m-d H:i:s'),
                ':fin' => $fin->format('Y-m-d H:i:s'),
                ':salida' => $salida->format('Y-m-d H:i:s'),
                ':id' => (int)$row['id'],
            ]);
        }
    }
}

function fetchEmbarques(PDO $pdo, string $likePattern): array
{
    $stmt = $pdo->prepare(
        "SELECT e.id, e.codigo, MIN(p.fecha_pedido) AS fecha_pedido,
                ROW_NUMBER() OVER (ORDER BY MIN(p.fecha_pedido), e.id) AS ordinal
         FROM so_embarque e
         JOIN so_embarque_pre ep ON ep.embarque_id = e.id
         JOIN so_preembarque pre ON pre.id = ep.preembarque_id
         JOIN so_pedido p ON p.id = pre.pedido_id
         WHERE e.codigo LIKE :like
         GROUP BY e.id, e.codigo
         ORDER BY fecha_pedido, e.id"
    );
    $stmt->execute([':like' => $likePattern]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
}

function buildTimeline(string $fechaPedido, int $ordinal): array
{
    $base = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', $fechaPedido . ' 06:30:00');
    if (!$base) {
        $base = new DateTimeImmutable('now');
    }
    $offset = ($ordinal - 1) * 10;
    $llegada = $base->modify('+' . $offset . ' minutes');
    $cargaInicio = $llegada->modify('+30 minutes');
    $cargaFin = $cargaInicio->modify('+40 minutes');
    $salida = $cargaFin->modify('+15 minutes');

    return [
        'carga_fecha' => $base->format('Y-m-d'),
        'llegada_at' => $llegada->format('Y-m-d H:i:s'),
        'carga_inicio_at' => $cargaInicio->format('Y-m-d H:i:s'),
        'carga_fin_at' => $cargaFin->format('Y-m-d H:i:s'),
        'salida_at' => $salida->format('Y-m-d H:i:s'),
        'parada_base' => $salida->modify('+30 minutes'),
    ];
}

function fetchId(PDO $pdo, string $sql, array $params): int
{
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    return (int)$stmt->fetchColumn();
}
