<?php

declare(strict_types=1);

$root = dirname(__DIR__);
require_once $root . '/config/db.php';

$options = getopt('', [
    'dry-run',
    'backup::',
    'no-backup',
    'help',
]);

if (isset($options['help'])) {
    fwrite(STDOUT, <<<TXT
Rebuild wh_stock from wh_pallet/wh_pallet_item

Usage:
  php scripts/rebuild_wh_stock.php [--dry-run] [--backup[=NAME]] [--no-backup]

Options:
  --dry-run        Ejecuta el cálculo y muestra las acciones sin escribir en la base.
  --backup[=NAME]  Genera tabla copia de wh_stock. Se genera nombre automático si no se indica.
  --no-backup      Omite la copia de respaldo.
  --help           Muestra esta ayuda.
TXT);
    exit(0);
}

$dryRun = array_key_exists('dry-run', $options);
$wantBackup = !array_key_exists('no-backup', $options);
$backupName = null;

if ($wantBackup) {
    if (array_key_exists('backup', $options)) {
        $backupName = $options['backup'];
        if ($backupName === false || $backupName === null || $backupName === '') {
            $backupName = 'backup_wh_stock_' . date('Ymd_His');
        }
    } else {
        $backupName = 'backup_wh_stock_' . date('Ymd_His');
    }

    $backupName = (string) $backupName;
    $backupName = preg_replace('/[^A-Za-z0-9_]/', '_', $backupName);
    if ($backupName === '' || $backupName === null) {
        throw new RuntimeException('Nombre de backup inválido');
    }
}

try {
    $pdo = getPDO();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec('SET SESSION sql_mode = REPLACE(@@sql_mode, "ONLY_FULL_GROUP_BY", "")');

    fwrite(STDOUT, ($dryRun ? '[dry-run] ' : '') . "Reconstruyendo wh_stock...\n");

    if ($wantBackup) {
        fwrite(STDOUT, "Creando backup en tabla {$backupName}...\n");
        if (!$dryRun) {
            $pdo->exec('CREATE TABLE `' . $backupName . '` LIKE wh_stock');
            $pdo->exec('INSERT INTO `' . $backupName . '` SELECT * FROM wh_stock');
        }
    }

    fwrite(STDOUT, "Generando dataset temporal...\n");
    $pdo->exec('DROP TEMPORARY TABLE IF EXISTS tmp_wh_stock_rebuild');
    $pdo->exec(
        'CREATE TEMPORARY TABLE tmp_wh_stock_rebuild AS
        SELECT
            p.deposito_id,
            p.posicion_id,
            pi.producto_id,
            COALESCE(pi.lote_id, 0) AS lote_id,
            pi.pallet_id,
            SUM(pi.uv_cajas) AS qty_uv,
            SUM(pi.uc_unidades) AS qty_uc,
            MAX(CASE WHEN pi.uc_unidades > 0 THEN 1 ELSE 0 END) AS pickeado
        FROM wh_pallet p
        JOIN wh_pallet_item pi ON pi.pallet_id = p.id
        WHERE p.deleted_at IS NULL
        GROUP BY
            p.deposito_id,
            p.posicion_id,
            pi.producto_id,
            COALESCE(pi.lote_id, 0),
            pi.pallet_id'
    );

    $totalRows = (int) $pdo->query('SELECT COUNT(*) FROM tmp_wh_stock_rebuild')->fetchColumn();
    fwrite(STDOUT, "Registros calculados: {$totalRows}\n");

    $oldPositions = $pdo->query('SELECT DISTINCT posicion_id FROM wh_stock WHERE posicion_id IS NOT NULL')->fetchAll(PDO::FETCH_COLUMN);
    $newPositions = $pdo->query('SELECT DISTINCT posicion_id FROM tmp_wh_stock_rebuild WHERE posicion_id IS NOT NULL')->fetchAll(PDO::FETCH_COLUMN);
    $positions = array_unique(array_merge($oldPositions ?: [], $newPositions ?: []));

    if ($dryRun) {
        fwrite(STDOUT, "[dry-run] Se omite escritura en wh_stock.\n");
    } else {
        fwrite(STDOUT, "Aplicando reemplazo de wh_stock...\n");
        $pdo->beginTransaction();
        $pdo->exec('DELETE FROM wh_stock');
        $insertSql = 'INSERT INTO wh_stock (deposito_id, posicion_id, producto_id, lote_id, pallet_id, qty_uv, qty_uc, pickeado)
                      SELECT deposito_id, posicion_id, producto_id, lote_id, pallet_id, qty_uv, qty_uc, pickeado
                      FROM tmp_wh_stock_rebuild
                      WHERE qty_uv <> 0 OR qty_uc <> 0';
        $pdo->exec($insertSql);
        $pdo->commit();
    }

    $countAfter = (int) $pdo->query('SELECT COUNT(*) FROM wh_stock')->fetchColumn();
    fwrite(STDOUT, "Registros finales en wh_stock: {$countAfter}\n");

    if ($positions) {
        fwrite(STDOUT, ($dryRun ? '[dry-run] ' : '') . 'Actualizando flags de posiciones...' . "\n");
        if (!$dryRun) {
            $stmtRefresh = $pdo->prepare('CALL sp_wh_pos_refresh_flags(:pos_id)');
            foreach ($positions as $posId) {
                if ($posId === null) {
                    continue;
                }
                $stmtRefresh->execute([':pos_id' => (int) $posId]);
            }
        }
        fwrite(STDOUT, "Posiciones procesadas: " . count($positions) . "\n");
    } else {
        fwrite(STDOUT, "No hubo posiciones para recalcular.\n");
    }

    $checkSql = 'SELECT s.pallet_id AS pallet_id, s.posicion_id, SUM(pi.uv_cajas) AS uv_items, SUM(pi.uc_unidades) AS uc_items,
               SUM(s.qty_uv) AS uv_stock, SUM(s.qty_uc) AS uc_stock
           FROM wh_stock s
           JOIN wh_pallet_item pi ON pi.pallet_id = s.pallet_id
           GROUP BY s.pallet_id, s.posicion_id
           HAVING SUM(s.qty_uv) < SUM(pi.uv_cajas) OR SUM(s.qty_uc) < SUM(pi.uc_unidades)
           LIMIT 10';
    $discrepancies = $pdo->query($checkSql)->fetchAll(PDO::FETCH_ASSOC);
    if ($discrepancies) {
        fwrite(STDOUT, "ATENCION: aún existen discrepancias:\n");
        foreach ($discrepancies as $row) {
            fwrite(STDOUT, sprintf(
                "  pallet %d en pos %s => items UV=%d / stock UV=%d, items UC=%d / stock UC=%d\n",
                $row['pallet_id'],
                $row['posicion_id'] ?? 'NULL',
                $row['uv_items'],
                $row['uv_stock'],
                $row['uc_items'],
                $row['uc_stock']
            ));
        }
    } else {
        fwrite(STDOUT, "Validación OK: no se detectaron diferencias entre pallets y wh_stock.\n");
    }

    fwrite(STDOUT, "Proceso completado.\n");
} catch (Throwable $e) {
    fwrite(STDERR, 'ERROR: ' . $e->getMessage() . "\n");
    exit(1);
}
