<?php
declare(strict_types=1);

/**
 * Genera registros de devoluciones para embarques simulados tomando como base
 * la observación de los pedidos ("Dev planificada X") o, en su defecto, la
 * hoja CSV de pruebas.
 *
 * Uso típico:
 *   php scripts/record_embarque_devoluciones.php --embarque-like=EMB-202511-%
 *
 * Opciones útiles:
 *   --embarque-like=PAT    Filtra códigos de embarque (default EMB-202511-%)
 *   --pedido-like=PAT      Filtra códigos de pedido asociados (default SO-AUTO-%)
 *   --order-prefix=STR     Prefijo esperado en el CSV para mapear pedidos (default SO-AUTO)
 *   --offset=N             Offset utilizado al importar CSV (default 0)
 *   --csv=PATH             Ruta al CSV con devoluciones planificadas (default tests/data/salidas_noviembre.csv)
 *   --motivo="Motivo 1"    Motivo asignado a cada devolución ("", Motivo 1/2/3)
 *   --dry-run              Muestra lo que haría sin tocar la base
 *   --rewrite              Borra devoluciones existentes del embarque antes de insertar
 *   --limit=N              Procesa a lo sumo N embarques
 */

$ROOT = dirname(__DIR__);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';
require_once $ROOT . '/app/Support/ApiHelpers.php';
require_once $ROOT . '/app/Support/InventarioReport.php';

$options = getopt('', [
    'embarque-like::',
    'pedido-like::',
    'order-prefix::',
    'offset::',
    'motivo::',
    'dry-run',
    'rewrite',
    'limit::',
    'csv::',
]);

$embarqueLike = $options['embarque-like'] ?? 'EMB-202511-%';
$pedidoLike = $options['pedido-like'] ?? 'SO-AUTO-%';
$orderPrefix = $options['order-prefix'] ?? 'SO-AUTO';
$offset = isset($options['offset']) ? max(0, (int)$options['offset']) : 0;
$motivo = isset($options['motivo']) ? trim((string)$options['motivo']) : 'Motivo 1';
$dryRun = array_key_exists('dry-run', $options);
$rewrite = array_key_exists('rewrite', $options);
$limit = isset($options['limit']) ? max(0, (int)$options['limit']) : 0;
$csvPath = $options['csv'] ?? ($ROOT . '/tests/data/salidas_noviembre.csv');

$allowedMotivos = ['', 'Motivo 1', 'Motivo 2', 'Motivo 3'];
if (!in_array($motivo, $allowedMotivos, true)) {
    fwrite(STDERR, "Motivo inválido. Valores admitidos: \"\", Motivo 1, Motivo 2, Motivo 3.\n");
    exit(1);
}

$csvPlanMap = [];
if ($csvPath && is_file($csvPath)) {
    try {
        $csvPlanMap = buildCsvPlanMap($csvPath, $orderPrefix, $offset);
    } catch (Throwable $e) {
        fwrite(STDERR, "Advertencia: no se pudo leer CSV de devoluciones: {$e->getMessage()}\n");
    }
}

try {
    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec('SET NAMES utf8mb4');

    $hasFacturaColumn = columnExists($pdo, 'so_embarque_devoluciones', 'factura');
    $skuColumnName = inventario_detect_column($pdo, 'para_productos', ['sku', 'codigo', 'cod', 'sku_cliente']);
    $skuSelectExpr = $skuColumnName ? ('pr.`' . $skuColumnName . '`') : 'CAST(pr.id AS CHAR)';

    $embarques = fetchEmbarquePedidos($pdo, $embarqueLike, $pedidoLike);
    if (!$embarques) {
        echo "No se encontraron embarques para el patrón {$embarqueLike}.\n";
        exit(0);
    }

    if ($limit > 0 && count($embarques) > $limit) {
        $embarques = array_slice($embarques, 0, $limit);
    }

    $countStmt = $pdo->prepare('SELECT COUNT(*) FROM so_embarque_devoluciones WHERE embarque_id = ?');
    $deleteStmt = $pdo->prepare('DELETE FROM so_embarque_devoluciones WHERE embarque_id = ?');

    if ($hasFacturaColumn) {
        $insertStmt = $pdo->prepare(
            'INSERT INTO so_embarque_devoluciones (embarque_id, factura, sku, pallets, cajas_sueltas, unidades_sueltas, unidades_danadas, motivo, lote)
             VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)'
        );
    } else {
        $insertStmt = $pdo->prepare(
            'INSERT INTO so_embarque_devoluciones (embarque_id, sku, pallets, cajas_sueltas, unidades_sueltas, unidades_danadas, motivo, lote)
             VALUES (?, ?, ?, ?, ?, ?, ?, ?)'
        );
    }

    $summary = [
        'processed' => 0,
        'skipped_existing' => 0,
        'rows_created' => 0,
        'rows_preview' => 0,
        'rewritten' => 0,
    ];

    foreach ($embarques as $embRow) {
        $embarqueId = (int)$embRow['id'];
        $embarqueCode = (string)$embRow['codigo'];
        $existing = fetchCount($countStmt, $embarqueId);

        if ($existing > 0 && !$rewrite) {
            echo "Omitido {$embarqueCode}: ya tiene devoluciones (use --rewrite para reemplazar).\n";
            $summary['skipped_existing']++;
            continue;
        }

        $rowsToInsert = [];
        $totalUnitsForEmbarque = 0;
        $planSummary = [];

        foreach ($embRow['pedidos'] as $pedido) {
            $pedidoId = (int)$pedido['id'];
            $pedidoCode = (string)$pedido['codigo'];
            $planUnits = $csvPlanMap[$pedidoCode] ?? extractPlanFromObservation($pedido['observacion']);
            if ($planUnits <= 0) {
                continue;
            }

            $items = fetchPedidoItems($pdo, $pedidoId, $skuSelectExpr);
            if (!$items) {
                echo "Aviso {$pedidoCode}: sin ítems para calcular devoluciones.\n";
                continue;
            }

            $productInfo = inventario_fetch_product_info($pdo, array_column($items, 'producto_id'));
            $factura = fetchPedidoFactura($pdo, $pedidoId);
            $pedidoRows = buildDevolucionRows($items, $productInfo, $planUnits, $motivo, $factura, $pedidoCode);
            if (!$pedidoRows) {
                echo "Aviso {$pedidoCode}: no se pudo distribuir la devolución planificada ({$planUnits}).\n";
                continue;
            }

            $allocatedUnits = 0;
            foreach ($pedidoRows as &$r) {
                $r['pedido_code'] = $pedidoCode; // sólo para salida en consola
                $allocatedUnits += (int)$r['total_unidades'];
            }
            unset($r);

            $planSummary[] = [$pedidoCode, $planUnits, $allocatedUnits];
            $totalUnitsForEmbarque += $allocatedUnits;
            $rowsToInsert = array_merge($rowsToInsert, $pedidoRows);
        }

        if ($rowsToInsert) {
            foreach ($rowsToInsert as &$row) {
                unset($row['pedido_code']);
            }
            unset($row);
            $rowsToInsert = mergeDuplicateRows($rowsToInsert);
            $totalUnitsForEmbarque = array_sum(array_map(static function (array $r): int {
                return (int)($r['total_unidades'] ?? 0);
            }, $rowsToInsert));
        }

        if (!$rowsToInsert) {
            echo "Embarque {$embarqueCode}: sin devoluciones planificadas.\n";
            continue;
        }

        if ($dryRun) {
            echo "[DRY-RUN] Embarque {$embarqueCode}: " . count($rowsToInsert) . " fila(s) -> {$totalUnitsForEmbarque} unidad(es).\n";
            foreach ($planSummary as [$pCode, $planUnits, $allocUnits]) {
                echo "  Pedido {$pCode}: plan {$planUnits}, asignado {$allocUnits}.\n";
            }
            foreach ($rowsToInsert as $r) {
                echo sprintf(
                    "  - %s | pallets=%d cajas=%d unidades=%d motivo=%s lote=%s\n",
                    $r['sku'],
                    $r['pallets'],
                    $r['cajas_sueltas'],
                    $r['unidades_sueltas'],
                    $r['motivo'] === '' ? 'N/A' : $r['motivo'],
                    $r['lote']
                );
            }
            $summary['rows_preview'] += count($rowsToInsert);
            continue;
        }

        $pdo->beginTransaction();
        try {
            if ($existing > 0) {
                $deleteStmt->execute([$embarqueId]);
                $summary['rewritten']++;
            }

            foreach ($rowsToInsert as $r) {
                $sku = strtoupper(substr((string)$r['sku'], 0, 100));
                if ($sku === '') {
                    continue;
                }
                $facturaVal = $r['factura'] !== null ? substr((string)$r['factura'], 0, 100) : null;
                $loteVal = substr($r['lote'], 0, 100);

                if ($hasFacturaColumn) {
                    $insertStmt->execute([
                        $embarqueId,
                        $facturaVal,
                        $sku,
                        $r['pallets'],
                        $r['cajas_sueltas'],
                        $r['unidades_sueltas'],
                        $r['unidades_danadas'],
                        $r['motivo'],
                        $loteVal,
                    ]);
                } else {
                    $insertStmt->execute([
                        $embarqueId,
                        $sku,
                        $r['pallets'],
                        $r['cajas_sueltas'],
                        $r['unidades_sueltas'],
                        $r['unidades_danadas'],
                        $r['motivo'],
                        $loteVal,
                    ]);
                }
            }

            $pdo->commit();
            $summary['rows_created'] += count($rowsToInsert);
            $summary['processed']++;
            echo "Embarque {$embarqueCode}: registradas " . count($rowsToInsert) . " devoluciones ({$totalUnitsForEmbarque} unidad(es)).\n";
        } catch (Throwable $tx) {
            if ($pdo->inTransaction()) {
                $pdo->rollBack();
            }
            throw $tx;
        }
    }

    if ($dryRun) {
        echo "\nResumen (dry-run): embarques evaluados " . count($embarques) . ", filas simuladas {$summary['rows_preview']}.\n";
    } else {
        echo "\nResumen: procesados {$summary['processed']}, omitidos {$summary['skipped_existing']}, filas nuevas {$summary['rows_created']}";
        if ($summary['rewritten'] > 0) {
            echo ", embarques reescritos {$summary['rewritten']}";
        }
        echo ".\n";
    }

    exit(0);
} catch (Throwable $e) {
    fwrite(STDERR, 'Error: ' . $e->getMessage() . "\n");
    exit(1);
}

function buildCsvPlanMap(string $path, string $orderPrefix, int $offset): array
{
    $fp = fopen($path, 'r');
    if ($fp === false) {
        throw new RuntimeException('No se pudo abrir el CSV: ' . $path);
    }

    $headers = fgetcsv($fp);
    if ($headers === false) {
        fclose($fp);
        throw new RuntimeException('CSV vacío: ' . $path);
    }

    $index = 0;
    $planMap = [];
    while (($row = fgetcsv($fp)) !== false) {
        if (!$row) {
            continue;
        }
        $assoc = [];
        foreach ($headers as $idx => $key) {
            $assoc[$key] = $row[$idx] ?? null;
        }
        $index++;
        $code = sprintf('%s-%04d', $orderPrefix, $offset + $index);
        $planMap[$code] = (int)($assoc['devolucion_planificada'] ?? 0);
    }

    fclose($fp);
    return $planMap;
}

function fetchEmbarquePedidos(PDO $pdo, string $embarqueLike, string $pedidoLike): array
{
    $stmt = $pdo->prepare(
        'SELECT e.id AS embarque_id, e.codigo AS embarque_codigo,
                ped.id AS pedido_id, ped.codigo AS pedido_codigo, ped.observacion AS pedido_observacion
           FROM so_embarque e
           JOIN so_embarque_pre ep ON ep.embarque_id = e.id
           JOIN so_preembarque pre ON pre.id = ep.preembarque_id
           JOIN so_pedido ped ON ped.id = pre.pedido_id
          WHERE e.codigo LIKE :emb AND ped.codigo LIKE :ped
          ORDER BY e.codigo, ped.codigo, ped.id'
    );
    $stmt->execute([
        ':emb' => $embarqueLike,
        ':ped' => $pedidoLike,
    ]);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
    if (!$rows) {
        return [];
    }

    $grouped = [];
    foreach ($rows as $row) {
        $embId = (int)$row['embarque_id'];
        if (!isset($grouped[$embId])) {
            $grouped[$embId] = [
                'id' => $embId,
                'codigo' => (string)$row['embarque_codigo'],
                'pedidos' => [],
            ];
        }
        $grouped[$embId]['pedidos'][] = [
            'id' => (int)$row['pedido_id'],
            'codigo' => (string)$row['pedido_codigo'],
            'observacion' => $row['pedido_observacion'],
        ];
    }

    return array_values($grouped);
}

function fetchPedidoItems(PDO $pdo, int $pedidoId, string $skuSelectExpr): array
{
    $sql = 'SELECT di.producto_id, di.expected_uv, di.expected_uc, ' . $skuSelectExpr . ' AS sku
              FROM so_pedido_dest_item di
              JOIN so_pedido_dest pd ON pd.id = di.pedido_dest_id
              JOIN para_productos pr ON pr.id = di.producto_id
             WHERE pd.pedido_id = ?
             ORDER BY di.id';
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$pedidoId]);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
    foreach ($rows as &$row) {
        $row['producto_id'] = (int)($row['producto_id'] ?? 0);
        $row['expected_uv'] = (int)($row['expected_uv'] ?? 0);
        $row['expected_uc'] = (int)($row['expected_uc'] ?? 0);
        $row['sku'] = trim((string)($row['sku'] ?? ''));
    }
    unset($row);
    return $rows;
}

function fetchPedidoFactura(PDO $pdo, int $pedidoId): ?string
{
    static $stmt = null;
    if ($stmt === null) {
        $stmt = $pdo->prepare('SELECT doc_numero FROM so_pedido_dest WHERE pedido_id = ? AND doc_numero IS NOT NULL AND doc_numero <> "" ORDER BY id LIMIT 1');
    }
    $stmt->execute([$pedidoId]);
    $val = $stmt->fetchColumn();
    return $val !== false ? (string)$val : null;
}

function fetchCount(PDOStatement $stmt, int $embarqueId): int
{
    $stmt->execute([$embarqueId]);
    return (int)$stmt->fetchColumn();
}

function extractPlanFromObservation($observacion): int
{
    $obs = trim((string)$observacion);
    if ($obs === '') {
        return 0;
    }
    if (preg_match('/Dev\s+planificada\s+(\d+)/i', $obs, $m)) {
        return max(0, (int)$m[1]);
    }
    return 0;
}

function buildDevolucionRows(array $items, array $productInfo, int $planUnits, string $motivo, ?string $factura, string $pedidoCode): array
{
    $unitsPerItem = [];
    $totalUnits = 0.0;

    foreach ($items as $idx => $item) {
        $pid = (int)$item['producto_id'];
        $info = $productInfo[$pid] ?? null;
        $unitsPerCase = $info['unidades_por_caja'] ?? null;
        $casesPerPallet = $info['cajas_por_pallet'] ?? null;
        $unitsFromUv = convertUvToUnits($item['expected_uv'], $unitsPerCase, $casesPerPallet);
        $unitsFromUc = (int)$item['expected_uc'];
        $total = $unitsFromUv + $unitsFromUc;
        if ($total <= 0) {
            continue;
        }
        $unitsPerItem[] = [
            'index' => $idx,
            'sku' => $item['sku'],
            'producto_id' => $pid,
            'units_total' => $total,
            'units_per_case' => $unitsPerCase,
            'cases_per_pallet' => $casesPerPallet,
        ];
        $totalUnits += $total;
    }

    if (!$unitsPerItem || $totalUnits <= 0) {
        return [];
    }

    $planUnits = min($planUnits, (int)round($totalUnits));
    if ($planUnits <= 0) {
        return [];
    }

    $allocations = allocateUnits($unitsPerItem, $planUnits);
    if (!$allocations) {
        return [];
    }

    $rows = [];
    $lote = buildLoteCode($pedidoCode);

    foreach ($unitsPerItem as $meta) {
        $allocated = $allocations[$meta['index']] ?? 0;
        if ($allocated <= 0) {
            continue;
        }
        $breakdown = inventario_breakdown_units((float)$allocated, $meta['cases_per_pallet'], $meta['units_per_case']);
        $pallets = (int)max(0, round((float)($breakdown['pallets'] ?? 0)));
        $cajas = (int)max(0, round((float)($breakdown['cajas'] ?? 0)));
        $sueltas = (int)max(0, round((float)($breakdown['unidades_sueltas'] ?? $allocated)));
        $total = (int)round((float)($breakdown['total_unidades'] ?? $allocated));

        $rows[] = [
            'sku' => $meta['sku'] !== '' ? $meta['sku'] : (string)$meta['producto_id'],
            'pallets' => $pallets,
            'cajas_sueltas' => $cajas,
            'unidades_sueltas' => $sueltas,
            'unidades_danadas' => 0,
            'motivo' => $motivo,
            'lote' => $lote,
            'factura' => $factura,
            'total_unidades' => $total,
        ];
    }

    return $rows;
}

function convertUvToUnits(int $uv, ?float $unitsPerCase, ?float $casesPerPallet): float
{
    if ($uv <= 0) {
        return 0.0;
    }
    if ($casesPerPallet !== null && $casesPerPallet > 0 && $unitsPerCase !== null && $unitsPerCase > 0) {
        return $uv * $casesPerPallet * $unitsPerCase;
    }
    if ($unitsPerCase !== null && $unitsPerCase > 0) {
        return $uv * $unitsPerCase;
    }
    if ($casesPerPallet !== null && $casesPerPallet > 0) {
        return $uv * $casesPerPallet;
    }
    return (float)$uv;
}

function allocateUnits(array $unitsPerItem, int $planUnits): array
{
    $totalUnits = 0.0;
    foreach ($unitsPerItem as $meta) {
        $totalUnits += $meta['units_total'];
    }
    if ($totalUnits <= 0) {
        return [];
    }

    $alloc = [];
    $fractions = [];
    $remaining = $planUnits;

    foreach ($unitsPerItem as $meta) {
        $ideal = ($meta['units_total'] / $totalUnits) * $planUnits;
        $base = (int)floor($ideal);
        $base = min($base, (int)round($meta['units_total']));
        $alloc[$meta['index']] = $base;
        $remaining -= $base;
        $fractions[] = [
            'index' => $meta['index'],
            'fraction' => $ideal - $base,
            'capacity' => (int)round($meta['units_total']) - $base,
        ];
    }

    if ($remaining > 0) {
        usort($fractions, static function (array $a, array $b): int {
            if ($a['fraction'] === $b['fraction']) {
                return 0;
            }
            return ($a['fraction'] < $b['fraction']) ? 1 : -1;
        });
        foreach ($fractions as $row) {
            if ($remaining <= 0) {
                break;
            }
            if ($row['capacity'] <= 0) {
                continue;
            }
            $alloc[$row['index']]++;
            $remaining--;
        }
    }

    if ($remaining > 0) {
        foreach ($unitsPerItem as $meta) {
            if ($remaining <= 0) {
                break;
            }
            $index = $meta['index'];
            $capacity = (int)round($meta['units_total']);
            $current = $alloc[$index] ?? 0;
            if ($current >= $capacity) {
                continue;
            }
            $alloc[$index] = $current + 1;
            $remaining--;
        }
    }

    return $alloc;
}

function buildLoteCode(string $pedidoCode): string
{
    $clean = preg_replace('/[^A-Z0-9]+/', '', strtoupper($pedidoCode));
    $suffix = substr($clean, -8);
    if ($suffix === '') {
        $suffix = 'GEN';
    }
    return 'DEV-' . $suffix;
}

function mergeDuplicateRows(array $rows): array
{
    if (!$rows) {
        return [];
    }

    $merged = [];
    foreach ($rows as $row) {
        $keyParts = [
            strtoupper((string)($row['sku'] ?? '')),
            strtoupper((string)($row['motivo'] ?? '')),
            strtoupper((string)($row['lote'] ?? '')),
        ];
        if (array_key_exists('factura', $row)) {
            $keyParts[] = strtoupper((string)($row['factura'] ?? ''));
        }
        $key = implode('|', $keyParts);

        if (!isset($merged[$key])) {
            $merged[$key] = $row;
            continue;
        }

        $merged[$key]['pallets'] = (int)($merged[$key]['pallets'] ?? 0) + (int)($row['pallets'] ?? 0);
        $merged[$key]['cajas_sueltas'] = (int)($merged[$key]['cajas_sueltas'] ?? 0) + (int)($row['cajas_sueltas'] ?? 0);
        $merged[$key]['unidades_sueltas'] = (int)($merged[$key]['unidades_sueltas'] ?? 0) + (int)($row['unidades_sueltas'] ?? 0);
        $merged[$key]['unidades_danadas'] = (int)($merged[$key]['unidades_danadas'] ?? 0) + (int)($row['unidades_danadas'] ?? 0);
        if (array_key_exists('total_unidades', $row)) {
            $merged[$key]['total_unidades'] = (int)($merged[$key]['total_unidades'] ?? 0) + (int)($row['total_unidades'] ?? 0);
        }
    }

    return array_values($merged);
}
