// replace_sass_color_functions.js
// Usage: node replace_sass_color_functions.js [--apply]
// Without --apply it does a dry run and prints files that would change.

const fs = require('fs');
const path = require('path');

const root = path.resolve(__dirname, '..', 'public', 'assets', 'scss');
const apply = process.argv.includes('--apply');

function walk(dir) {
  let results = [];
  const list = fs.readdirSync(dir);
  list.forEach(file => {
    const full = path.join(dir, file);
    const stat = fs.statSync(full);
    if (stat && stat.isDirectory()) {
      results = results.concat(walk(full));
    } else if (/\.scss$/i.test(file)) {
      results.push(full);
    }
  });
  return results;
}

function replaceContent(src) {
  let changed = false;
  let out = src;

  // Replace lighten($color, 20%) => color.adjust($color, $lightness: 20%)
  // Replace darken($color, 20%)  => color.adjust($color, $lightness: -20%)

  // Use a regex that handles nested parentheses roughly and spaces
  const lightenRe = /lighten\(\s*([^,\)]+?)\s*,\s*([0-9]+(?:\.[0-9]+)?%?)\s*\)/g;
  const darkenRe = /darken\(\s*([^,\)]+?)\s*,\s*([0-9]+(?:\.[0-9]+)?%?)\s*\)/g;

  if (lightenRe.test(out) || darkenRe.test(out)) {
    changed = true;
  }

  out = out.replace(lightenRe, function(_, colorExpr, amount) {
    // keep amount as-is
    return `color.adjust(${colorExpr.trim()}, $lightness: ${amount})`;
  });

  out = out.replace(darkenRe, function(_, colorExpr, amount) {
    // ensure negative sign
    const numeric = amount.trim();
    // If amount ends with %, keep it
    if (numeric.startsWith('-')) {
      return `color.adjust(${colorExpr.trim()}, $lightness: ${numeric})`;
    }
    return `color.adjust(${colorExpr.trim()}, $lightness: -${numeric})`;
  });

  // If changed, ensure @use "sass:color"; is present.
  if (changed) {
    const useColorRegex = /@use\s+["']sass:color["'];/;
    if (!useColorRegex.test(out)) {
      // Find position after initial @use/@forward/@import blocks or after initial comments
      const lines = out.split(/\r?\n/);
      let insertAt = 0;
      for (let i = 0; i < lines.length; i++) {
        const ln = lines[i].trim();
        if (ln === '' || ln.startsWith('/*') || ln.startsWith('//') || ln.startsWith('*')) {
          insertAt = i + 1;
          continue;
        }
        if (/^@use|^@forward|^@import/.test(ln)) {
          insertAt = i + 1;
          continue;
        }
        break;
      }
      lines.splice(insertAt, 0, "@use \"sass:color\";");
      out = lines.join('\n');
    }
  }

  return {changed, out};
}

const files = walk(root);
let modified = [];

files.forEach(file => {
  try {
    const src = fs.readFileSync(file, 'utf8');
    const {changed, out} = replaceContent(src);
    if (changed) {
      modified.push(file);
      if (apply) {
        // backup
        fs.writeFileSync(file + '.bak', src, 'utf8');
        fs.writeFileSync(file, out, 'utf8');
        console.log('[APPLIED] ', file);
      } else {
        console.log('[DRYRUN] ', file);
      }
    }
  } catch (err) {
    console.error('ERR reading', file, err.message);
  }
});

console.log('\nSummary: ' + modified.length + ' files would be modified.');
if (!apply) console.log('Run with --apply to modify files (backups saved with .bak)');

process.exit(0);
