<?php
// scripts/report_pallet_updates.php
// Usage:
//   php report_pallet_updates.php --pl=123
//   php report_pallet_updates.php --recent=10   (not implemented)

declare(strict_types=1);

require_once __DIR__ . '/../config/db.php';

$opts = getopt('', ['pl::', 'help']);
if (isset($opts['help']) || !isset($opts['pl'])) {
    echo "Usage: php scripts/report_pallet_updates.php --pl=PL_ID\n";
    echo "Reports how many pallets linked to the PL have cliente_id populated and lists mismatches.\n";
    exit(0);
}

$plId = (int) $opts['pl'];
if ($plId <= 0) {
    fwrite(STDERR, "Invalid pl id\n");
    exit(2);
}

try {
    $pdo = getPDO();
} catch (Throwable $e) {
    fwrite(STDERR, "DB connection error: " . $e->getMessage() . "\n");
    exit(3);
}

// Check tables/columns
function exists_table(PDO $pdo, string $t): bool {
    $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?");
    $st->execute([$t]);
    return (int)$st->fetchColumn() > 0;
}

if (!exists_table($pdo, 'pl_packinglist')) {
    fwrite(STDERR, "pl_packinglist table not found in DB.\n");
    exit(4);
}
if (!exists_table($pdo, 'pl_rcv_link')) {
    fwrite(STDERR, "pl_rcv_link table not found in DB.\n");
    exit(5);
}
if (!exists_table($pdo, 'wh_pallet')) {
    fwrite(STDERR, "wh_pallet table not found in DB.\n");
    exit(6);
}

$colCheck = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'wh_pallet' AND COLUMN_NAME = 'cliente_id'");
$colCheck->execute();
$hasClienteCol = (int)$colCheck->fetchColumn() > 0;

$st = $pdo->prepare("SELECT id, codigo, cliente_id FROM pl_packinglist WHERE id = ? LIMIT 1");
$st->execute([$plId]);
$pl = $st->fetch();
if (!$pl) {
    fwrite(STDERR, "PL id $plId not found.\n");
    exit(7);
}

$plCliente = $pl['cliente_id'] ?? null;

$sql = "SELECT p.id AS pallet_id, p.codigo AS codigo, p.cliente_id AS pallet_cliente
          FROM pl_rcv_link l
          JOIN wh_pallet p ON p.id = l.pallet_id
         WHERE l.packinglist_id = ?
      GROUP BY p.id, p.codigo, p.cliente_id
      ORDER BY p.codigo";

$st = $pdo->prepare($sql);
$st->execute([$plId]);
$rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

$total = count($rows);
$withCliente = 0;
$mismatch = [];

foreach ($rows as $r) {
    if ($r['pallet_cliente'] !== null) $withCliente++;
    // if pl has cliente and pallet differs, mark mismatch
    if ($plCliente !== null && $r['pallet_cliente'] !== null && (string)$r['pallet_cliente'] !== (string)$plCliente) {
        $mismatch[] = $r;
    }
    if ($plCliente !== null && $r['pallet_cliente'] === null) {
        $mismatch[] = $r; // missing cliente where PL has one
    }
}

echo "PL id: $plId\n";
echo "PL cliente_id: " . ($plCliente === null ? '<null>' : $plCliente) . "\n";
echo "Total pallets linked: $total\n";
if ($hasClienteCol) {
    echo "Pallets with cliente_id populated: $withCliente\n";
} else {
    echo "Note: wh_pallet.cliente_id column NOT present in DB.\n";
}

if ($mismatch) {
    echo "MISMATCHES / missing cliente entries (showing up to 200 chars):\n";
    foreach ($mismatch as $mm) {
        printf(" - pallet_id=%d code=%s pallet_cliente=%s\n", $mm['pallet_id'], $mm['codigo'], ($mm['pallet_cliente'] === null ? '<null>' : $mm['pallet_cliente']));
    }
} else {
    echo "No mismatches detected (every pallet has same cliente or PL cliente is null).\n";
}

exit(0);
