<?php
// scripts/run_migration.php
// Usage: php run_migration.php path/to/migration.sql

declare(strict_types=1);

if ($argc < 2) {
    fwrite(STDERR, "Usage: php run_migration.php path/to/migration.sql\n");
    exit(2);
}

$file = $argv[1];
if (!is_file($file) || !is_readable($file)) {
    fwrite(STDERR, "Migration file not found or not readable: $file\n");
    exit(3);
}

require_once __DIR__ . '/../config/db.php';

try {
    $pdo = getPDO();
} catch (Throwable $e) {
    fwrite(STDERR, "Could not connect to DB: " . $e->getMessage() . "\n");
    exit(4);
}

$sql = file_get_contents($file);
if ($sql === false) {
    fwrite(STDERR, "Could not read file: $file\n");
    exit(5);
}

// Split statements by semicolon. This is simplistic but OK for small migrations
$parts = preg_split('/;\s*\n/', $sql);

echo "Found " . count($parts) . " statements (approx).\n";

$applied = 0;
foreach ($parts as $p) {
    $stmt = trim($p);
    if ($stmt === '') continue;
    try {
        $pdo->exec($stmt);
        $applied++;
        echo "OK: statement executed.\n";
    } catch (PDOException $e) {
        fwrite(STDERR, "ERROR executing statement: " . $e->getMessage() . "\nStatement: " . substr($stmt,0,200) . "...\n");
        exit(6);
    }
}

echo "Applied $applied statements.\n";

// Verify column exists: wh_pallet.cliente_id
$dbName = getenv('DB_NAME') ?: 'sol';
$check = $pdo->prepare("SELECT COUNT(*) AS c FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = ? AND TABLE_NAME = 'wh_pallet' AND COLUMN_NAME = 'cliente_id'");
$check->execute([$dbName]);
$res = $check->fetch();
$exists = (int)($res['c'] ?? 0);
if ($exists > 0) {
    echo "Verification: column wh_pallet.cliente_id exists in database '$dbName'.\n";
    exit(0);
} else {
    fwrite(STDERR, "Verification failed: column wh_pallet.cliente_id not found in database '$dbName'.\n");
    exit(7);
}
