<?php
// Seed a demo inbound packing list with 10 products for testing.
declare(strict_types=1);

$ROOT = dirname(__DIR__);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

$pdo = get_pdo();
$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

// Fetch candidate products that have palletization data.
$sql = <<<SQL
SELECT p.id,
       p.sku,
       p.denominacion,
       pal.cajas_por_pallet,
       pal.capas_por_pallet
  FROM para_productos p
  JOIN para_producto_paletizado pal ON pal.producto_id = p.id
 WHERE pal.cajas_por_pallet IS NOT NULL
   AND pal.cajas_por_pallet > 0
   AND p.activo = 1
 ORDER BY p.id
 LIMIT 40
SQL;

$products = $pdo->query($sql)->fetchAll(PDO::FETCH_ASSOC);
if (!$products) {
    throw new RuntimeException('No products with palletization data available.');
}

$even = [];
$odd = [];
foreach ($products as $product) {
    if (((int)$product['id']) % 2 === 0) {
        $even[] = $product;
    } else {
        $odd[] = $product;
    }
}

if (count($even) < 5 || count($odd) < 5) {
    throw new RuntimeException('Not enough even/odd products with pallet data (need at least 5 of each).');
}

$selected = array_merge(array_slice($odd, 0, 5), array_slice($even, 0, 5));

$pdo->beginTransaction();
try {
    $today = new DateTimeImmutable('now');
    $plCode = 'PL-DEMO-' . $today->format('Ymd-His');

    $stmt = $pdo->prepare('INSERT INTO pl_packinglist (codigo, cliente_ref, fecha, estado, obs) VALUES (?, ?, ?, ?, ?)');
    $stmt->execute([
        $plCode,
        'CLIENTE_DEMO',
        $today->format('Y-m-d'),
        'CREADO',
        'Carga generada automáticamente para pruebas'
    ]);
    $packingListId = (int)$pdo->lastInsertId();

    $itemInsert = $pdo->prepare(
        'INSERT INTO pl_packinglist_item (
            packinglist_id,
            sku_cliente,
            producto_id,
            descripcion,
            expected_uv,
            expected_uc
        ) VALUES (?, ?, ?, ?, ?, ?)' 
    );

    foreach ($selected as $product) {
        $productId = (int)$product['id'];
        $isEven = ($productId % 2) === 0;

        if ($isEven) {
            $expectedUv = 10; // 10 cajas
            $expectedUc = 100; // 100 unidades sueltas
        } else {
            $expectedUv = 2 * (int)$product['cajas_por_pallet']; // dos pallets completos
            $expectedUc = 0;
        }

        $itemInsert->execute([
            $packingListId,
            $product['sku'],
            $productId,
            $product['denominacion'],
            $expectedUv,
            $expectedUc
        ]);
    }

    // Create a basic ingreso record linked to the packing list.
    $ingStmt = $pdo->prepare(
        'INSERT INTO pl_ingreso (
            packinglist_id,
            deposito_id,
            fecha_ingreso,
            doc_tipo,
            doc_numero,
            observacion
        ) VALUES (?, ?, ?, ?, ?, ?)'
    );
    $ingStmt->execute([
        $packingListId,
        1, // deposito DEP1
        $today->format('Y-m-d'),
        'OTRO',
        'DOC-' . $today->format('His'),
        'Ingreso simulado para pruebas'
    ]);

    $pdo->commit();

    echo json_encode([
        'packinglist_id' => $packingListId,
        'packinglist_codigo' => $plCode,
        'items_inserted' => count($selected)
    ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), PHP_EOL;
} catch (Throwable $e) {
    $pdo->rollBack();
    throw $e;
}
